<?php
// ======================================================================
//  0. START SESSION & DB CONNECTION
// ======================================================================
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'db-config.php';

// ======================================================================
//  1. CONFIGURATION & HELPER FUNCTIONS
// ======================================================================
function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }
function echo_list_items($text) {
    if (empty(trim($text ?? ''))) return;
    $items = preg_split('/[\r\n,]+/', trim($text));
    foreach ($items as $item) {
        if (!empty(trim($item))) echo "<li>" . e($item) . "</li>\n";
    }
}

// ======================================================================
//  2. HANDLE AGENT REFERRAL LINK
// ======================================================================
if (isset($_GET['ref']) && !empty($_GET['ref'])) {
    // Store the agent's ID from the URL into the session
    $_SESSION['referring_agent_id'] = (int)$_GET['ref'];
}

// ======================================================================
//  3. HANDLE BOOKING FORM SUBMISSION (DATABASE INSERT)
// ======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['request_booking'])) {
    $package_id = $_POST['package_id'] ?? 'Unknown';
    $package_name = $_POST['package_name'] ?? 'Unknown';
    $customer_name = trim($_POST['customer_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone_number = trim($_POST['phone'] ?? '');
    $room_type = $_POST['room_type'] ?? 'Sharing';
    $pax = (int)($_POST['party_size'] ?? 1);

    // Basic Validation
    if (empty($customer_name) || empty($email) || empty($phone_number) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $_SESSION['form_message'] = "Please fill in all required fields with valid information.";
        $_SESSION['form_msg_type'] = "error";
    } else {
        $conn->begin_transaction();
        try {
            // Determine the user ID of the person making the booking (can be null)
            $user_id_to_save = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;
            // Get the agent ID if it was stored from a referral link (can be null)
            $referred_by_agent_id = isset($_SESSION['referring_agent_id']) ? (int)$_SESSION['referring_agent_id'] : null;

            // Insert the inquiry into the database
            $sql1 = "INSERT INTO umrah_inquiries (package_id, package_name, customer_name, customer_phone, customer_email, room_type, pax, user_id, referred_by_agent_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt1 = $conn->prepare($sql1);
            // The types must match the columns: s s s s s s i i i
            $stmt1->bind_param("ssssssiii", $package_id, $package_name, $customer_name, $phone_number, $email, $room_type, $pax, $user_id_to_save, $referred_by_agent_id);
            if (!$stmt1->execute()) {
                throw new Exception($stmt1->error);
            }
            $stmt1->close();

            // Create a notification for the admin panel
            $notification_msg = "New inquiry for " . e($package_name) . " from " . e($customer_name) . ".";
            if ($referred_by_agent_id) {
                 $notification_msg .= " (Referred by Agent ID: " . $referred_by_agent_id . ")";
            }
            $sql2 = "INSERT INTO notifications (type, message, link) VALUES ('New Umrah Inquiry', ?, 'admin/view-inquiries.php')";
            $stmt2 = $conn->prepare($sql2);
            $stmt2->bind_param("s", $notification_msg);
            $stmt2->execute();
            $stmt2->close();
            
            $conn->commit();
            
            // Unset referral ID after use
            if (isset($_SESSION['referring_agent_id'])) {
                unset($_SESSION['referring_agent_id']);
            }

            // Redirect to a thank you page to prevent form resubmission
            header("Location: thank-you.php");
            exit();

        } catch (Exception $e) {
            $conn->rollback();
            // Log the detailed error for the developer
            error_log("Booking Form Database Error: " . $e->getMessage());
            // Show a generic error to the user
            $_SESSION['form_message'] = "A server error occurred. Please try again later.";
            $_SESSION['form_msg_type'] = "error";
        }
    }
    // Redirect back to the same page to show the error message
    header("Location: " . basename(__FILE__) . "?id=" . urlencode($package_id) . "#booking-form-anchor");
    exit();
}


// ======================================================================
//  4. FETCH PACKAGE DATA FOR DISPLAY
// ======================================================================
if (!isset($_GET['id']) || empty(trim($_GET['id']))) {
    http_response_code(400); die("Error: No package ID specified.");
}
$package_code = trim($_GET['id']);
$stmt = $conn->prepare("SELECT up.*, a.airline_name, a.logo_url FROM umrah_packages up LEFT JOIN airlines a ON up.airline_id = a.id WHERE up.package_id = ? AND up.is_active = 1");
$stmt->bind_param("s", $package_code);
$stmt->execute();
$result = $stmt->get_result();
if ($result->num_rows === 0) {
    http_response_code(404); die("Error: This package is either not found or is no longer available.");
}
$package = $result->fetch_assoc();
$stmt->close();
$conn->close();

// Prepare form message if it exists in session
$booking_message = '';
if (isset($_SESSION['form_message'])) {
    $msg_type = $_SESSION['form_msg_type'] === 'error' ? 'error' : 'success';
    $booking_message = '<div class="notice ' . $msg_type . '">' . e($_SESSION['form_message']) . '</div>';
    unset($_SESSION['form_message'], $_SESSION['form_msg_type']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($package['package_name']) ?> - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <style>
        .tour-detail-page { padding: 2rem 0; }
        .tour-layout-grid { display: grid; grid-template-columns: 1fr; gap: 2rem; }
        @media (min-width: 992px) { .tour-layout-grid { grid-template-columns: 2fr 1fr; } }
        .booking-card { position: sticky; top: 20px; }
        .notice { padding: 15px 20px; margin-bottom: 20px; border-radius: 8px; border: 1px solid transparent; }
        .notice.success { color: #155724; background-color: #d4edda; border-color: #c3e6cb; }
        .notice.error { color: #721c24; background-color: #f8d7da; border-color: #f5c6cb; }

        /* --- AGENT SHARING PANEL STYLES --- */
        .agent-share-panel { background-color: #fff8e1; border: 1px solid #ffecb3; border-left: 5px solid #ffc107; padding: 20px; margin-bottom: 25px; border-radius: 8px; }
        .agent-share-panel h3 { margin-top: 0; font-size: 1.2rem; color: #333; }
        .agent-share-panel p { margin: 5px 0 15px 0; font-size: 0.9rem; color: #555; }
        .share-link-wrapper { display: flex; gap: 10px; }
        .share-link-input { flex-grow: 1; padding: 10px; font-size: 0.9rem; border: 1px solid #ccc; border-radius: 5px; background-color: #fff; color: #333; }
        .copy-link-btn { padding: 10px 20px; font-weight: 600; background-color: var(--primary-gold, #cfa968); color: white; border: none; border-radius: 5px; cursor: pointer; transition: background-color 0.2s; }
        .copy-link-btn:hover { background-color: #b89354; }
        .copy-link-btn i { margin-right: 8px; }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    <main class="tour-detail-page">
        <div class="container">
            <div class="tour-layout-grid">
                <div class="tour-main-content">
                    <div class="tour-header">
                        <h1><?= e($package['package_name']) ?></h1>
                        <div class="tour-meta-info"><span>Package Code: <?= e($package['package_id']) ?></span></div>
                    </div>
                    
                    <!-- AGENT SHARING PANEL -->
                    <?php if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent'): ?>
                        <section class="agent-share-panel">
                            <h3><i class="fa-solid fa-share-nodes"></i> Your Agent Sharing Link</h3>
                            <p>Share this unique link with your customers. Any bookings made through it will be credited to you.</p>
                            <div class="share-link-wrapper">
                                <?php
                                $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
                                $page_name = basename(__FILE__);
                                $base_url = "$protocol://$_SERVER[HTTP_HOST]/" . $page_name . '?id=' . urlencode($package['package_id']);
                                $referral_link = $base_url . '&ref=' . urlencode($_SESSION['user_id']);
                                ?>
                                <input type="text" value="<?= e($referral_link) ?>" id="referralLinkInput" class="share-link-input" readonly>
                                <button class="copy-link-btn" id="copyLinkBtn"><i class="fa-solid fa-copy"></i> Copy</button>
                            </div>
                        </section>
                    <?php endif; ?>

                    <div class="image-gallery">
                        <div class="gallery-main-image"><img src="<?= e($package['main_image_link'] ?? 'images/default-package.jpg') ?>" alt="Main image for <?= e($package['package_name']) ?>"></div>
                    </div>

                    <section class="tour-section"><h2>Overview</h2><p><?= nl2br(e($package['overview'])) ?></p></section>
                    <section class="tour-section"><h2>Itinerary</h2><ul><?php echo_list_items($package['itinerary']); ?></ul></section>
                    <section class="tour-section"><h2>Package Includes</h2><ul><?php echo_list_items($package['whats_included']); ?></ul></section>
                    <?php if(!empty(trim($package['whats_extra'] ?? ''))): ?><section class="tour-section"><h2>What's Extra</h2><ul><?php echo_list_items($package['whats_extra']); ?></ul></section><?php endif; ?>
                </div>

                <aside class="tour-booking-sidebar">
                    <div class="booking-card" id="booking-form-anchor">
                        <div class="booking-price">Request Booking</div>
                        <?= $booking_message ?>
                        <form action="<?= e(basename(__FILE__)) ?>?id=<?= urlencode($package_code) ?>" method="POST" class="booking-form">
                            <input type="hidden" name="request_booking" value="1">
                            <input type="hidden" name="package_id" value="<?= e($package['package_id']) ?>">
                            <input type="hidden" name="package_name" value="<?= e($package['package_name']) ?>">

                            <div class="form-group"><label for="customer_name">Full Name*:</label><input type="text" id="customer_name" name="customer_name" class="booking-input" required></div>
                            <div class="form-group"><label for="phone">Phone Number*:</label><input type="tel" id="phone" name="phone" class="booking-input" required></div>
                            <div class="form-group"><label for="email">Email Address*:</label><input type="email" id="email" name="email" class="booking-input" required></div>
                            <div class="form-group"><label for="room-type">Room Type:</label>
                                <select id="room-type" name="room_type" class="booking-input">
                                    <?php if (!empty($package['price_quint'])) : ?><option value="Sharing">Sharing</option><?php endif; ?>
                                    <?php if (!empty($package['price_quad'])) : ?><option value="Quad">Quad</option><?php endif; ?>
                                    <?php if (!empty($package['price_triple'])) : ?><option value="Triple">Triple</option><?php endif; ?>
                                    <?php if (!empty($package['price_double'])) : ?><option value="Double">Double</option><?php endif; ?>
                                </select>
                            </div>
                             <div class="form-group"><label>Total Travelers:</label>
                                <div class="party-size-selector"><button type="button" class="party-btn minus">-</button><input type="text" id="party-size" name="party_size" value="1" readonly><button type="button" class="party-btn plus">+</button></div>
                            </div>
                            <button type="submit" class="btn-booking">Send Inquiry</button>
                        </form>
                    </div>
                </aside>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const minusBtn = document.querySelector('.party-btn.minus');
        const plusBtn = document.querySelector('.party-btn.plus');
        const partyInput = document.getElementById('party-size');
        if(minusBtn && plusBtn && partyInput) {
            minusBtn.addEventListener('click', () => { if (parseInt(partyInput.value) > 1) partyInput.value--; });
            plusBtn.addEventListener('click', () => { partyInput.value = parseInt(partyInput.value) + 1; });
        }
        
        const copyBtn = document.getElementById('copyLinkBtn');
        const linkInput = document.getElementById('referralLinkInput');
        if (copyBtn && linkInput) {
            copyBtn.addEventListener('click', function() {
                linkInput.select();
                document.execCommand('copy');
                copyBtn.innerHTML = '<i class="fa-solid fa-check"></i> Copied!';
                setTimeout(() => { copyBtn.innerHTML = '<i class="fa-solid fa-copy"></i> Copy'; }, 2000);
            });
        }
    });
    </script>
</body>
</html>