<?php

/**
 * manage-agent-requests.php (v2.1 - Advanced Filtering)
 * - Admin dashboard to view and manage all incoming voucher requests from agents.
 * - UPGRADED: Implemented a full filter bar for searching by ID, Family Name, Agent, and Status.
 * - UPGRADED: All filters work together and are maintained during pagination.
 * - UPGRADED: SQL queries are now fully dynamic and use prepared statements for security.
 */
session_start();
include_once '../db-config.php'; // Handles security, sessions, and DB connection

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- 1. SETUP FILTERS AND PAGINATION ---
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

// Get filter values from GET request
$filter_request_id = trim($_GET['request_id'] ?? '');
$filter_family_name = trim($_GET['family_name'] ?? '');
$filter_agent_id = (int)($_GET['agent_id'] ?? 0);
$filter_status = $_GET['status'] ?? 'pending'; // Default to 'pending'
if (!in_array($filter_status, ['all', 'pending', 'in-progress', 'completed', 'rejected'])) {
    $filter_status = 'pending';
}

// --- 2. FETCH DATA for filter dropdowns ---
$agents_list = $conn->query("SELECT id, name FROM users WHERE user_type = 'agent' ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- 3. DYNAMICALLY BUILD SQL BASED ON FILTERS ---
$base_sql_from = "FROM voucher_requests vr
                  LEFT JOIN users u ON vr.agent_id = u.id";
$where_clauses = [];
$params = [];
$types = "";

if (!empty($filter_request_id)) {
    $where_clauses[] = "vr.id = ?";
    $params[] = $filter_request_id;
    $types .= "i";
}
if (!empty($filter_family_name)) {
    $where_clauses[] = "vr.family_head_name LIKE ?";
    $params[] = "%" . $filter_family_name . "%";
    $types .= "s";
}
if ($filter_agent_id > 0) {
    $where_clauses[] = "vr.agent_id = ?";
    $params[] = $filter_agent_id;
    $types .= "i";
}
if ($filter_status !== 'all') {
    $status_filter_db = ucwords(str_replace('-', ' ', $filter_status));
    $where_clauses[] = "vr.status = ?";
    $params[] = $status_filter_db;
    $types .= "s";
}

$where_sql = count($where_clauses) > 0 ? "WHERE " . implode(" AND ", $where_clauses) : "";

// --- 4. GET TOTAL COUNT FOR THE CURRENT FILTERS ---
$count_sql = "SELECT COUNT(vr.id) as total $base_sql_from $where_sql";
$stmt_count = $conn->prepare($count_sql);
if (!empty($params)) {
    $stmt_count->bind_param($types, ...$params);
}
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_items / $items_per_page);
$stmt_count->close();

// --- 5. FETCH THE PAGINATED DATA for display ---
$data_sql = "SELECT vr.*, u.name as agent_name
            $base_sql_from $where_sql
            ORDER BY vr.request_date DESC 
            LIMIT ?, ?";

$stmt_data = $conn->prepare($data_sql);
$final_params = $params;
$final_params[] = $offset;
$final_params[] = $items_per_page;
$final_types = $types . "ii";

$stmt_data->bind_param($final_types, ...$final_params);
$stmt_data->execute();
$requests_result = $stmt_data->get_result();

$success_message = $_SESSION['success_message'] ?? null;
unset($_SESSION['success_message']);

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Manage Voucher Requests</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-container {
            display: flex;
            flex-wrap: wrap;
            gap: 1rem;
            align-items: flex-end;
            padding: 1.5rem;
            background-color: #f8f9fa;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .filter-container .form-group {
            margin-bottom: 0;
        }

        .filter-actions {
            display: flex;
            gap: 0.5rem;
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="overlay" id="overlay"></div>

        <div class="main-content">
            <header class="main-header">
                <div class="user-info"><span>Manage Voucher Requests</span></div>
            </header>
            <main class="content-body">
                <div class="page-header">
                    <h1 class="page-title">Manage Voucher Requests</h1>
                </div>

                <div class="filter-container">
                    <form action="manage-agent-requests.php" method="GET">
                        <div style="display: flex; flex-wrap: wrap; gap: 1rem; align-items: flex-end;">
                            <div class="form-group"><label>Request ID</label><input type="number" name="request_id" class="form-control" value="<?= e($filter_request_id) ?>" placeholder="e.g. 42"></div>
                            <div class="form-group"><label>Family Head Name</label><input type="text" name="family_name" class="form-control" value="<?= e($filter_family_name) ?>" placeholder="Search name..."></div>
                            <div class="form-group"><label>Agent</label>
                                <select name="agent_id" class="form-control">
                                    <option value="0">All Agents</option>
                                    <?php foreach ($agents_list as $agent): ?>
                                        <option value="<?= e($agent['id']) ?>" <?= ($filter_agent_id == $agent['id']) ? 'selected' : '' ?>><?= e($agent['name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-group"><label>Status</label>
                                <select name="status" class="form-control">
                                    <option value="all" <?= ($filter_status == 'all') ? 'selected' : '' ?>>All Statuses</option>
                                    <option value="pending" <?= ($filter_status == 'pending') ? 'selected' : '' ?>>Pending</option>
                                    <option value="in-progress" <?= ($filter_status == 'in-progress') ? 'selected' : '' ?>>In Progress</option>
                                    <option value="completed" <?= ($filter_status == 'completed') ? 'selected' : '' ?>>Completed</option>
                                    <option value="rejected" <?= ($filter_status == 'rejected') ? 'selected' : '' ?>>Rejected</option>
                                </select>
                            </div>
                            <div class="filter-actions">
                                <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filter</button>
                                <a href="manage-agent-requests.php" class="btn btn-secondary"><i class="fas fa-times"></i> Clear</a>
                            </div>
                        </div>
                    </form>
                </div>

                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>

                <div class="content-card">
                    <div class="table-responsive">
                        <table>
                            <thead>
                                <tr>
                                    <th>Request #</th>
                                    <th>Agent</th>
                                    <th>Family Head</th>
                                    <th>Request Date</th>
                                    <th>Status</th>
                                    <th style="width: 150px;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($requests_result->num_rows > 0): ?>
                                    <?php while ($request = $requests_result->fetch_assoc()):
                                        $status_class = 'status-' . strtolower(str_replace(' ', '-', $request['status']));
                                    ?>
                                        <tr>
                                            <td><strong>#<?= e($request['id']) ?></strong></td>
                                            <td><?= e($request['agent_name']) ?></td>
                                            <td><?= e($request['family_head_name']) ?></td>
                                            <td><?= date('d M Y, h:i A', strtotime($request['request_date'])) ?></td>
                                            <td><span class="status-badge <?= $status_class ?>"><?= e($request['status']) ?></span></td>
                                            <td class="actions-cell">
                                                <a href="process-request.php?id=<?= $request['id'] ?>" class="btn btn-sm btn-primary" title="View & Process Request">
                                                    <i class="fas fa-cogs"></i> Process
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="text-center">No requests found for the selected criteria.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <!-- PAGINATION CONTROLS -->
                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls">
                            <?php
                            $query_params = $_GET;
                            if ($current_page > 1) {
                                $query_params['page'] = $current_page - 1;
                                echo '<a href="manage-agent-requests.php?' . http_build_query($query_params) . '" class="btn btn-secondary btn-sm">« Previous</a>';
                            }
                            echo '<span class="page-info">Page ' . $current_page . ' of ' . $total_pages . '</span>';
                            if ($current_page < $total_pages) {
                                $query_params['page'] = $current_page + 1;
                                echo '<a href="manage-agent-requests.php?' . http_build_query($query_params) . '" class="btn btn-secondary btn-sm">Next »</a>';
                            }
                            ?>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>