<?php
session_start();
include_once '../db-config.php';

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- HANDLE DELETE REQUEST ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- Single Delete ---
    if (isset($_POST['delete_voucher'])) {
        $voucher_id_to_delete = (int)($_POST['voucher_id'] ?? 0);
        if ($voucher_id_to_delete > 0) {
            $conn->begin_transaction();
            try {
                $conn->execute_query("DELETE FROM voucher_mutamers WHERE voucher_id = ?", [$voucher_id_to_delete]);
                $conn->execute_query("DELETE FROM voucher_accommodations WHERE voucher_id = ?", [$voucher_id_to_delete]);
                $conn->execute_query("DELETE FROM voucher_flights WHERE voucher_id = ?", [$voucher_id_to_delete]);
                $conn->execute_query("DELETE FROM vouchers WHERE id = ?", [$voucher_id_to_delete]);
                $conn->commit();
                $_SESSION['success_message'] = "Voucher #" . $voucher_id_to_delete . " deleted successfully.";
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error_message'] = "Error deleting voucher: " . $e->getMessage();
            }
        }
    }
    // --- Multi-Select Delete ---
    elseif (isset($_POST['delete_selected'])) {
        if (!empty($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
            $ids_to_delete = array_map('intval', $_POST['selected_ids']);
            if (!empty($ids_to_delete)) {
                $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
                $types = str_repeat('i', count($ids_to_delete));
                
                $conn->begin_transaction();
                try {
                    $conn->execute_query("DELETE FROM voucher_mutamers WHERE voucher_id IN ($placeholders)", $ids_to_delete);
                    $conn->execute_query("DELETE FROM voucher_accommodations WHERE voucher_id IN ($placeholders)", $ids_to_delete);
                    $conn->execute_query("DELETE FROM voucher_flights WHERE voucher_id IN ($placeholders)", $ids_to_delete);
                    $conn->execute_query("DELETE FROM vouchers WHERE id IN ($placeholders)", $ids_to_delete);
                    $conn->commit();
                    $_SESSION['success_message'] = count($ids_to_delete) . " voucher(s) deleted successfully.";
                } catch (Exception $e) {
                    $conn->rollback();
                    $_SESSION['error_message'] = "Error during bulk deletion: " . $e->getMessage();
                }
            }
        } else {
            $_SESSION['error_message'] = "No vouchers were selected for deletion.";
        }
    }
    header("Location: manage-vouchers.php");
    exit;
}

// --- FILTERS AND PAGINATION ---
$items_per_page = 15;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;
$filter_voucher_id = trim($_GET['voucher_id'] ?? '');
$filter_family_name = trim($_GET['family_name'] ?? '');
$filter_user_id = (int)($_GET['user_id'] ?? 0);
$filter_vendor_id = (int)($_GET['vendor_id'] ?? 0);
$filter_status = $_GET['status'] ?? 'all';
if (!in_array($filter_status, ['all', 'tentative', 'confirmed'])) { $filter_status = 'all'; }

// --- DATA FOR DROPDOWNS ---
$users_list = $conn->query("SELECT id, name FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- BUILD SQL QUERY ---
$base_sql = "FROM vouchers v LEFT JOIN users u ON v.user_id = u.id LEFT JOIN vendors vn ON v.vendor_id = vn.id";
$where_clauses = []; $params = []; $types = "";
if (!empty($filter_voucher_id)) { $where_clauses[] = "v.id = ?"; $params[] = $filter_voucher_id; $types .= "i"; }
if (!empty($filter_family_name)) { $where_clauses[] = "v.family_head_name LIKE ?"; $params[] = "%" . $filter_family_name . "%"; $types .= "s"; }
if ($filter_user_id > 0) { $where_clauses[] = "v.user_id = ?"; $params[] = $filter_user_id; $types .= "i"; }
if ($filter_vendor_id > 0) { $where_clauses[] = "v.vendor_id = ?"; $params[] = $filter_vendor_id; $types .= "i"; }
if ($filter_status !== 'all') { $where_clauses[] = "v.status = ?"; $params[] = ucfirst($filter_status); $types .= "s"; }
$where_sql = count($where_clauses) > 0 ? "WHERE " . implode(" AND ", $where_clauses) : "";

// --- COUNT & FETCH DATA ---
$count_sql = "SELECT COUNT(v.id) as total $base_sql $where_sql";
$stmt_count = $conn->prepare($count_sql);
if (!empty($params)) { $stmt_count->bind_param($types, ...$params); }
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_items / $items_per_page);

$data_sql = "SELECT v.*, u.name as user_name, u.company_name, u.user_type, vn.name as vendor_name $base_sql $where_sql ORDER BY v.voucher_date DESC, v.id DESC LIMIT ?, ?";
$final_params = $params; $final_params[] = $offset; $final_params[] = $items_per_page;
$final_types = $types . "ii";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($final_types, ...$final_params);
$stmt_data->execute();
$vouchers_result = $stmt_data->get_result();

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Vouchers</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Manage Vouchers</span></div>
            </header>
            <main class="content-body">
                <div class="page-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 20px;">
                    <h1 class="page-title">Vouchers</h1>
                    <div class="header-actions" style="display: flex; gap: 1rem;">
                         <button type="submit" form="bulk-delete-form" name="delete_selected" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete all selected vouchers?');"><i class="fas fa-trash"></i> Delete Selected</button>
                        <a href="create-voucher.php" class="btn btn-primary"><i class="fas fa-plus"></i> Create New Voucher</a>
                    </div>
                </div>

                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>

                <div class="content-card">
                    <div class="card-body">
                        <form action="manage-vouchers.php" method="GET" class="styled-form">
                            <div class="form-grid">
                                <div class="form-group"><label>Voucher ID</label><input type="number" name="voucher_id" class="form-control" value="<?= e($filter_voucher_id) ?>" placeholder="e.g. 123"></div>
                                <div class="form-group"><label>Family Head Name</label><input type="text" name="family_name" class="form-control" value="<?= e($filter_family_name) ?>" placeholder="Search name..."></div>
                                <div class="form-group"><label>User/Agent</label>
                                    <select name="user_id" class="form-control">
                                        <option value="0">All Users</option>
                                        <?php foreach ($users_list as $user): ?><option value="<?= e($user['id']) ?>" <?= ($filter_user_id == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?></option><?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group"><label>Vendor</label>
                                    <select name="vendor_id" class="form-control">
                                        <option value="0">All Vendors</option>
                                        <?php foreach ($vendors_list as $vendor): ?><option value="<?= e($vendor['id']) ?>" <?= ($filter_vendor_id == $vendor['id']) ? 'selected' : '' ?>><?= e($vendor['name']) ?></option><?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group"><label>Status</label>
                                    <select name="status" class="form-control">
                                        <option value="all" <?= ($filter_status == 'all') ? 'selected' : '' ?>>All Statuses</option>
                                        <option value="tentative" <?= ($filter_status == 'tentative') ? 'selected' : '' ?>>Tentative</option>
                                        <option value="confirmed" <?= ($filter_status == 'confirmed') ? 'selected' : '' ?>>Confirmed</option>
                                    </select>
                                </div>
                            </div>
                            <div class="form-actions">
                                <a href="manage-vouchers.php" class="btn btn-secondary"><i class="fas fa-times"></i> Clear</a>
                                <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filter</button>
                            </div>
                        </form>
                    </div>
                </div>

                <div class="content-card">
                    <div class="card-body">
                        <form action="manage-vouchers.php" method="POST" id="bulk-delete-form">
                            <div class="table-responsive">
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th style="width: 40px;"><input type="checkbox" id="select-all-checkbox"></th>
                                            <th>Voucher #</th><th>Manual #</th><th>Issue Date</th><th>Family Head</th><th>Pax</th><th>Status</th><th>Vendor</th><th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($vouchers_result->num_rows > 0): while ($v = $vouchers_result->fetch_assoc()): ?>
                                            <tr>
                                                <td><input type="checkbox" name="selected_ids[]" class="row-checkbox" value="<?= $v['id'] ?>"></td>
                                                <td><strong>#<?= e($v['id']) ?></strong></td>
                                                <td><?= e($v['manual_no']) ?></td>
                                                <td><?= date('d M Y', strtotime($v['voucher_date'])) ?></td>
                                                <td><?= e($v['family_head_name']) ?><br><small style="color:var(--text-muted)"><?= e($v['user_name'] ?? 'Direct') ?></small></td>
                                                <td><?= e($v['pax_summary']) ?></td>
                                                <td><span class="status-badge status-<?= strtolower(e($v['status'])) ?>"><?= e(ucfirst($v['status'])) ?></span></td>
                                                <td><?= e($v['vendor_name'] ?? 'N/A') ?></td>
                                                <td class="actions-cell">
                                                    <div style="display: flex; gap: 0.5rem;">
                                                        <a href="view-voucher.php?id=<?= $v['id'] ?>" class="btn btn-sm btn-info" title="View/Print" target="_blank"><i class="fas fa-eye"></i></a>
                                                        <a href="edit-voucher.php?id=<?= $v['id'] ?>" class="btn btn-sm btn-primary" title="Edit"><i class="fas fa-edit"></i></a>
                                                        <form action="manage-vouchers.php" method="POST" onsubmit="return confirm('Delete this voucher and all its details?');" style="display:inline;">
                                                            <input type="hidden" name="voucher_id" value="<?= $v['id'] ?>">
                                                            <button type="submit" name="delete_voucher" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; else: ?>
                                            <tr><td colspan="9" class="text-center p-4">No vouchers found.</td></tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </form>
                        <?php if ($total_pages > 1): ?>
                            <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                                <?php $query_params = $_GET; if ($current_page > 1): $query_params['page'] = $current_page - 1; ?>
                                    <a href="?<?= http_build_query($query_params) ?>" class="btn btn-sm btn-secondary">« Previous</a>
                                <?php endif; ?>
                                <span style="padding: 0 10px;">Page <?= $current_page ?> of <?= $total_pages ?></span>
                                <?php if ($current_page < $total_pages): $query_params['page'] = $current_page + 1; ?>
                                    <a href="?<?= http_build_query($query_params) ?>" class="btn btn-sm btn-secondary">Next »</a>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const menuToggle = document.getElementById('menu-toggle');
        if (menuToggle) {
            menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
        }

        const selectAllCheckbox = document.getElementById('select-all-checkbox');
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                document.querySelectorAll('.row-checkbox').forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
            });
        }
    });
</script>
</body>
</html>