<?php
session_start();
include_once '../db-config.php';

//======================================================================
//  SECURITY GATEWAY
//======================================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

//======================================================================
//  BACKEND LOGIC (API & ACTIONS)
//======================================================================

// --- API LOGIC: Handle "mark as read" requests from JavaScript ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
    $data = json_decode(file_get_contents('php://input'), true);
    if (isset($data['action']) && $data['action'] === 'mark_read' && isset($data['id'])) {
        $notification_id = (int)$data['id'];
        $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ?");
        $stmt->bind_param("i", $notification_id);
        header('Content-Type: application/json');
        if ($stmt->execute()) {
            echo json_encode(['success' => true]);
        } else {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Database update failed.']);
        }
        $stmt->close();
        exit();
    }
}

// --- PAGE ACTION: Handle the "Mark All as Read" form submission ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['mark_all_read'])) {
    $conn->query("UPDATE notifications SET is_read = 1 WHERE is_read = 0");
    $_SESSION['success_message'] = "All unread notifications have been marked as read.";
    header("Location: notifications.php");
    exit();
}

//======================================================================
//  DATA FETCHING & PAGINATION
//======================================================================
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

$total_items_query = $conn->query("SELECT COUNT(*) as total FROM notifications");
$total_items = $total_items_query->fetch_assoc()['total'];
$total_pages = ($total_items > 0) ? ceil($total_items / $items_per_page) : 1;

$stmt = $conn->prepare("SELECT * FROM notifications ORDER BY created_at DESC LIMIT ?, ?");
$stmt->bind_param("ii", $offset, $items_per_page);
$stmt->execute();
$notifications_result = $stmt->get_result();
$stmt->close();

$success_message = $_SESSION['success_message'] ?? null;
unset($_SESSION['success_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Notifications</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .page-header { display: flex; justify-content: space-between; align-items: center; }
        .notification-list { border: 1px solid var(--border-color); border-radius: 8px; overflow: hidden; background-color: #fff; }
        .notification-item { padding: 1rem; border-bottom: 1px solid var(--border-color); transition: background-color 0.3s, opacity 0.3s; cursor: pointer; text-decoration: none; color: inherit; display: block; }
        .notification-item:last-child { border-bottom: none; }
        .notification-item.unread { background-color: #fcfdfe; font-weight: bold; border-left: 4px solid var(--primary-color); }
        .notification-item.read { background-color: #fff; color: var(--text-muted); opacity: 0.85; border-left: 4px solid transparent; }
        .notification-item:hover { background-color: #f8f9fa; }
        .notification-summary { display: flex; align-items: center; gap: 1rem; }
        .notification-icon { font-size: 1.5rem; color: var(--primary-color); }
        .notification-item.read .notification-icon { color: #adb5bd; }
        .notification-content { flex-grow: 1; }
        .notification-header { display: flex; justify-content: space-between; align-items: baseline; }
        .notification-time { font-size: 0.8em; color: var(--text-muted); font-weight: normal; }
        .notification-content p { margin: 5px 0 0; font-weight: normal; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Notifications</span></div>
            </header>

            <main class="content-body">
                <div class="page-header">
                    <h1 class="page-title">Notifications</h1>
                    <form action="notifications.php" method="POST">
                        <button type="submit" name="mark_all_read" class="btn btn-secondary btn-sm">Mark All as Read</button>
                    </form>
                </div>

                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>

                <div class="content-card">
                    <div class="notification-list" id="notification-list">
                        <?php if ($notifications_result->num_rows > 0): ?>
                            <?php while ($notification = $notifications_result->fetch_assoc()):
                                $is_read_class = $notification['is_read'] ? 'read' : 'unread';
                                $has_link = !empty($notification['link']);
                                $tag = $has_link ? 'a' : 'div';
                                $href = $has_link ? e($notification['link']) : '#';
                            ?>
                                <<?= $tag ?> href="<?= $href ?>" class="notification-item <?= $is_read_class ?>" data-id="<?= $notification['id'] ?>">
                                    <div class="notification-summary">
                                        <div class="notification-icon"><i class="fas fa-bell"></i></div>
                                        <div class="notification-content">
                                            <div class="notification-header">
                                                <strong><?= e(ucwords(str_replace('_', ' ', $notification['type']))) ?></strong>
                                                <span class="notification-time"><?= date('d M, Y H:i', strtotime($notification['created_at'])) ?></span>
                                            </div>
                                            <p><?= e($notification['message']) ?></p>
                                        </div>
                                    </div>
                                </<?= $tag ?>>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <div class="empty-state">You have no notifications.</div>
                        <?php endif; ?>
                    </div>

                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                            <?php if ($current_page > 1): ?><a href="?page=<?= $current_page - 1 ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                            <span style="padding: 0 10px;">Page <?= $current_page ?> of <?= $total_pages ?></span>
                            <?php if ($current_page < $total_pages): ?><a href="?page=<?= $current_page + 1 ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const notificationList = document.getElementById('notification-list');
            if (notificationList) {
                notificationList.addEventListener('click', function(event) {
                    const item = event.target.closest('.notification-item');
                    if (!item) return;

                    // Only handle with JS if it's a DIV (no link) and is unread
                    if (item.tagName === 'DIV' && item.classList.contains('unread')) {
                        event.preventDefault();
                        const notificationId = item.dataset.id;
                        
                        // Optimistically update UI
                        item.classList.remove('unread');
                        item.classList.add('read');

                        fetch('notifications.php', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/json', 'X-Requested-With': 'XMLHttpRequest' },
                            body: JSON.stringify({ action: 'mark_read', id: notificationId })
                        }).then(response => {
                            if (!response.ok) {
                                console.error('Server error: Could not mark notification as read.');
                                // Revert UI on failure
                                item.classList.add('unread');
                                item.classList.remove('read');
                            }
                        }).catch(error => console.error('Fetch error:', error));
                    }
                });
            }
        });
    </script>
</body>
</html>