<?php
session_start(); // Ensure session is started at the very top
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- VALIDATE INPUT ---
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    die("Invalid Request ID.");
}
$request_id = (int)$_GET['id'];

// --- HANDLE ALL FORM SUBMISSIONS ON THIS PAGE ---
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // --- ACTION 1: Update Status and Notes ---
    if (isset($_POST['action']) && $_POST['action'] === 'update_status') {
        $new_status = $_POST['status'];
        $admin_notes = $_POST['admin_notes'];
        $voucher_id = !empty($_POST['voucher_id']) ? (int)$_POST['voucher_id'] : null;

        $stmt = $conn->prepare("UPDATE voucher_requests SET status = ?, admin_notes = ?, voucher_id = ? WHERE id = ?");
        $stmt->bind_param("ssii", $new_status, $admin_notes, $voucher_id, $request_id);

        if ($stmt->execute()) {
            $_SESSION['success_message'] = "Request #" . $request_id . " has been updated.";
        } else {
            $_SESSION['error_message'] = "Failed to update request.";
        }
        // Refresh the page to show updated info
        header("Location: process-request.php?id=" . $request_id);
        exit();
    }

    // --- ACTION 2: Prepare data and redirect to Create Invoice page ---
    if (isset($_POST['action']) && $_POST['action'] === 'create_invoice') {
        // Fetch the full request details again to ensure data is fresh
        $stmt_fetch = $conn->prepare("SELECT * FROM voucher_requests WHERE id = ?");
        $stmt_fetch->bind_param("i", $request_id);
        $stmt_fetch->execute();
        $request_data = $stmt_fetch->get_result()->fetch_assoc();

        if ($request_data) {
            $mutamers = $conn->query("SELECT * FROM request_mutamers WHERE request_id = $request_id")->fetch_all(MYSQLI_ASSOC);
            $accommodations = $conn->query("SELECT * FROM request_accommodations WHERE request_id = $request_id")->fetch_all(MYSQLI_ASSOC);

            // Package the data for the session
            $prefill_data = [
                'user_id' => $request_data['agent_id'],
                'guest_name' => $request_data['family_head_name'],
                'pilgrims' => [],
                'hotels' => []
            ];

            foreach ($mutamers as $mutamer) {
                $prefill_data['pilgrims'][] = [
                    'passenger_details' => $mutamer['mutamer_name'],
                    'passport_no' => $mutamer['passport_no'],
                    'pax' => $mutamer['pax_type'],
                    'bed' => $mutamer['bed_required']
                ];
            }

            foreach ($accommodations as $accom) {
                $prefill_data['hotels'][] = [
                    'city' => $accom['city'],
                    'hotel_name' => $accom['hotel_preference'],
                    'check_in' => $accom['check_in_date'],
                    'nights' => $accom['nights'],
                    'room_type' => $accom['room_type'],
                    'meal_plan' => $accom['meal_plan']
                ];
            }

            // Store in session and redirect
            $_SESSION['invoice_prefill_data'] = $prefill_data;
            header("Location: create-invoice.php");
            exit();
        }
    }
}


// --- FETCH ALL REQUEST DATA FOR DISPLAY (GET Request) ---
$stmt_req = $conn->prepare("SELECT vr.*, u.name as agent_name FROM voucher_requests vr JOIN users u ON vr.agent_id = u.id WHERE vr.id = ?");
$stmt_req->bind_param("i", $request_id);
$stmt_req->execute();
$request = $stmt_req->get_result()->fetch_assoc();

if (!$request) {
    die("Request not found.");
}

$request['mutamers'] = $conn->query("SELECT * FROM request_mutamers WHERE request_id = $request_id")->fetch_all(MYSQLI_ASSOC);
$request['accommodations'] = $conn->query("SELECT * FROM request_accommodations WHERE request_id = $request_id")->fetch_all(MYSQLI_ASSOC);

?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Process Request #<?= e($request_id) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .processing-layout {
            display: flex;
            gap: 2rem;
            align-items: flex-start;
        }

        .request-details-panel {
            flex: 2;
        }

        .admin-actions-panel {
            flex: 1;
            position: sticky;
            top: 20px;
        }

        .view-section {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            padding: 1.5rem;
            background-color: #fdfdfd;
        }

        .view-section h3 {
            margin-top: 0;
            border-bottom: 1px solid #eee;
            padding-bottom: 0.5rem;
            margin-bottom: 1rem;
            font-size: 1.2em;
            color: var(--primary-color);
        }

        .view-section dl {
            display: grid;
            grid-template-columns: max-content 1fr;
            gap: 10px 20px;
        }

        .view-section dt {
            font-weight: bold;
            color: #555;
        }

        .view-section dd {
            margin: 0;
        }

        .btn-create-item {
            display: block;
            width: 100%;
            text-align: center;
            margin-top: 1rem;
        }

        .btn-create-invoice {
            background-color: #3498db;
            border-color: #3498db;
        }

        .btn-create-invoice:hover {
            background-color: #2980b9;
        }

        .btn-create-voucher {
            background-color: #27ae60;
            border-color: #27ae60;
        }

        .btn-create-voucher:hover {
            background-color: #2ecc71;
        }
        @media (max-width: 992px) {
            .processing-layout {
                flex-direction: column;
            }
            .admin-actions-panel {
                position: static;
                width: 100%;
            }
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo e(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body">
                <div class="page-header">
                    <h1 class="page-title">Process Voucher Request #<?= e($request_id) ?></h1>
                    <a href="manage-agent-requests.php" class="btn btn-secondary"><i class="fa-solid fa-arrow-left"></i> Back to List</a>
                </div>

                <?php if (isset($_SESSION['success_message'])): ?>
                    <div class="notice success"><?= e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
                <?php endif; ?>
                <?php if (isset($_SESSION['error_message'])): ?>
                    <div class="notice error"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                <?php endif; ?>


                <div class="processing-layout">
                    <div class="request-details-panel">
                        <div class="view-section">
                            <h3><i class="fas fa-info-circle"></i> Basic Information</h3>
                            <dl>
                                <dt>Agent:</dt>
                                <dd><?= e($request['agent_name']) ?></dd>
                                <dt>Family Head:</dt>
                                <dd><?= e($request['family_head_name']) ?></dd>
                                <dt>Package Type:</dt>
                                <dd><?= e($request['package_type'] ?: 'N/A') ?></dd>
                                <dt>Duration:</dt>
                                <dd><?= e($request['package_duration_nights'] ? $request['package_duration_nights'] . ' Nights' : 'N/A') ?></dd>
                                <dt>Pax Summary:</dt>
                                <dd><?= e($request['pax_summary'] ?: 'N/A') ?></dd>
                            </dl>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-calendar-alt"></i> Desired Travel Dates</h3>
                            <dl>
                                <dt>Departure:</dt>
                                <dd><?= e($request['desired_departure_date'] ? date('M j, Y', strtotime($request['desired_departure_date'])) : 'N/A') ?></dd>
                                <dt>Return:</dt>
                                <dd><?= e($request['desired_return_date'] ? date('M j, Y', strtotime($request['desired_return_date'])) : 'N/A') ?></dd>
                            </dl>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-hotel"></i> Accommodation Preferences</h3><?php if (!empty($request['accommodations'])): ?><div class="table-responsive">
                                    <table class="data-table">
                                        <thead>
                                            <tr>
                                                <th>City</th>
                                                <th>Preference</th>
                                                <th>Check-In</th>
                                                <th>Nights</th>
                                                <th>Room Type</th>
                                                <th>Meal Plan</th>
                                            </tr>
                                        </thead>
                                        <tbody><?php foreach ($request['accommodations'] as $accom): echo '<tr><td>' . e($accom['city']) . '</td><td>' . nl2br(e($accom['hotel_preference'])) . '</td><td>' . e($accom['check_in_date'] ? date('M j, Y', strtotime($accom['check_in_date'])) : 'N/A') . '</td><td>' . e($accom['nights'] ?: 'N/A') . '</td><td>' . e($accom['room_type'] ?: 'N/A') . '</td><td>' . e($accom['meal_plan'] ?: 'N/A') . '</td></tr>';
                                                                                                endforeach; ?></tbody>
                                    </table>
                                </div><?php else: ?><p>No accommodation preferences were specified.</p><?php endif; ?>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-users"></i> Traveler Details</h3><?php if (!empty($request['mutamers'])): ?><div class="table-responsive">
                                    <table class="data-table">
                                        <thead>
                                            <tr>
                                                <th>Name</th>
                                                <th>Passport #</th>
                                                <th>Pax Type</th>
                                                <th>Bed Required</th>
                                            </tr>
                                        </thead>
                                        <tbody><?php foreach ($request['mutamers'] as $mutamer): echo '<tr><td>' . e($mutamer['mutamer_name']) . '</td><td>' . e($mutamer['passport_no'] ?: 'N/A') . '</td><td>' . e($mutamer['pax_type']) . '</td><td>' . e($mutamer['bed_required']) . '</td></tr>';
                                                                                        endforeach; ?></tbody>
                                    </table>
                                </div><?php else: ?><p>No travelers were listed.</p><?php endif; ?>
                        </div>
                        <div class="view-section">
                            <h3><i class="fas fa-sticky-note"></i> Agent Notes</h3>
                            <p><?= nl2br(e($request['agent_notes'] ?: 'No special notes provided.')) ?></p>
                        </div>
                    </div>

                    <div class="admin-actions-panel">
                        <div class="content-card">
                            <h3>Admin Actions</h3>
                            <form action="process-request.php?id=<?= e($request_id) ?>" method="POST" class="styled-form">
                                <input type="hidden" name="action" value="update_status">
                                <div class="form-group">
                                    <label for="status">Update Status</label>
                                    <select name="status" id="status" class="form-control">
                                        <option value="Pending" <?= $request['status'] == 'Pending' ? 'selected' : '' ?>>Pending</option>
                                        <option value="In Progress" <?= $request['status'] == 'In Progress' ? 'selected' : '' ?>>In Progress</option>
                                        <option value="Completed" <?= $request['status'] == 'Completed' ? 'selected' : '' ?>>Completed</option>
                                        <option value="Rejected" <?= $request['status'] == 'Rejected' ? 'selected' : '' ?>>Rejected</option>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label for="admin_notes">Admin Notes (visible to agent)</label>
                                    <textarea name="admin_notes" id="admin_notes" rows="5" class="form-control"><?= e($request['admin_notes']) ?></textarea>
                                </div>
                                <div class="form-group">
                                    <label for="voucher_id">Link to Official Voucher ID (Optional)</label>
                                    <input type="number" name="voucher_id" id="voucher_id" class="form-control" value="<?= e($request['voucher_id']) ?>" placeholder="e.g., 101">
                                </div>
                                <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Save Changes</button>
                            </form>
                            <hr>

                            <!-- Create Invoice Button Form -->
                            <form action="process-request.php?id=<?= e($request_id) ?>" method="POST">
                                <input type="hidden" name="action" value="create_invoice">
                                <button type="submit" class="btn btn-primary btn-create-item btn-create-invoice">
                                    <i class="fas fa-file-invoice"></i> Create Invoice from Request
                                </button>
                            </form>

                            <!-- Create Voucher Button Link -->
                            <a href="create-voucher.php?request_id=<?= e($request_id) ?>" class="btn btn-primary btn-create-item btn-create-voucher">
                                <i class="fas fa-ticket-alt"></i> Create Official Voucher
                            </a>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const menuToggle = document.getElementById('menu-toggle');
            if (menuToggle) {
                menuToggle.addEventListener('click', function () {
                    document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
                });
            }
        });
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>