<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}
function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

$invoice_id = (int)($_GET['id'] ?? 0);
if ($invoice_id <= 0) {
    $_SESSION['error_message'] = "Invalid Ticket Invoice ID.";
    header("Location: ticket-invoices-list.php");
    exit;
}

// --- FETCH DATA FOR DROPDOWNS ---
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- HANDLE FORM SUBMISSION: UPDATE INVOICE ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action']) && $_POST['action'] === 'update_invoice') {
    $conn->begin_transaction();
    try {
        // Clear existing child records to re-insert them cleanly
        $conn->execute_query("DELETE FROM ticket_invoice_passengers WHERE ticket_invoice_id = ?", [$invoice_id]);
        $conn->execute_query("DELETE FROM ticket_invoice_flights WHERE ticket_invoice_id = ?", [$invoice_id]);

        // Process main invoice data
        $user_id_to_save = !empty($_POST['user_id']) ? (int)$_POST['user_id'] : null;
        $vendor_id_to_save = !empty($_POST['vendor_id']) ? (int)$_POST['vendor_id'] : null;
        
        // Server-side recalculation for data integrity
        $adult_qty = (int)($_POST['adult_qty'] ?? 0); $adult_rate = (float)($_POST['adult_rate'] ?? 0); $adult_cost = (float)($_POST['adult_cost'] ?? 0);
        $child_qty = (int)($_POST['child_qty'] ?? 0); $child_rate = (float)($_POST['child_rate'] ?? 0); $child_cost = (float)($_POST['child_cost'] ?? 0);
        $infant_qty = (int)($_POST['infant_qty'] ?? 0); $infant_rate = (float)($_POST['infant_rate'] ?? 0); $infant_cost = (float)($_POST['infant_cost'] ?? 0);
        $service_fee = (float)($_POST['service_fee_pkr'] ?? 0); $vendor_commission = (float)($_POST['vendor_commission_pkr'] ?? 0); $discount = (float)($_POST['discount_pkr'] ?? 0);
        
        $total_fare = ($adult_qty * $adult_rate) + ($child_qty * $child_rate) + ($infant_qty * $infant_rate);
        $grand_total = ($total_fare + $service_fee) - $discount;
        $total_fare_cost = ($adult_qty * $adult_cost) + ($child_qty * $child_cost) + ($infant_qty * $infant_cost);
        $grand_total_cost = ($total_fare_cost + $service_fee) - $vendor_commission;

        // UPDATE main invoice table
        $sql_update = "UPDATE ticket_invoices SET user_id=?, vendor_id=?, guest_name=?, invoice_number=?, issue_date=?, status=?, adult_qty=?, adult_rate=?, adult_cost=?, child_qty=?, child_rate=?, child_cost=?, infant_qty=?, infant_rate=?, infant_cost=?, total_fare_pkr=?, total_fare_pkr_cost=?, service_fee_pkr=?, vendor_commission_pkr=?, discount_pkr=?, grand_total_pkr=?, grand_total_pkr_cost=?, notes=? WHERE id=?";
        $stmt_update = $conn->prepare($sql_update);
        $stmt_update->bind_param("iissssiddisddiddddddddsi", $user_id_to_save, $vendor_id_to_save, $_POST['guest_name'], $_POST['invoice_number'], $_POST['issue_date'], $_POST['status'], $adult_qty, $adult_rate, $adult_cost, $child_qty, $child_rate, $child_cost, $infant_qty, $infant_rate, $infant_cost, $total_fare, $total_fare_cost, $service_fee, $vendor_commission, $discount, $grand_total, $grand_total_cost, $_POST['notes'], $invoice_id);
        $stmt_update->execute();

        // RE-INSERT passengers
        if (!empty($_POST['passenger_name'])) {
            $sql_pass = "INSERT INTO ticket_invoice_passengers (ticket_invoice_id, full_name, passenger_type, passport_no, pnr, ticket_number, dob, passport_issue_date, passport_expiry_date) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_pass = $conn->prepare($sql_pass);
            foreach ($_POST['passenger_name'] as $key => $name) {
                if (!empty(trim($name))) {
                    $stmt_pass->bind_param("issssssss", $invoice_id, $name, $_POST['passenger_type'][$key], $_POST['passenger_passport'][$key], $_POST['passenger_pnr'][$key], $_POST['passenger_ticket_no'][$key], $_POST['passenger_dob'][$key], $_POST['passenger_issue_date'][$key], $_POST['passenger_expiry_date'][$key]);
                    $stmt_pass->execute();
                }
            }
        }
        // RE-INSERT flights
        if (!empty($_POST['flight_airline'])) {
            $sql_flight = "INSERT INTO ticket_invoice_flights (ticket_invoice_id, airline, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt_flight = $conn->prepare($sql_flight);
            foreach ($_POST['flight_airline'] as $key => $airline) {
                if (!empty(trim($airline))) {
                    $stmt_flight->bind_param("isssss", $invoice_id, $airline, $_POST['flight_no'][$key], $_POST['flight_sector'][$key], $_POST['flight_departure'][$key], $_POST['flight_arrival'][$key]);
                    $stmt_flight->execute();
                }
            }
        }
        $conn->commit();
        $_SESSION['success_message'] = "Invoice #" . e($_POST['invoice_number']) . " updated successfully!";
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error updating invoice: " . $e->getMessage();
    }
    header("Location: ticket-invoice-edit.php?id=" . $invoice_id);
    exit();
}

// --- FETCH ALL DATA FOR PAGE DISPLAY ---
$invoice = $conn->query("SELECT * FROM ticket_invoices WHERE id = $invoice_id")->fetch_assoc();
if (!$invoice) {
    $_SESSION['error_message'] = "Invoice not found.";
    header("Location: ticket-invoices-list.php");
    exit;
}
$passengers = $conn->query("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = $invoice_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
$flights = $conn->query("SELECT * FROM ticket_invoice_flights WHERE ticket_invoice_id = $invoice_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);

$success_message = $_SESSION['success_message'] ?? null; unset($_SESSION['success_message']);
$error_message = $_SESSION['error_message'] ?? null; unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Ticket Invoice #<?= e($invoice['invoice_number']) ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
<style>
        .main-content { min-width: 0; }
        .content-body { padding: 1.5rem; }
        .dynamic-table th, .dynamic-table td { white-space: nowrap; vertical-align: middle; }
        .summary-grid { display: grid; grid-template-columns: 1fr auto; gap: 5px 20px; align-items: center;}

        /* --- NEW STYLES FOR TABLE INPUTS AND SELECTS --- */
        .dynamic-table .form-control,
        .dynamic-table .form-select {
            padding: 0.375rem 0.75rem; /* Standard Bootstrap padding for better spacing */
            min-width: 150px;       /* Minimum width to prevent squishing */
        }
        .dynamic-table .form-select {
            -webkit-appearance: auto; /* Fixes dropdown arrow on some browsers */
            appearance: auto;
            background-image: none; /* Hides the extra default browser arrow */
        }
    </style>
</head>
<body>
<div class="dashboard-wrapper">
    <?php include 'sidebar.php'; ?>
    <div class="main-content">
        <header class="main-header">
            <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
            <div class="user-info"><span>Edit Ticket Invoice #<?= e($invoice['invoice_number']) ?></span></div>
        </header>
        <main class="content-body">
            <div class="d-flex justify-content-between align-items-center mb-3">
                <a href="ticket-invoices-list.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to List</a>
                <a href="ticket-invoice-view.php?id=<?= $invoice_id ?>" class="btn btn-info" target="_blank"><i class="fas fa-eye"></i> View Invoice</a>
            </div>

            <?php if ($success_message): ?><div class="alert alert-success"><?= e($success_message) ?></div><?php endif; ?>
            <?php if ($error_message): ?><div class="alert alert-danger"><?= e($error_message) ?></div><?php endif; ?>

            <form action="ticket-invoice-edit.php?id=<?= $invoice_id ?>" method="POST" id="invoice-main-form">
                <input type="hidden" name="action" value="update_invoice">

                <div class="card mb-4">
                    <div class="card-header"><h3 class="h5 mb-0"><i class="fas fa-file-invoice me-2"></i>Invoice Details</h3></div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-4"><label class="form-label">Invoice No</label><input type="text" class="form-control" name="invoice_number" value="<?= e($invoice['invoice_number']) ?>"></div>
                            <div class="col-md-4"><label class="form-label">Dated*</label><input type="date" class="form-control" name="issue_date" required value="<?= e($invoice['issue_date']) ?>"></div>
                            <div class="col-md-4"><label class="form-label">Status</label>
                                <select name="status" class="form-select">
                                    <option value="Tentative" <?= $invoice['status'] == 'Tentative' ? 'selected' : '' ?>>Tentative</option>
                                    <option value="Approved" <?= $invoice['status'] == 'Approved' ? 'selected' : '' ?>>Approved</option>
                                    <option value="Cancelled" <?= $invoice['status'] == 'Cancelled' ? 'selected' : '' ?>>Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-4"><label class="form-label">Lead Passenger*</label><input type="text" class="form-control" name="guest_name" required value="<?= e($invoice['guest_name']) ?>"></div>
                            <div class="col-md-4"><label class="form-label">Assign User</label>
                                <select name="user_id" class="form-select">
                                    <option value="">-- Direct Customer --</option>
                                    <?php foreach ($users_list as $user): ?><option value="<?= $user['id'] ?>" <?= $invoice['user_id'] == $user['id'] ? 'selected' : '' ?>><?= e($user['name']) ?> [<?= substr($user['user_type'], 0, 1) ?>]</option><?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4"><label class="form-label">Assign Vendor</label>
                                <select name="vendor_id" class="form-select">
                                    <option value="">-- No Vendor --</option>
                                    <?php foreach ($vendors_list as $vendor): ?><option value="<?= $vendor['id'] ?>" <?= $invoice['vendor_id'] == $vendor['id'] ? 'selected' : '' ?>><?= e($vendor['name']) ?></option><?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card mb-4">
                    <div class="card-header"><h3 class="h5 mb-0"><i class="fas fa-users me-2"></i>Passenger Information</h3></div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered dynamic-table">
                                <thead class="table-light"><tr><th>Name*</th><th>Type</th><th>Passport</th><th>DOB</th><th>Issue Date</th><th>Expiry Date</th><th>PNR</th><th>Ticket No.</th><th>Action</th></tr></thead>
                                <tbody id="passengers-tbody">
                                    <?php foreach ($passengers as $p): ?>
                                    <tr>
                                        <td><input type="text" class="form-control" name="passenger_name[]" required value="<?= e($p['full_name']) ?>"></td>
                                        <td><select class="form-select" name="passenger_type[]"><option value="Adult" <?= $p['passenger_type']=='Adult'?'selected':''?>>Adult</option><option value="Child" <?= $p['passenger_type']=='Child'?'selected':''?>>Child</option><option value="Infant" <?= $p['passenger_type']=='Infant'?'selected':''?>>Infant</option></select></td>
                                        <td><input type="text" class="form-control" name="passenger_passport[]" value="<?= e($p['passport_no']) ?>"></td>
                                        <td><input type="date" class="form-control" name="passenger_dob[]" value="<?= e($p['dob']) ?>"></td>
                                        <td><input type="date" class="form-control" name="passenger_issue_date[]" value="<?= e($p['passport_issue_date']) ?>"></td>
                                        <td><input type="date" class="form-control" name="passenger_expiry_date[]" value="<?= e($p['passport_expiry_date']) ?>"></td>
                                        <td><input type="text" class="form-control" name="passenger_pnr[]" value="<?= e($p['pnr']) ?>"></td>
                                        <td><input type="text" class="form-control" name="passenger_ticket_no[]" value="<?= e($p['ticket_number']) ?>"></td>
                                        <td class="text-center"><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <button type="button" class="btn btn-secondary mt-2" onclick="addPassengerRow()"><i class="fas fa-plus"></i> Add Passenger</button>
                    </div>
                </div>

                <div class="card mb-4">
                    <div class="card-header"><h3 class="h5 mb-0"><i class="fas fa-plane me-2"></i>Flight Itinerary</h3></div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered dynamic-table">
                                <thead class="table-light"><tr><th>Airline*</th><th>Flight No.</th><th>Sector</th><th>Departure</th><th>Arrival</th><th>Action</th></tr></thead>
                                <tbody id="flights-tbody">
                                    <?php foreach ($flights as $f): ?>
                                    <tr>
                                        <td><input type="text" class="form-control" name="flight_airline[]" required value="<?= e($f['airline']) ?>"></td>
                                        <td><input type="text" class="form-control" name="flight_no[]" value="<?= e($f['flight_no']) ?>"></td>
                                        <td><input type="text" class="form-control" name="flight_sector[]" value="<?= e($f['sector']) ?>"></td>
                                        <td><input type="datetime-local" class="form-control" name="flight_departure[]" value="<?= !empty($f['departure_datetime']) ? date('Y-m-d\TH:i', strtotime($f['departure_datetime'])) : '' ?>"></td>
                                        <td><input type="datetime-local" class="form-control" name="flight_arrival[]" value="<?= !empty($f['arrival_datetime']) ? date('Y-m-d\TH:i', strtotime($f['arrival_datetime'])) : '' ?>"></td>
                                        <td class="text-center"><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <button type="button" class="btn btn-secondary mt-2" onclick="addFlightRow()"><i class="fas fa-plus"></i> Add Flight</button>
                    </div>
                </div>

                <div class="card mb-4">
                    <div class="card-header"><h3 class="h5 mb-0"><i class="fas fa-calculator me-2"></i>Financials</h3></div>
                    <div class="card-body">
                        <div class="row g-4">
                            <div class="col-lg-7">
                                <h4 class="h6">Fare Breakdown</h4>
                                <div class="row g-2 mb-2 d-none d-md-flex small text-muted"><div class="col">Type</div><div class="col">Qty</div><div class="col">Sale Rate</div><div class="col">Cost</div></div>
                                <div class="row g-2 mb-2 align-items-center">
                                    <div class="col-12 col-md"><label class="form-label d-md-none">Adults</label></div>
                                    <div class="col-md"><input type="number" class="form-control" id="adult_qty" name="adult_qty" value="<?= e($invoice['adult_qty']) ?>" min="0" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="adult_rate" name="adult_rate" value="<?= e($invoice['adult_rate']) ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="adult_cost" name="adult_cost" value="<?= e($invoice['adult_cost'] ?? '0.00') ?>" step="0.01" oninput="calculateTotals()"></div>
                                </div>
                                <div class="row g-2 mb-2 align-items-center">
                                    <div class="col-12 col-md"><label class="form-label d-md-none">Children</label></div>
                                    <div class="col-md"><input type="number" class="form-control" id="child_qty" name="child_qty" value="<?= e($invoice['child_qty']) ?>" min="0" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="child_rate" name="child_rate" value="<?= e($invoice['child_rate']) ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="child_cost" name="child_cost" value="<?= e($invoice['child_cost'] ?? '0.00') ?>" step="0.01" oninput="calculateTotals()"></div>
                                </div>
                                <div class="row g-2 align-items-center">
                                    <div class="col-12 col-md"><label class="form-label d-md-none">Infants</label></div>
                                    <div class="col-md"><input type="number" class="form-control" id="infant_qty" name="infant_qty" value="<?= e($invoice['infant_qty']) ?>" min="0" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="infant_rate" name="infant_rate" value="<?= e($invoice['infant_rate']) ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <div class="col-md"><input type="number" class="form-control" id="infant_cost" name="infant_cost" value="<?= e($invoice['infant_cost'] ?? '0.00') ?>" step="0.01" oninput="calculateTotals()"></div>
                                </div>
                            </div>
                            <div class="col-lg-5">
                                <h4 class="h6">Summary</h4>
                                <div class="bg-light p-3 border rounded summary-grid">
                                    <div class="summary-label">Total Fare:</div><div id="summary-total-fare" class="text-end fw-bold">0.00</div>
                                    <div class="summary-label">Service Fee:</div><div><input type="number" name="service_fee_pkr" id="service-fee-pkr-input" class="form-control form-control-sm text-end" value="<?= e($invoice['service_fee_pkr']) ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <div class="summary-label">Discount:</div><div><input type="number" name="discount_pkr" id="discount-pkr-input" class="form-control form-control-sm text-end" value="<?= e($invoice['discount_pkr']) ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <hr style="grid-column: 1 / -1; margin: 5px 0;">
                                    <div class="summary-label grand-total text-success">Grand Total:</div><div class="grand-total text-end text-success" id="summary-grand-total">0.00</div>
                                    <hr style="grid-column: 1 / -1; margin: 10px 0; border-style: dashed;">
                                    <div class="summary-label text-danger">Vendor Commission:</div><div><input type="number" name="vendor_commission_pkr" id="vendor-commission-pkr-input" class="form-control form-control-sm text-end" value="<?= e($invoice['vendor_commission_pkr'] ?? '0.00') ?>" step="0.01" oninput="calculateTotals()"></div>
                                    <div class="summary-label text-danger">Total Cost:</div><div class="text-end text-danger fw-bold" id="summary-total-cost">0.00</div>
                                    <div class="summary-label grand-total text-primary">Net Profit:</div><div class="grand-total text-end text-primary" id="summary-net-profit">0.00</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card mb-4">
                    <div class="card-header"><h3 class="h5 mb-0"><i class="fas fa-sticky-note me-2"></i>Notes</h3></div>
                    <div class="card-body"><textarea name="notes" class="form-control" rows="4"><?= e($invoice['notes']) ?></textarea></div>
                </div>

                <div class="text-center my-4"><button type="submit" class="btn btn-primary btn-lg"><i class="fas fa-save me-2"></i>Update Invoice</button></div>
            </form>
        </main>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
    const addRow = (tbodyId, rowHTML) => document.getElementById(tbodyId).insertRow().innerHTML = rowHTML;
    const removeRow = (btn) => { btn.closest('tr').remove(); calculateTotals(); };

    function addPassengerRow() {
        addRow('passengers-tbody', `<td><input type="text" class="form-control" name="passenger_name[]" required></td><td><select class="form-select" name="passenger_type[]"><option value="Adult" selected>Adult</option><option value="Child">Child</option><option value="Infant">Infant</option></select></td><td><input type="text" class="form-control" name="passenger_passport[]"></td><td><input type="date" class="form-control" name="passenger_dob[]"></td><td><input type="date" class="form-control" name="passenger_issue_date[]"></td><td><input type="date" class="form-control" name="passenger_expiry_date[]"></td><td><input type="text" class="form-control" name="passenger_pnr[]"></td><td><input type="text" class="form-control" name="passenger_ticket_no[]"></td><td class="text-center"><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>`);
    }

    function addFlightRow() {
        addRow('flights-tbody', `<td><input type="text" class="form-control" name="flight_airline[]" required></td><td><input type="text" class="form-control" name="flight_no[]"></td><td><input type="text" class="form-control" name="flight_sector[]"></td><td><input type="datetime-local" class="form-control" name="flight_departure[]"></td><td><input type="datetime-local" class="form-control" name="flight_arrival[]"></td><td class="text-center"><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)"><i class="fas fa-trash"></i></button></td>`);
    }

    function calculateTotals() {
        const format = (num) => isNaN(num) ? '0.00' : num.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
        const getVal = (id) => parseFloat(document.getElementById(id).value) || 0;
        const totalFare = (getVal('adult_qty') * getVal('adult_rate')) + (getVal('child_qty') * getVal('child_rate')) + (getVal('infant_qty') * getVal('infant_rate'));
        const totalCost = (getVal('adult_qty') * getVal('adult_cost')) + (getVal('child_qty') * getVal('child_cost')) + (getVal('infant_qty') * getVal('infant_cost'));
        const serviceFee = getVal('service-fee-pkr-input');
        const vendorCommission = getVal('vendor-commission-pkr-input');
        const discount = getVal('discount-pkr-input');
        const grandTotal = (totalFare + serviceFee) - discount;
        const grandTotalCost = (totalCost + serviceFee) - vendorCommission;
        const netProfit = grandTotal - grandTotalCost;

        document.getElementById('summary-total-fare').textContent = format(totalFare);
        document.getElementById('summary-grand-total').textContent = format(grandTotal);
        document.getElementById('summary-total-cost').textContent = format(grandTotalCost);
        document.getElementById('summary-net-profit').textContent = format(netProfit);
    }
    document.addEventListener('DOMContentLoaded', () => {
        if (document.getElementById('passengers-tbody').rows.length === 0) addPassengerRow();
        if (document.getElementById('flights-tbody').rows.length === 0) addFlightRow();
        calculateTotals();
        document.getElementById('menu-toggle').addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
    });
</script>
</body>
</html>