<?php
session_start();
include_once '../db-config.php';

if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- API endpoint for status updates and bulk delete ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- AJAX status update ---
    if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if ($data['action'] === 'update_status') {
                $stmt = $conn->prepare("UPDATE custom_package_inquiries SET status = ? WHERE id = ?");
                $stmt->bind_param("si", $data['status'], $data['id']);
                if ($stmt->execute()) echo json_encode(['success' => true]);
                else throw new Exception($stmt->error);
                $stmt->close();
            }
        } catch (Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
        exit();
    }

    // --- BULK DELETE LOGIC ---
    if (isset($_POST['action']) && $_POST['action'] === 'bulk_delete') {
        if (empty($_POST['inquiry_ids']) || !is_array($_POST['inquiry_ids'])) {
            $_SESSION['error_message'] = "Please select inquiries to delete.";
        } else {
            $ids_to_delete = array_map('intval', $_POST['inquiry_ids']);
            $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
            $types = str_repeat('i', count($ids_to_delete));
            
            $sql = "DELETE FROM custom_package_inquiries WHERE id IN ($placeholders)";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param($types, ...$ids_to_delete);
            
            if ($stmt->execute()) {
                $deleted_count = $stmt->affected_rows;
                $_SESSION['success_message'] = "Successfully deleted {$deleted_count} inquiries.";
            } else {
                $_SESSION['error_message'] = "Error deleting inquiries: " . $stmt->error;
            }
            $stmt->close();
        }
        header("Location: " . $_SERVER['PHP_SELF'] . "?" . $_SERVER['QUERY_STRING']);
        exit();
    }
}

// --- Fetch data for page display with filtering and pagination ---
$allowed_filters = ['all', 'pending', 'in-progress', 'completed', 'rejected'];
$current_filter = $_GET['filter'] ?? 'all';
if (!in_array($current_filter, $allowed_filters)) $current_filter = 'all';

$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

$base_sql_from = "FROM custom_package_inquiries";
$where_clause = "";
$params = [];
$types = "";
if ($current_filter !== 'all') {
    $status_for_query = str_replace('-', ' ', ucwords($current_filter, '-'));
    $where_clause = " WHERE status = ?";
    $params[] = $status_for_query;
    $types = "s";
}

// Get total count for pagination
$count_sql = "SELECT COUNT(*) as total " . $base_sql_from . $where_clause;
$stmt_count = $conn->prepare($count_sql);
if (!empty($params)) $stmt_count->bind_param($types, ...$params);
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$stmt_count->close();
$total_pages = ($total_items > 0) ? ceil($total_items / $items_per_page) : 1;

// Get counts for filter bar
$all_inquiries_count = $conn->query("SELECT COUNT(*) FROM custom_package_inquiries")->fetch_row()[0];
$pending_count = $conn->query("SELECT COUNT(*) FROM custom_package_inquiries WHERE status = 'Pending'")->fetch_row()[0];
$inprogress_count = $conn->query("SELECT COUNT(*) FROM custom_package_inquiries WHERE status = 'In Progress'")->fetch_row()[0];
$completed_count = $conn->query("SELECT COUNT(*) FROM custom_package_inquiries WHERE status = 'Completed'")->fetch_row()[0];
$rejected_count = $conn->query("SELECT COUNT(*) FROM custom_package_inquiries WHERE status = 'Rejected'")->fetch_row()[0];

// Get data for the current page
$data_sql = "SELECT * " . $base_sql_from . $where_clause . " ORDER BY created_at DESC LIMIT ?, ?";
$data_params = $params;
$data_params[] = $offset;
$data_params[] = $items_per_page;
$data_types = $types . "ii";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($data_types, ...$data_params);
$stmt_data->execute();
$inquiries_result = $stmt_data->get_result();
$stmt_data->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Custom Package Inquiries</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-bar { display: flex; flex-wrap: wrap; gap: 8px; margin-bottom: 25px; }
        .filter-bar a { text-decoration: none; padding: 8px 15px; border-radius: 6px; color: var(--text-dark); font-weight: 600; font-size: 0.9rem; transition: all 0.2s ease; display: flex; align-items: center; gap: 8px; background-color: #e9ecef; }
        .filter-bar a:hover { background-color: #dee2e6; }
        .filter-bar a.active { background-color: var(--primary-color); color: #fff; }
        .count-badge { background-color: var(--secondary-color); color: var(--sidebar-bg); font-size: 0.8rem; padding: 2px 8px; border-radius: 10px; }
        .toolbar { margin-bottom: 1rem; display: flex; align-items: center; gap: 1rem; }

        .inquiry-list .inquiry-item { padding: 15px 20px; border-bottom: 1px solid var(--border-color); background-color: #fff; }
        .inquiry-summary { display: flex; align-items: center; gap: 15px; cursor: pointer; }
        .inquiry-primary-info { flex-grow: 1; }
        .inquiry-meta { display: flex; align-items: center; gap: 15px; }
        .expand-arrow { transition: transform 0.3s; }
        .inquiry-item.expanded .expand-arrow { transform: rotate(180deg); }
        .inquiry-details { max-height: 0; overflow: hidden; transition: max-height 0.4s ease-in-out, padding 0.4s ease-in-out, margin 0.4s ease-in-out; }
        .inquiry-item.expanded .inquiry-details { max-height: 1500px; padding-top: 20px; margin-top: 20px; border-top: 1px solid #e9ecef; }
        .details-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(280px, 1fr)); gap: 15px 30px; }
        .detail-item strong { display: block; }
        .package-details-table { width: 100%; margin-top: 1rem; border-collapse: collapse; }
        .package-details-table th, .package-details-table td { text-align: left; padding: 12px; border-bottom: 1px solid #eee; }
        .package-details-table th { background-color: #f8f9fa; font-weight: 600; color: #495057; font-size: 0.9rem; }
        .package-details-table .total-row td { font-weight: bold; border-top: 2px solid #ddd; }
        .package-details-table .grand-total-row td { background-color: #e9f5fd; color: #0353a4; font-weight: bold; font-size: 1.1rem; }
        .inquiry-actions { margin-top: 20px; padding-top: 15px; border-top: 1px solid #e9ecef; }

        @media screen and (max-width: 768px) {
            .inquiry-summary { flex-wrap: wrap; gap: 10px; }
            .inquiry-meta { width: 100%; justify-content: space-between; margin-top: 10px; }
            .package-details-table thead { display: none; }
            .package-details-table, .package-details-table tbody, .package-details-table tr, .package-details-table td { display: block; width: 100%; }
            .package-details-table tr { margin-bottom: 1rem; border: 1px solid #e9ecef; border-radius: 5px; padding: 10px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); }
            .package-details-table td { text-align: right; padding-left: 50%; position: relative; border-bottom: 1px solid #f1f3f5; }
            .package-details-table td:last-child { border-bottom: none; }
            .package-details-table td::before { content: attr(data-label); position: absolute; left: 10px; width: calc(50% - 20px); text-align: left; font-weight: 600; color: #333; }
            .package-details-table .total-row td, .package-details-table .grand-total-row td { text-align: right !important; }
            .package-details-table .grand-total-row { background-color: transparent; }
             .package-details-table .grand-total-row td { background-color: #e9f5fd; border-radius: 4px; }
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header"><button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button><div class="user-info"><span>Custom Package Inquiries</span></div></header>
            <main class="content-body">
                <h1 class="page-title">Custom Umrah Package Inquiries</h1>

                <?php if(isset($_SESSION['success_message'])): ?>
                    <div class="notice success"><?= e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
                <?php endif; ?>
                <?php if(isset($_SESSION['error_message'])): ?>
                    <div class="notice error"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                <?php endif; ?>

                <div class="filter-bar">
                    <a href="?filter=all" class="<?= $current_filter === 'all' ? 'active' : '' ?>">All (<?= $all_inquiries_count ?>)</a>
                    <a href="?filter=pending" class="<?= $current_filter === 'pending' ? 'active' : '' ?>">Pending <?php if ($pending_count > 0) echo "<span class='count-badge'>$pending_count</span>"; ?></a>
                    <a href="?filter=in-progress" class="<?= $current_filter === 'in-progress' ? 'active' : '' ?>">In Progress (<?= $inprogress_count ?>)</a>
                    <a href="?filter=completed" class="<?= $current_filter === 'completed' ? 'active' : '' ?>">Completed (<?= $completed_count ?>)</a>
                    <a href="?filter=rejected" class="<?= $current_filter === 'rejected' ? 'active' : '' ?>">Rejected (<?= $rejected_count ?>)</a>
                </div>

                <div class="content-card">
                    <form id="bulk-action-form" method="POST">
                        <div class="toolbar">
                            <button type="submit" name="action" value="bulk_delete" class="btn btn-danger"><i class="fas fa-trash"></i> Delete Selected</button>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="select-all">
                                <label class="form-check-label" for="select-all">Select All</label>
                            </div>
                        </div>
                        <div class="inquiry-list" id="inquiry-list">
                            <?php if ($inquiries_result && $inquiries_result->num_rows > 0): while ($inquiry = $inquiries_result->fetch_assoc()): 
                                $details = json_decode($inquiry['package_details_json'], true);
                                $summary = $details['summary'] ?? [];
                                $pax = $details['pax'] ?? [];
                                $components = $details['components'] ?? [];
                            ?>
                                <div class="inquiry-item" data-id="<?= e($inquiry['id']) ?>">
                                    <div class="inquiry-summary">
                                        <input type="checkbox" name="inquiry_ids[]" value="<?php echo $inquiry['id']; ?>" class="form-check-input inquiry-checkbox" style="margin-right: 15px;">
                                        <div class="inquiry-primary-info">
                                            <strong><?= e($inquiry['customer_name'] ?: 'Guest User') ?></strong>
                                            <span>Ref: <?= e($inquiry['booking_ref']) ?> | Submitted: <?= date('d M Y, H:i', strtotime($inquiry['created_at'])) ?></span>
                                        </div>
                                        <div class="inquiry-meta">
                                            <span class="status-badge status-<?= str_replace(' ', '-', e($inquiry['status'])) ?>"><?= e($inquiry['status']) ?></span>
                                            <div class="expand-arrow"><i class="fas fa-chevron-down"></i></div>
                                        </div>
                                    </div>
                                    <div class="inquiry-details">
                                        <div class="details-grid">
                                            <div class="detail-item"><strong>Customer:</strong> <?= e($inquiry['customer_name'] ?: 'N/A') ?></div>
                                            <div class="detail-item"><strong>User ID:</strong> <?= e($inquiry['user_id'] ?: 'N/A') ?></div>
                                            <div class="detail-item"><strong>Pax:</strong> A:<?= e($pax['adults'] ?? 0) ?>, C:<?= e($pax['children'] ?? 0) ?>, I:<?= e($pax['infants'] ?? 0) ?></div>
                                        </div>
                                        <h4 style="margin-top: 1.5rem; font-size: 1rem;">Package Breakdown</h4>
                                        <table class="package-details-table">
                                            <thead><tr><th>Item</th><th>Details</th><th>Cost</th></tr></thead>
                                            <tbody>
                                            <?php foreach($components as $key => $comp): if(empty($comp['cost_sar']) && empty($comp['cost_pkr'])) continue; ?>
                                            <tr>
                                                <td data-label="Item"><?= e(ucfirst(str_replace('_', ' ', $key))) ?></td>
                                                <td data-label="Details"><?= e($comp['details']) ?></td>
                                                <td data-label="Cost"><?= !empty($comp['cost_pkr']) ? 'PKR '.e(number_format($comp['cost_pkr'])) : 'SAR '.e(number_format($comp['cost_sar'], 2)) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <tr class="total-row">
                                                <td data-label="Subtotal">Total (SAR)</td>
                                                <td data-label="Amount" colspan="2"><?= e($summary['grand_total_sar'] ?? 'SAR 0.00') ?></td>
                                            </tr>
                                            <tr class="total-row">
                                                <td data-label="Grand Total">Grand Total (PKR)</td>
                                                <td data-label="Amount" colspan="2"><?= e($summary['grand_total_pkr'] ?? 'PKR 0') ?></td>
                                            </tr>
                                            <tr class="grand-total-row">
                                                <td data-label="Per Person">Total Per Person</td>
                                                <td data-label="Amount" colspan="2"><?= e($summary['total_per_person'] ?? 'PKR 0') ?></td>
                                            </tr>
                                            </tbody>
                                        </table>
                                        <div class="inquiry-actions">
                                            <label>Update Status:</label>
                                            <select class="form-control status-change-select" style="width: auto;">
                                                <option value="Pending" <?= ($inquiry['status'] == 'Pending') ? 'selected' : '' ?>>Pending</option>
                                                <option value="In Progress" <?= ($inquiry['status'] == 'In Progress') ? 'selected' : '' ?>>In Progress</option>
                                                <option value="Completed" <?= ($inquiry['status'] == 'Completed') ? 'selected' : '' ?>>Completed</option>
                                                <option value="Rejected" <?= ($inquiry['status'] == 'Rejected') ? 'selected' : '' ?>>Rejected</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                            <?php endwhile; else: ?>
                                <div class="empty-state"><p>No custom package inquiries found for "<?= e($current_filter) ?>".</p></div>
                            <?php endif; ?>
                        </div>
                    </form>
                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                            <?php if ($current_page > 1): ?><a href="?filter=<?= e($current_filter) ?>&page=<?= $current_page - 1 ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                            <span style="padding: 0 10px;">Page <?php echo $current_page; ?> of <?php echo $total_pages; ?></span>
                            <?php if ($current_page < $total_pages): ?><a href="?filter=<?= e($current_filter) ?>&page=<?= $current_page + 1 ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </main>
        </div>
    </div>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const bulkActionForm = document.getElementById('bulk-action-form');
            const selectAllCheckbox = document.getElementById('select-all');
            const inquiryCheckboxes = document.querySelectorAll('.inquiry-checkbox');

            document.getElementById('inquiry-list').addEventListener('click', function(e) {
                if (!e.target.closest('select, a, input[type=checkbox]')) {
                    const item = e.target.closest('.inquiry-item');
                    if(item) item.classList.toggle('expanded');
                }
            });

            document.querySelectorAll('.status-change-select').forEach(sel => {
                sel.addEventListener('change', function() {
                    const id = this.closest('.inquiry-item').dataset.id;
                    const status = this.value;
                    fetch(window.location.pathname, {
                        method: 'POST',
                        headers: {'Content-Type': 'application/json', 'X-Requested-With': 'XMLHttpRequest'},
                        body: JSON.stringify({action: 'update_status', id: id, status: status})
                    }).then(res => res.json()).then(data => {
                        if(data.success) {
                            const badge = this.closest('.inquiry-item').querySelector('.status-badge');
                            badge.className = 'status-badge status-' + status.replace(' ', '-');
                            badge.textContent = status;
                        } else { alert('Failed to update status.'); }
                    });
                });
            });

            if (selectAllCheckbox) {
                selectAllCheckbox.addEventListener('change', function() {
                    inquiryCheckboxes.forEach(checkbox => {
                        checkbox.checked = this.checked;
                    });
                });
            }

            if (bulkActionForm) {
                bulkActionForm.addEventListener('submit', function(e) {
                    const actionButton = document.activeElement;
                    if (!actionButton || actionButton.value !== 'bulk_delete') return;

                    const anyChecked = [...inquiryCheckboxes].some(c => c.checked);
                    if (!anyChecked) {
                        alert('Please select at least one inquiry to delete.');
                        e.preventDefault();
                        return;
                    }
                    
                    if (!confirm('Are you sure you want to permanently delete the selected inquiries?')) {
                        e.preventDefault();
                    }
                });
            }
        });
    </script>
</body>
</html>