<?php
session_start();
include_once '../db-config.php';

// --- Helper Functions ---
function nf($number) {
    return number_format((float)$number, 0, '.', ',');
}
function nf_decimal($number) {
    $formatted = number_format((float)$number, 2, '.', ',');
    return preg_replace('/\.00$/', '', $formatted);
}

// --- Fetch all invoice data ---
$invoice_id = (int)($_GET['id'] ?? 0);
if ($invoice_id <= 0) die("Invalid Invoice ID.");

$stmt = $conn->prepare("SELECT * FROM invoices WHERE id = ?");
$stmt->bind_param("i", $invoice_id);
$stmt->execute();
$invoice = $stmt->get_result()->fetch_assoc();
if (!$invoice) die("Invoice not found.");

// --- FETCH AGENT'S LOGO if available ---
$logo_to_display = '../images/logo.png'; // Default company logo
if (!empty($invoice['user_id'])) {
    $stmt_agent = $conn->prepare("SELECT logo_path FROM users WHERE id = ?");
    $stmt_agent->bind_param("i", $invoice['user_id']);
    $stmt_agent->execute();
    $agent_result = $stmt_agent->get_result();
    if ($agent = $agent_result->fetch_assoc()) {
        if (!empty($agent['logo_path'])) {
            $logo_to_display = '../uploads/logos/' . htmlspecialchars($agent['logo_path']);
        }
    }
    $stmt_agent->close();
}

// --- Fetch Line Items ---
$pilgrims = $conn->query("SELECT * FROM invoice_pilgrims WHERE invoice_id = $invoice_id")->fetch_all(MYSQLI_ASSOC);
$hotels = $conn->query("SELECT * FROM invoice_hotels WHERE invoice_id = $invoice_id")->fetch_all(MYSQLI_ASSOC);
$transports = $conn->query("SELECT * FROM invoice_transports WHERE invoice_id = $invoice_id")->fetch_all(MYSQLI_ASSOC);
$other_services = $conn->query("SELECT * FROM invoice_other_services WHERE invoice_id = $invoice_id")->fetch_all(MYSQLI_ASSOC);
$airline_tickets = $conn->query("SELECT * FROM invoice_airline_tickets WHERE invoice_id = $invoice_id")->fetch_all(MYSQLI_ASSOC);

// --- Correctly Fetch Payments and Calculate Totals ---
$amount_paid = 0;
$stmt_payments = $conn->prepare("SELECT SUM(credit_amount) as total_paid FROM payments WHERE invoice_id = ? AND invoice_type = 'package'");
$stmt_payments->bind_param("i", $invoice_id);
$stmt_payments->execute();
$payment_result = $stmt_payments->get_result()->fetch_assoc();
if ($payment_result && !empty($payment_result['total_paid'])) {
    $amount_paid = (float)$payment_result['total_paid'];
}
$stmt_payments->close();

$amount_due = (float)$invoice['grand_total_pkr'] - $amount_paid;

// --- Calculate Subtotals for Display ---
$pilgrim_total = array_sum(array_column($pilgrims, 'visa_price_sar'));
$hotel_total = array_sum(array_column($hotels, 'total_sar'));
$transport_total = array_sum(array_column($transports, 'total_amount'));
$services_total = array_sum(array_column($other_services, 'total_amount'));
$tickets_total = array_sum(array_column($airline_tickets, 'total_amount'));

$invoice_display_number = htmlspecialchars($invoice['invoice_number'] ?: $invoice['id']);

// Close DB connection
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice #<?= $invoice_display_number ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/html2pdf.js/0.10.1/html2pdf.bundle.min.js"></script>
    <style>
        :root {
            --theme-color: #31a7e2;
            --border-color: #000000;
        }
        body {
            font-family: Arial, sans-serif;
            background-color: #e9e9e9;
            margin: 0;
            padding: 10px;
            font-size: 9pt;
            color: #000;
        }
        .actions-bar {
            max-width: 900px;
            margin: 0 auto 15px auto;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }
        .btn {
            padding: 8px 15px; border: none; border-radius: 4px; color: white;
            font-size: 14px; cursor: pointer;
        }
        .btn-print { background-color: #2980b9; }
        .invoice-wrapper {
            max-width: 900px; margin: 0 auto; padding: 20px; border: 1px solid #ccc;
            position: relative; overflow: hidden; background-color: white;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        .watermark {
            position: absolute; top: 50%; left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 100px; color: rgba(0, 0, 0, 0.07); font-weight: bold;
            z-index: 1; pointer-events: none; text-transform: uppercase;
            white-space: nowrap;
        }
        .invoice-content { position: relative; z-index: 2; }
        table { width: 100%; border-collapse: collapse; }
        td, th { padding: 4px; vertical-align: middle; }

        /* --- Header --- */
        .header-table { margin-bottom: 20px; }
        .header-table td { border: none; padding: 0; vertical-align: top; }
        .logo-container {
            width: 50%;
            text-align: left;
        }
        .logo-container img {
            max-height: 80px;
            max-width: 200px;
        }
        .meta-container { width: 50%; }
        .meta-table td {
            background-color: var(--theme-color); border: 1px solid var(--border-color);
            padding: 5px 8px; font-weight: bold; color: white;
        }
        .meta-table td:first-child { width: 100px; }
        .meta-table td:last-child { background-color: #fff; text-align: center; color: #000; }
        
        /* --- Sections --- */
        .section-title {
            background-color: black; color: white; font-weight: bold;
            text-align: center; border: 1px solid var(--border-color);
            padding: 5px; margin-top: 15px; font-size: 10pt;
        }
        .detail-table { border: 1px solid var(--border-color); margin-bottom: 10px; }
        .detail-table th {
            background-color: var(--theme-color); border: 1px solid var(--border-color);
            font-weight: bold; padding: 5px; color: white; white-space: nowrap;
        }
        .detail-table td {
            border: 1px solid var(--border-color); padding: 4px; text-align: center;
        }
        .detail-table .text-left { text-align: left; padding-left: 5px; }
        .detail-table .sub-header th { background-color: #f2f2f2; color: #000; }
        .total-row td { border: 1px solid var(--border-color); font-weight: bold; }
        .total-row .total-label { text-align: right; padding-right: 15px; }
        .total-row .total-value { background-color: var(--theme-color); color: white; text-align: center; font-size: 10pt; }
        
        /* --- Footer --- */
        .footer-container { padding-top: 20px; overflow: hidden; clear: both; }
        .footer-notes { float: left; width: 53%; font-size: 8pt; line-height: 1.5; }
        .footer-notes h4 { margin-top: 0; margin-bottom: 5px; font-size: 9pt; }
        .summary-totals { float: right; width: 45%; }
        .summary-totals table td {
            border: 1px solid var(--border-color); padding: 6px 10px; font-size: 10pt;
        }
        .summary-totals table td:first-child { font-weight: bold; width: 65%; }
        .summary-totals table td:last-child { text-align: right; font-weight: bold; }
        .summary-totals .grand-total td { background-color: var(--theme-color); color: white; font-weight: bold; }
        .summary-totals .payment-received-row td { color: #27ae60; }
        .summary-totals .remaining-amount-row td {
            background-color: #c0392b; color: white; font-weight: bold; font-size: 11pt;
        }
        .final-warning {
            clear: both; text-align: center; font-weight: bold; font-size: 9pt;
            margin-top: 20px; padding-top: 10px; border-top: 1px dashed var(--border-color);
        }

        /* Responsive container for tables */
        .table-responsive-container {
            overflow-x: auto;
            margin-bottom: 10px;
        }
        .table-responsive-container > .detail-table {
             min-width: 800px; /* Force scroll on small screens */
        }

        /* --- MOBILE RESPONSIVENESS --- */
        @media screen and (max-width: 768px) {
            body { font-size: 8pt; padding: 5px; }
            .invoice-wrapper { padding: 10px; }
            .watermark { font-size: 60px; }
            .header-table > tbody > tr > td {
                display: block;
                width: 100%;
                text-align: center;
            }
            .logo-container { margin-bottom: 15px; }
            .meta-container { margin: 0 auto; max-width: 350px; }
            
            .footer-notes, .summary-totals {
                float: none;
                width: 100%;
                margin-bottom: 20px;
            }
        }

        /* --- FINAL PRINT FIX --- */
        @media print {
            * {
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
            body, html { margin: 0 !important; padding: 0 !important; background: #fff !important; }
            .actions-bar, .no-print { display: none !important; }
            body * { visibility: hidden; }
            .invoice-wrapper, .invoice-wrapper * { visibility: visible; }
            .invoice-wrapper {
                position: absolute; left: 0; top: 0;
                margin: 0 !important; padding: 10px !important;
                width: 100% !important; max-width: 100% !important;
                border: none !important; box-shadow: none !important;
                box-sizing: border-box;
            }
            .table-responsive-container { overflow-x: visible; }
        }
    </style>
</head>

<body>
    <div class="actions-bar no-print">
        <button onclick="window.print()" class="btn btn-print">Print</button>
    </div>

    <div class="invoice-wrapper" id="invoice-to-print">
        <?php if (!empty($invoice['status'])): ?>
            <div class="watermark"><?= strtoupper(htmlspecialchars($invoice['status'])) ?></div>
        <?php endif; ?>

        <div class="invoice-content">
            <header>
                <table class="header-table">
                    <tr>
                        <td class="logo-container">
                             <img src="<?= $logo_to_display ?>" alt="Logo">
                        </td>
                        <td class="meta-container">
                            <table class="meta-table">
                                <tr>
                                    <td>Invoice No:</td>
                                    <td><?= $invoice_display_number ?></td>
                                </tr>
                                <tr>
                                    <td>Guest Name:</td>
                                    <td><?= htmlspecialchars($invoice['guest_name']) ?></td>
                                </tr>
                                <tr>
                                    <td>Dated:</td>
                                    <td><?= date('d, M, Y', strtotime($invoice['issue_date'])) ?></td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                </table>
            </header>

            <main>
                <?php if (!empty($pilgrims)): ?>
                    <div class="section-title">Pilgrims Detail</div>
                    <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th>PASSPORT NO</th>
                                    <th>PASSENGER DETAILS</th>
                                    <th>DOB</th>
                                    <th>DOX</th>
                                    <th>PAX</th>
                                    <th>BED</th>
                                    <th>Visa (SAR)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($pilgrims as $p): ?>
                                <tr>
                                    <td><?= htmlspecialchars($p['passport_no']) ?></td>
                                    <td class="text-left"><?= htmlspecialchars($p['passenger_details']) ?></td>
                                    <td><?= $p['dob'] ? date('d, M, Y', strtotime($p['dob'])) : '' ?></td>
                                    <td><?= $p['dox'] ? date('d, M, Y', strtotime($p['dox'])) : '' ?></td>
                                    <td><?= htmlspecialchars($p['pax']) ?></td>
                                    <td><?= htmlspecialchars($p['bed']) ?></td>
                                    <td><?= nf($p['visa_price_sar']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="total-row">
                                    <td colspan="6" class="total-label">Total Visa Charges:</td>
                                    <td class="total-value"><?= nf($pilgrim_total) ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <?php if (!empty($hotels)): ?>
                    <div class="section-title">Accommodation</div>
                     <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th rowspan="2">City</th>
                                    <th rowspan="2">Hotel Name</th>
                                    <th rowspan="2">Helpline</th>
                                    <th rowspan="2">Check-in</th>
                                    <th rowspan="2">Nights</th>
                                    <th rowspan="2">Check-out</th>
                                    <th colspan="3">Room</th>
                                    <th rowspan="2">Net Amount (SAR)</th>
                                </tr>
                                <tr class="sub-header">
                                    <th>Type</th>
                                    <th>QTY</th>
                                    <th>Rate</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($hotels as $h): ?>
                                <tr>
                                    <td><?= htmlspecialchars($h['city']) ?></td>
                                    <td class="text-left"><?= htmlspecialchars($h['hotel_name']) ?></td>
                                    <td><?= htmlspecialchars($h['helpline'] ?? '') ?></td>
                                    <td><?= date('d, M, Y', strtotime($h['check_in'])) ?></td>
                                    <td><?= htmlspecialchars($h['nights']) ?></td>
                                    <td><?= date('d, M, Y', strtotime($h['check_out'])) ?></td>
                                    <td><?= htmlspecialchars($h['room_type']) ?></td>
                                    <td><?= htmlspecialchars($h['rooms']) ?></td>
                                    <td><?= nf($h['rate_sar']) ?></td>
                                    <td><?= nf($h['total_sar']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="total-row">
                                    <td colspan="9" class="total-label">Total Accommodation:</td>
                                    <td class="total-value"><?= nf($hotel_total) ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <?php if (!empty($transports)): ?>
                    <div class="section-title">Transportation</div>
                     <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th>Vehicle Type</th>
                                    <th>Route</th>
                                    <th>Rate (SAR)</th>
                                    <th>Qty</th>
                                    <th>Adults</th>
                                    <th>Child</th>
                                    <th>Net Amount (SAR)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transports as $t): ?>
                                <tr>
                                    <td class="text-left"><?= htmlspecialchars($t['vehicle_type']) ?></td>
                                    <td class="text-left"><?= htmlspecialchars($t['route']) ?></td>
                                    <td><?= nf($t['rate']) ?></td>
                                    <td><?= htmlspecialchars($t['qty']) ?></td>
                                    <td><?= htmlspecialchars($t['adult_qty']) ?></td>
                                    <td><?= htmlspecialchars($t['child_qty']) ?></td>
                                    <td><?= nf($t['total_amount']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="total-row">
                                    <td colspan="6" class="total-label">Total Transportation:</td>
                                    <td class="total-value"><?= nf($transport_total) ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <?php if (!empty($other_services)): ?>
                    <div class="section-title">Other Services</div>
                     <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th rowspan="2">Service Name</th>
                                    <th colspan="2">Adult</th>
                                    <th colspan="2">Child</th>
                                    <th colspan="2">Infant</th>
                                    <th rowspan="2">Net Amount (SAR)</th>
                                </tr>
                                <tr class="sub-header">
                                    <th>Rate</th><th>Qty</th>
                                    <th>Rate</th><th>Qty</th>
                                    <th>Rate</th><th>Qty</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($other_services as $s): ?>
                                <tr>
                                    <td class="text-left"><?= htmlspecialchars($s['service_name']) ?></td>
                                    <td><?= nf($s['adult_rate']) ?></td>
                                    <td><?= htmlspecialchars($s['adult_qty']) ?></td>
                                    <td><?= nf($s['child_rate']) ?></td>
                                    <td><?= htmlspecialchars($s['child_qty']) ?></td>
                                    <td><?= nf($s['infant_rate']) ?></td>
                                    <td><?= htmlspecialchars($s['infant_qty']) ?></td>
                                    <td><?= nf($s['total_amount']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="total-row">
                                    <td colspan="7" class="total-label">Total Services:</td>
                                    <td class="total-value"><?= nf($services_total) ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <?php if (!empty($airline_tickets)): ?>
                    <div class="section-title">Airline Tickets</div>
                     <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th rowspan="2">Airline</th>
                                    <th rowspan="2">Sector</th>
                                    <th colspan="2">Adult</th>
                                    <th colspan="2">Child</th>
                                    <th colspan="2">Infant</th>
                                    <th rowspan="2">Net Amount (PKR)</th>
                                </tr>
                                <tr class="sub-header">
                                    <th>Qty</th><th>Rate</th>
                                    <th>Qty</th><th>Rate</th>
                                    <th>Qty</th><th>Rate</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($airline_tickets as $t): ?>
                                <tr>
                                    <td class="text-left"><?= htmlspecialchars($t['airline']) ?></td>
                                    <td><?= htmlspecialchars($t['sector']) ?></td>
                                    <td><?= htmlspecialchars($t['adult_qty']) ?></td>
                                    <td><?= nf($t['adult_rate']) ?></td>
                                    <td><?= htmlspecialchars($t['child_qty']) ?></td>
                                    <td><?= nf($t['child_rate']) ?></td>
                                    <td><?= htmlspecialchars($t['infant_qty']) ?></td>
                                    <td><?= nf($t['infant_rate']) ?></td>
                                    <td><?= nf($t['total_amount']) ?></td>
                                </tr>
                                <?php endforeach; ?>
                                <tr class="total-row">
                                    <td colspan="8" class="total-label">Total Airline Charges:</td>
                                    <td class="total-value"><?= nf($tickets_total) ?></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </main>

            <footer class="footer-container">
                <div class="footer-notes">
                    <?php if (!empty($invoice['notes'])): ?>
                        <h4>Terms & Conditions:</h4>
                        <p><?= nl2br(htmlspecialchars($invoice['notes'])) ?></p>
                    <?php endif; ?>
                </div>
                <div class="summary-totals">
                    <table>
                        <tr>
                            <td>Total Amount (SAR)</td>
                            <td><?= nf($invoice['total_sar']) ?></td>
                        </tr>
                        <tr>
                            <td>Exchange Rate:</td>
                            <td><?= nf_decimal($invoice['exchange_rate']) ?></td>
                        </tr>
                        <tr>
                            <td>Total Amount (PKR)</td>
                            <td><?= nf($invoice['total_pkr_without_ticket']) ?></td>
                        </tr>
                        <?php if ((float)$invoice['total_ticket_pkr'] > 0): ?>
                        <tr>
                            <td>Ticket Amount (PKR)</td>
                            <td><?= nf($invoice['total_ticket_pkr']) ?></td>
                        </tr>
                        <?php endif; ?>
                        <?php if ((float)$invoice['discount_pkr'] > 0): ?>
                            <tr>
                                <td>Discount (PKR)</td>
                                <td>- <?= nf($invoice['discount_pkr']) ?></td>
                            </tr>
                        <?php endif; ?>
                        <tr class="grand-total">
                            <td>Grand Total</td>
                            <td><?= nf($invoice['grand_total_pkr']) ?></td>
                        </tr>
                        <?php if ($amount_paid > 0): ?>
                            <tr class="payment-received-row">
                                <td>Payment Received</td>
                                <td>- <?= nf($amount_paid) ?></td>
                            </tr>
                            <tr class="remaining-amount-row">
                                <td>Remaining Amount</td>
                                <td><?= nf($amount_due) ?></td>
                            </tr>
                        <?php endif; ?>
                    </table>
                </div>
            </footer>

            <div class="final-warning">
                SUBJECT TO AVAILABILITY AND EX RATES APPLIED AS PER DATE OF FULL PAYMENT
            </div>
        </div>
    </div>
</body>
</html>