<?php
session_start(); // Ensure session is started at the very top
include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php");
    exit;
}

// Get the logged-in agent's ID and name from the session
$logged_in_agent_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name']; // For the sidebar

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- PAGINATION SETUP ---
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

// --- GET TOTAL COUNT OF INQUIRIES FOR THIS AGENT ---
$count_sql = "SELECT COUNT(*) as total FROM umrah_inquiries WHERE user_id = ?";
$stmt_count = $conn->prepare($count_sql);
$stmt_count->bind_param("i", $logged_in_agent_id);
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ($total_items > 0) ? ceil($total_items / $items_per_page) : 1;
$stmt_count->close();

// --- FETCH THE PAGINATED INQUIRIES FOR THIS AGENT ---
$data_sql = "SELECT * FROM umrah_inquiries WHERE user_id = ? ORDER BY created_at DESC LIMIT ?, ?";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param("iii", $logged_in_agent_id, $offset, $items_per_page);
$stmt_data->execute();
$inquiries_result = $stmt_data->get_result();
$stmt_data->close();

?>
<!DOCTYPE html>
<html>

<head>
    <title>My Umrah Inquiries</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <!-- Styles adapted from your admin inquiry page for the accordion list -->
</head>

<style>
    /* =======================================================
   MANAGE INQUIRIES STYLES (v3 - Polished, Final Version)
   ======================================================= */

    /* --- Filter Bar at the top --- */
    .filter-bar {
        display: flex;
        flex-wrap: wrap;
        gap: 8px;
        background-color: var(--card-bg);
        padding: 10px;
        border-radius: 8px;
        margin-bottom: 25px;
        box-shadow: var(--shadow);
    }

    .filter-bar a {
        text-decoration: none;
        padding: 8px 15px;
        border-radius: 6px;
        color: var(--text-dark);
        font-weight: 600;
        font-size: 0.9rem;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 8px;
    }

    .filter-bar a:hover {
        background-color: #eef1f4;
    }

    .filter-bar a.active {
        background-color: var(--primary-color);
        color: #fff;
    }

    .filter-bar .filter-count {
        color: #adb5bd;
    }

    .filter-bar a.active .filter-count {
        color: #fff;
        opacity: 0.7;
    }

    .count-badge {
        background-color: var(--secondary-color);
        color: var(--sidebar-bg);
        font-size: 0.8rem;
        padding: 2px 8px;
        border-radius: 10px;
    }

    /* --- Inquiry List & Items --- */
    .inquiry-list {
        border: 1px solid var(--border-color);
        border-radius: 8px;
        overflow: hidden;
    }

    .inquiry-item {
        padding: 15px 20px;
        border-bottom: 1px solid var(--border-color);
        background-color: #fff;
        transition: background-color 0.2s;
    }

    .inquiry-list .inquiry-item:last-child {
        border-bottom: none;
    }

    .inquiry-summary {
        display: flex;
        align-items: center;
        gap: 15px;
        cursor: pointer;
    }

    .inquiry-icon {
        font-size: 1.8rem;
        color: var(--primary-color);
    }

    .inquiry-primary-info {
        flex-grow: 1;
    }

    .inquiry-primary-info strong {
        font-size: 1.1rem;
        color: var(--text-dark);
    }

    .inquiry-primary-info span {
        font-size: 0.85rem;
        color: #6c757d;
    }

    .inquiry-meta {
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .expand-arrow {
        font-size: 0.9rem;
        color: #6c757d;
        transition: transform 0.3s ease;
    }

    .inquiry-item.expanded .expand-arrow {
        transform: rotate(180deg);
    }

    /* --- Status Badges --- */
    .status-badge {
        padding: 5px 12px;
        border-radius: 20px;
        font-size: 0.75rem;
        font-weight: 700;
        text-transform: uppercase;
        color: #000;
        min-width: 90px;
        text-align: center;
    }

    .status-new {
        background-color: #0d6efd;
    }

    .status-in-progress {
        background-color: #ffc107;
        color: #000;
    }

    .status-completed {
        background-color: #198754;
    }

    .status-rejected {
        background-color: #dc3545;
    }

    /* --- Details Panel --- */
    .inquiry-details {
        max-height: 0;
        overflow: hidden;
        transition: all 0.4s ease-in-out;
    }

    .inquiry-item.expanded .inquiry-details {
        max-height: 800px;
        padding-top: 20px;
        margin-top: 20px;
        border-top: 1px solid #e9ecef;
    }

    .details-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 15px 30px;
        /* row-gap column-gap */
        font-size: 0.95rem;
        margin-bottom: 15px;
    }

    .detail-item {
        color: #555;
    }

    .detail-item strong {
        color: var(--text-dark);
        display: block;
        margin-bottom: 3px;
    }

    .inquiry-message {
        margin-top: 15px;
    }

    .inquiry-message strong {
        color: var(--text-dark);
        display: block;
        margin-bottom: 5px;
    }

    .inquiry-message p {
        background-color: #f8f9fa;
        padding: 15px;
        border-radius: 5px;
        white-space: pre-wrap;
        line-height: 1.6;
    }

    /* --- Action Area within details --- */
    .inquiry-actions {
        margin-top: 20px;
        border-top: 1px solid #e9ecef;
        padding-top: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .inquiry-actions label {
        font-weight: 600;
        font-size: 0.9rem;
    }

    .status-change-select {
        padding: 8px 12px;
        border: 1px solid var(--border-color);
        border-radius: 6px;
        cursor: pointer;
    }

    /* --- Highlight animation on successful update --- */
    .inquiry-item.highlight-success {
        animation: highlight 1.5s ease-out;
    }

    @keyframes highlight {
        0% {
            background-color: #d1e7dd;
        }

        100% {
            background-color: #fff;
        }
    }

    /* --- Empty state when no inquiries are found --- */
    .empty-state {
        text-align: center;
        padding: 50px 20px;
        color: #6c757d;
    }

    .empty-state i {
        font-size: 3.5rem;
        opacity: 0.4;
        margin-bottom: 15px;
    }
</style>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">

            <?php include 'agent-sidebar.php'; ?>

            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-kaaba"></i> My Umrah Inquiries</h2>
                    <p class="content-description">Here is a list of all Umrah package inquiries you have submitted. Click on an item to see more details.</p>

                    <div class="inquiry-list" id="inquiry-list">
                        <?php if ($inquiries_result && $inquiries_result->num_rows > 0): ?>
                            <?php while ($inquiry = $inquiries_result->fetch_assoc()): ?>
                                <div class="inquiry-item" data-id="<?php echo $inquiry['id']; ?>">
                                    <div class="inquiry-summary">
                                        <div class="inquiry-icon"><i class="fas fa-kaaba"></i></div>
                                        <div class="inquiry-primary-info">
                                            <strong><?php echo e($inquiry['customer_name']); ?></strong>
                                            <span>Inquiry ID: #<?php echo $inquiry['id']; ?> | Submitted: <?php echo date('d M Y, g:ia', strtotime($inquiry['created_at'])); ?></span>
                                        </div>
                                        <div class="inquiry-meta">
                                            <div class="status-badge-wrapper"><span class="status-badge status-<?php echo strtolower(str_replace(' ', '-', $inquiry['status'])); ?>"><?php echo e($inquiry['status']); ?></span></div>
                                            <div class="expand-arrow"><i class="fas fa-chevron-down"></i></div>
                                        </div>
                                    </div>
                                    <div class="inquiry-details">
                                        <div class="details-grid">
                                            <div class="detail-item"><strong>Email:</strong> <span><a href="mailto:<?= e($inquiry['customer_email']) ?>"><?= e($inquiry['customer_email']) ?></a></span></div>
                                            <div class="detail-item"><strong>Phone:</strong> <span><a href="tel:<?= e($inquiry['customer_phone']) ?>"><?= e($inquiry['customer_phone']) ?></a></span></div>
                                            <div class="detail-item"><strong>Package Name:</strong> <?php echo e($inquiry['package_name']); ?></div>
                                            <div class="detail-item"><strong>Package ID:</strong> <?php echo e($inquiry['package_id']); ?></div>
                                            <div class="detail-item"><strong>Room Type:</strong> <?php echo e($inquiry['room_type']); ?></div>
                                            <div class="detail-item"><strong>Pax:</strong> <?php echo e($inquiry['pax']); ?></div>
                                        </div>
                                        <!-- NOTE: No "actions" section here, as agents can only view status -->
                                    </div>
                                </div>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <div class="empty-state" style="text-align: center; padding: 2rem;">
                                <i class="fas fa-folder-open" style="font-size: 3rem; color: #ccc;"></i>
                                <p style="margin-top: 1rem; color: #666;">You have not submitted any Umrah inquiries yet.</p>
                            </div>
                        <?php endif; ?>
                    </div>

                    <?php if ($total_pages > 1): ?>
                        <div class="pagination-controls">
                            <?php if ($current_page > 1): ?>
                                <a href="?page=<?= $current_page - 1 ?>" class="btn btn-secondary btn-sm">« Previous</a>
                            <?php else: ?>
                                <span class="btn btn-sm disabled">« Previous</span>
                            <?php endif; ?>

                            <span class="page-info">Page <?php echo $current_page; ?> of <?php echo $total_pages; ?></span>

                            <?php if ($current_page < $total_pages): ?>
                                <a href="?page=<?= $current_page + 1 ?>" class="btn btn-secondary btn-sm">Next »</a>
                            <?php else: ?>
                                <span class="btn btn-sm disabled">Next »</span>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>

                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const inquiryList = document.getElementById('inquiry-list');

            if (inquiryList) {
                // Event delegation for expanding/collapsing inquiry items
                inquiryList.addEventListener('click', function(event) {
                    const inquiryItem = event.target.closest('.inquiry-item');
                    if (!inquiryItem) return;

                    // Prevent toggling if a link is clicked inside the details
                    if (!event.target.closest('a')) {
                        inquiryItem.classList.toggle('expanded');
                    }
                });
            }
        });
    </script>

</body>

</html>