<?php

/**
 * agent-invoices.php (v3.5 - Complete Flight Invoice Modal)
 * - FETCHES: Now fetches both Umrah/Hajj invoices AND Flight Ticket invoices for the agent.
 * - DISPLAY: Renders two separate tables on the dashboard for each invoice type.
 * - MODAL: The modal is now dynamic. It checks the invoice type and builds the correct layout.
 * - FIXED: The flight ticket invoice modal now includes the complete Fare Breakdown details,
 *   making it a true copy of the standalone view.
 * - ADDED: CSS for the fare breakdown grid.
 */

session_start();
include 'db-config.php';

// --- SECURITY CHECKS ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: login.php");
    exit;
}

// --- Get the logged-in agent's full details, including their logo path ---
$user_id = $_SESSION['user_id'];
$agent_logo_path = null;
$stmt_user = $conn->prepare("SELECT name, email, logo_path FROM users WHERE id = ?");
$stmt_user->bind_param("i", $user_id);
$stmt_user->execute();
$current_user = $stmt_user->get_result()->fetch_assoc();
$stmt_user->close();

if ($current_user) {
    $user_name = $current_user['name'];
    $user_email = $current_user['email'];
    $agent_logo_path = $current_user['logo_path'];
} else {
    $user_name = 'Agent';
    $user_email = 'N/A';
}

// --- 1. FETCH ALL UMRAH/PACKAGE INVOICE DATA ---
$invoices_data = [];
$sql_invoices = "SELECT * FROM invoices WHERE user_id = ? ORDER BY issue_date DESC, id DESC";
$stmt_invoices = $conn->prepare($sql_invoices);
$stmt_invoices->bind_param("i", $user_id);
$stmt_invoices->execute();
$invoices_result = $stmt_invoices->get_result();

while ($invoice = $invoices_result->fetch_assoc()) {
    $current_invoice_id = $invoice['id'];

    // Fetch all related items
    $invoice['pilgrims'] = $conn->query("SELECT * FROM invoice_pilgrims WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['hotels'] = $conn->query("SELECT * FROM invoice_hotels WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['transports'] = $conn->query("SELECT * FROM invoice_transports WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['other_services'] = $conn->query("SELECT * FROM invoice_other_services WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $invoice['airline_tickets'] = $conn->query("SELECT * FROM invoice_airline_tickets WHERE invoice_id = $current_invoice_id")->fetch_all(MYSQLI_ASSOC);

    // Fetch payments and calculate totals
    $payments = $conn->query("SELECT payment_amount FROM payments WHERE invoice_id = $current_invoice_id AND invoice_type = 'package'")->fetch_all(MYSQLI_ASSOC);
    $amount_paid = array_sum(array_column($payments, 'payment_amount'));
    $invoice['amount_paid'] = $amount_paid;
    $invoice['amount_due'] = $invoice['grand_total_pkr'] - $amount_paid;

    $invoice['agent_logo_path'] = $agent_logo_path;
    $invoices_data[] = $invoice;
}
$stmt_invoices->close();


// --- 2. FETCH ALL FLIGHT TICKET INVOICE DATA ---
$ticket_invoices_data = [];
$sql_ticket_invoices = "SELECT * FROM ticket_invoices WHERE user_id = ? ORDER BY issue_date DESC, id DESC";
$stmt_ticket_invoices = $conn->prepare($sql_ticket_invoices);
$stmt_ticket_invoices->bind_param("i", $user_id);
$stmt_ticket_invoices->execute();
$ticket_invoices_result = $stmt_ticket_invoices->get_result();

while ($ticket_invoice = $ticket_invoices_result->fetch_assoc()) {
    $current_ticket_invoice_id = $ticket_invoice['id'];

    // Fetch related items for ticket invoices
    $ticket_invoice['passengers'] = $conn->query("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = $current_ticket_invoice_id")->fetch_all(MYSQLI_ASSOC);
    $ticket_invoice['flights'] = $conn->query("SELECT * FROM ticket_invoice_flights WHERE ticket_invoice_id = $current_ticket_invoice_id")->fetch_all(MYSQLI_ASSOC);

    // Fetch payments and calculate totals
    $payments = $conn->query("SELECT payment_amount FROM payments WHERE invoice_id = $current_ticket_invoice_id AND invoice_type = 'ticket'")->fetch_all(MYSQLI_ASSOC);
    $amount_paid = array_sum(array_column($payments, 'payment_amount'));
    $ticket_invoice['amount_paid'] = $amount_paid;
    $ticket_invoice['amount_due'] = $ticket_invoice['grand_total_pkr'] - $amount_paid;

    $ticket_invoice['agent_logo_path'] = $agent_logo_path;
    $ticket_invoices_data[] = $ticket_invoice;
}
$stmt_ticket_invoices->close();
?>
<!DOCTYPE html>
<html>

<head>
    <title>My Invoices</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        /* Styles for the invoice modal - designed to handle both invoice types */
        .modal-body {
            font-family: Arial, sans-serif;
            font-size: 9pt;
            color: #000;
        }

        .modal-body .invoice-wrapper {
            position: relative;
            max-width: 100%;
            box-shadow: none;
            border: none;
            padding: 20px;
        }

        .modal-body table {
            width: 100%;
            border-collapse: collapse;
        }

        .modal-body td,
        .modal-body th {
            padding: 4px;
            vertical-align: middle;
        }

        .modal-body .header-table td {
            border: none;
            padding: 0;
            vertical-align: top;
        }

        .modal-body .agent-logo img {
            max-height: 80px;
            max-width: 180px;
        }

        .modal-body .company-logo-container {
            text-align: center;
        }

        .modal-body .company-logo-container img {
            max-height: 50px;
        }

        .modal-body .company-details {
            font-size: 9pt;
            line-height: 1.4;
            padding-top: 5px;
        }

        .modal-body .meta-table td {
            background-color: #31a7e2;
            border: 1px solid #000;
            padding: 5px 8px;
            font-weight: bold;
        }

        .modal-body .meta-table td:first-child {
            width: 100px;
        }

        .modal-body .meta-table td:last-child {
            background-color: #fff;
            text-align: center;
        }

        .modal-body .section-title {
            background-color: black;
            color: white;
            font-weight: bold;
            text-align: center;
            border: 1px solid #000;
            padding: 5px;
            margin-top: 10px;
            font-size: 10pt;
        }

        .modal-body .detail-table {
            border: 1px solid #000;
            margin-bottom: 5px;
        }

        .modal-body .detail-table th {
            background-color: #31a7e2;
            border: 1px solid #000;
            font-weight: bold;
            padding: 5px;
        }

        .modal-body .detail-table td {
            border: 1px solid #000;
            padding: 4px;
            text-align: center;
        }

        .modal-body .detail-table .text-left {
            text-align: left;
            padding-left: 5px;
        }

        .modal-body .detail-table .sub-header th {
            border: 1px solid #000;
            font-weight: bold;
        }

        .modal-body .total-row td {
            border: 1px solid #000;
            font-weight: bold;
        }

        .modal-body .total-row .total-label {
            text-align: right;
            padding-right: 15px;
        }

        .modal-body .total-row .total-value {
            background-color: #31a7e2;
            text-align: center;
            font-size: 10pt;
        }

        .modal-body .footer-container {
            padding-top: 15px;
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: 20px;
        }

        .modal-body .footer-notes {
            flex: 1 1 55%;
            font-size: 8pt;
            line-height: 1.5;
        }

        .modal-body .footer-notes h4 {
            margin-top: 0;
            margin-bottom: 5px;
            font-size: 9pt;
        }

        .modal-body .summary-totals {
            flex: 0 0 43%;
        }

        .modal-body .summary-totals table td {
            border: 1px solid #000;
            padding: 6px 10px;
            font-size: 10pt;
        }

        .modal-body .summary-totals table td:first-child {
            font-weight: bold;
            width: 65%;
        }

        .modal-body .summary-totals table td:last-child {
            text-align: right;
            font-weight: normal;
        }

        .modal-body .summary-totals .grand-total td {
            background-color: #31a7e2;
            font-weight: bold;
        }

        .modal-body .summary-totals .discount-row td {
            font-weight: bold;
            color: #27ae60;
        }

        .modal-body .final-warning {
            clear: both;
            text-align: center;
            font-weight: bold;
            font-size: 9pt;
            margin-top: 20px;
            padding-top: 10px;
        }

        .modal-body .watermark {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 100px;
            font-weight: bold;
            z-index: 1;
            pointer-events: none;
            text-transform: uppercase;
            white-space: nowrap;
        }

        .modal-body .invoice-content {
            position: relative;
            z-index: 2;
        }

        .modal-body .summary-totals .payment-received-row td {
            font-weight: bold;
            color: #d35400;
        }

        .modal-body .summary-totals .remaining-amount-row td {
            background-color: #c0392b;
            color: white;
            font-weight: bold;
        }

        /* --- FIX: Added CSS for Ticket Invoice Fare Breakdown --- */
        .modal-body .fare-breakdown-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 5px;
            padding: 10px;
            border: 1px solid #000;
            background-color: rgba(0, 0, 0, 0.02);
        }

        .modal-body .fare-box {
            border: 1px solid #b0b0b0;
            background-color: #fdfdfd;
            border-radius: 4px;
            overflow: hidden;
        }

        .modal-body .fare-box-header {
            background-color: #31a7e2;
            padding: 8px;
            font-weight: bold;
            text-align: center;
            font-size: 11pt;
            color: #000;
        }

        .modal-body .fare-box-content {
            padding: 10px;
            display: flex;
            justify-content: space-around;
            gap: 10px;
        }

        .modal-body .fare-detail {
            text-align: center;
            flex: 1;
            padding: 8px 5px;
            background-color: #f0f0f0;
            border: 1px solid #ddd;
            border-radius: 3px;
        }

        .modal-body .fare-detail .label {
            display: block;
            font-size: 8pt;
            color: #555;
            margin-bottom: 4px;
            text-transform: uppercase;
        }

        .modal-body .fare-detail .value {
            display: block;
            font-weight: bold;
            font-size: 12pt;
            color: #000;
        }

        .payment-details-section {
            margin-top: 30px;
        }

        .payment-details-section h4 {
            font-size: 1.1em;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .bank-accounts-container {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }

        .bank-account-box {
            flex: 1;
            min-width: 300px;
            border: 1px solid #ddd;
            padding: 15px;
            border-radius: 5px;
            background-color: #fdfdfd;
            text-align: center;
        }

        .bank-logo {
            max-height: 100px;
            margin-bottom: 15px;
        }

        .bank-details-table {
            width: 100%;
            text-align: left;
        }

        .bank-details-table td {
            border: none;
            padding: 5px 0;
            font-size: 0.9em;
            vertical-align: top;
        }

        .bank-details-table td:first-child {
            font-weight: bold;
            width: 100px;
            color: #555;
        }

        .receipt-instruction {
            margin-top: 15px;
            text-align: center;
            font-size: 1em;
            font-weight: bold;
            background-color: #b2d9ecff;
            padding: 10px;
            border: 1px dashed #31a7e2;
            border-radius: 4px;
        }

        .content-card h2 {
            margin-bottom: 5px;
        }

        .content-card .content-description {
            margin-top: 0;
        }

        .content-card+.content-card {
            margin-top: 30px;
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">

                <!-- UMRAH / PACKAGE INVOICES TABLE -->
                <div class="content-card">
                    <h2><i class="fa-solid fa-kaaba"></i> My Umrah/Package Invoices</h2>
                    <p class="content-description">Review all package invoices you have created. Click on any row to view full details.</p>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Invoice #</th>
                                    <th>Customer Name</th>
                                    <th>Issue Date</th>
                                    <th>Status</th>
                                    <th>Total (PKR)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($invoices_data)): ?>
                                    <?php foreach ($invoices_data as $invoice): ?>
                                        <tr class="view-invoice-row" data-invoice-details='<?= htmlspecialchars(json_encode($invoice), ENT_QUOTES, 'UTF-8') ?>' data-type="umrah" style="cursor: pointer;">
                                            <td><strong><?= htmlspecialchars($invoice['invoice_number'] ?: $invoice['id']) ?></strong></td>
                                            <td><?= htmlspecialchars($invoice['guest_name']) ?></td>
                                            <td><?= date('M j, Y', strtotime($invoice['issue_date'])) ?></td>
                                            <td><span class="status-badge status-<?= strtolower(str_replace(' ', '-', $invoice['status'])) ?>"><?= htmlspecialchars($invoice['status']) ?></span></td>
                                            <td><?= number_format($invoice['grand_total_pkr'], 2) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="no-bookings-found">You have not created any package invoices.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- FLIGHT TICKET INVOICES TABLE -->
                <div class="content-card">
                    <h2><i class="fa-solid fa-plane-departure"></i> My Flight Ticket Invoices</h2>
                    <p class="content-description">Review all flight ticket invoices you have created. Click on any row to view full details.</p>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Invoice #</th>
                                    <th>Customer Name</th>
                                    <th>Issue Date</th>
                                    <th>Status</th>
                                    <th>Total (PKR)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($ticket_invoices_data)): ?>
                                    <?php foreach ($ticket_invoices_data as $ticket_invoice): ?>
                                        <tr class="view-invoice-row" data-invoice-details='<?= htmlspecialchars(json_encode($ticket_invoice), ENT_QUOTES, 'UTF-8') ?>' data-type="ticket" style="cursor: pointer;">
                                            <td><strong><?= htmlspecialchars($ticket_invoice['invoice_number'] ?: $ticket_invoice['id']) ?></strong></td>
                                            <td><?= htmlspecialchars($ticket_invoice['guest_name']) ?></td>
                                            <td><?= date('M j, Y', strtotime($ticket_invoice['issue_date'])) ?></td>
                                            <td>
                                                <span class="status-badge status-<?= strtolower(str_replace(' ', '-', $ticket_invoice['status'])) ?>">
                                                    <?= htmlspecialchars($ticket_invoice['status']) ?>
                                                </span>
                                            </td>
                                            <td><?= number_format($ticket_invoice['grand_total_pkr'], 2) ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="no-bookings-found">You have not created any flight ticket invoices.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>


            </div>
    </main>

    <?php include 'footer.php'; ?>

    <!-- MODAL HTML STRUCTURE -->
    <div class="modal-overlay" id="invoice-modal-overlay">
        <div class="modal-content" id="invoice-modal-content">
            <button class="modal-close-btn" id="modal-close-btn"><i class="fa-solid fa-times"></i></button>
            <div class="modal-body" id="modal-body"><!-- Invoice content will be built here by JavaScript --></div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const modalOverlay = document.getElementById('invoice-modal-overlay');
            const modalBody = document.getElementById('modal-body');
            const closeModalBtn = document.getElementById('modal-close-btn');
            const invoiceRows = document.querySelectorAll('.view-invoice-row');

            const nf = (num) => (parseFloat(num) || 0).toLocaleString('en-US', {
                minimumFractionDigits: 0,
                maximumFractionDigits: 0
            });
            const nf_decimal = (num) => {
                const formatted = (parseFloat(num) || 0).toLocaleString('en-US', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                });
                return formatted.endsWith('.00') ? formatted.slice(0, -3) : formatted;
            };
            const formatDate = (dateStr) => {
                if (!dateStr || dateStr === '0000-00-00') return 'N/A';
                const months = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
                const d = new Date(dateStr);
                return `${('0' + d.getDate()).slice(-2)}-${months[d.getMonth()]}-${d.getFullYear()}`;
            };
            const formatDateTime = (dateStr) => {
                if (!dateStr || dateStr === '0000-00-00 00:00:00') return 'N/A';
                const d = new Date(dateStr);
                const datePart = `${('0' + d.getDate()).slice(-2)}-${['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'][d.getMonth()]}-${d.getFullYear().toString().slice(-2)}`;
                const timePart = `${('0' + d.getHours()).slice(-2)}:${('0' + d.getMinutes()).slice(-2)}`;
                return `${datePart} ${timePart}`;
            }

            function openModal() {
                modalOverlay.classList.add('is-visible');
                document.body.style.overflow = 'hidden';
            }

            function closeModal() {
                modalOverlay.classList.remove('is-visible');
                document.body.style.overflow = '';
                modalBody.innerHTML = '';
            }

            invoiceRows.forEach(row => {
                row.addEventListener('click', function(event) {
                    if (event.target.closest('.btn-edit')) return;
                    try {
                        const invoiceData = JSON.parse(this.dataset.invoiceDetails);
                        const invoiceType = this.dataset.type;

                        if (invoiceType === 'ticket') {
                            populateTicketModal(invoiceData);
                        } else {
                            populateUmrahModal(invoiceData);
                        }
                        openModal();
                    } catch (e) {
                        console.error("Failed to parse invoice data:", e);
                        alert("An error occurred while loading invoice details.");
                    }
                });
            });

            if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
            if (modalOverlay) modalOverlay.addEventListener('click', function(event) {
                if (event.target === modalOverlay) closeModal();
            });

            function getAgentLogoHtml(invoice) {
                return invoice.agent_logo_path ? `<td style="width: 33%; text-align: left;" class="agent-logo"><img src="uploads/logos/${invoice.agent_logo_path}" alt="Agent Logo"></td>` : `<td style="width: 33%;"></td>`;
            }

            function getPaymentSummaryHtml(invoice) {
                if (invoice.amount_paid > 0) {
                    return `
                <tr class="payment-received-row">
                    <td>Payment Received</td>
                    <td>- ${nf_decimal(invoice.amount_paid)}</td>
                </tr>
                <tr class="remaining-amount-row">
                    <td>Remaining Amount</td>
                    <td>${nf_decimal(invoice.amount_due)}</td>
                </tr>
            `;
                }
                return '';
            }

            function getWatermarkHtml(invoice) {
                if (!invoice.status) return '';
                let color = 'rgba(231, 76, 60, 0.1)'; // Red for Tentative/Cancelled
                if (invoice.status === 'Approved') {
                    color = 'rgba(46, 204, 113, 0.1)'; // Green for Approved
                }
                return `<div class="watermark" style="color: ${color};">${invoice.status.toUpperCase()}</div>`;
            }

            function populateTicketModal(invoice) {
                const invoice_display_number = invoice.invoice_number || invoice.id;

                let passengersHtml = '';
                if (invoice.passengers && invoice.passengers.length > 0) {
                    passengersHtml = `<div class="section-title">Passenger Information</div><table class="detail-table"><thead><tr><th>Sr.</th><th>Full Name</th><th>Type</th><th>Passport No.</th><th>Passport Issue</th><th>Passport Expiry</th><th>Date of Birth</th><th>PNR</th><th>Ticket No.</th></tr></thead><tbody>
            ${invoice.passengers.map((p, i) => `<tr><td>${i+1}</td><td class="text-left">${p.full_name}</td><td>${p.passenger_type}</td><td>${p.passport_no}</td><td>${p.passport_issue_date}</td><td>${p.passport_expiry_date}</td><td>${formatDate(p.dob)}</td><td>${p.pnr}</td><td>${p.ticket_number}</td></tr>`).join('')}
            </tbody></table>`;
                }

                let flightsHtml = '';
                if (invoice.flights && invoice.flights.length > 0) {
                    flightsHtml = `<div class="section-title">Flight Itinerary</div><table class="detail-table"><thead><tr><th>Airline</th><th>Flight No.</th><th>Sector</th><th>Departure</th><th>Arrival</th></tr></thead><tbody>
            ${invoice.flights.map(f => `<tr><td class="text-left">${f.airline}</td><td>${f.flight_no}</td><td>${f.sector}</td><td>${formatDateTime(f.departure_datetime)}</td><td>${formatDateTime(f.arrival_datetime)}</td></tr>`).join('')}
            </tbody></table>`;
                }

                // --- FIX: Build the Fare Breakdown HTML ---
                let fareBreakdownHtml = '';
                if (invoice.adult_qty > 0 || invoice.child_qty > 0 || invoice.infant_qty > 0) {
                    fareBreakdownHtml += '<div class="section-title">Fare Breakdown</div><div class="fare-breakdown-grid">';
                    if (invoice.adult_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Adults</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.adult_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.adult_rate)}</span></div></div></div>`;
                    }
                    if (invoice.child_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Children</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.child_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.child_rate)}</span></div></div></div>`;
                    }
                    if (invoice.infant_qty > 0) {
                        fareBreakdownHtml += `<div class="fare-box"><div class="fare-box-header">Infants</div><div class="fare-box-content"><div class="fare-detail"><span class="label">Quantity</span><span class="value">${nf(invoice.infant_qty)}</span></div><div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value">${nf_decimal(invoice.infant_rate)}</span></div></div></div>`;
                    }
                    fareBreakdownHtml += '</div>';
                }

                const summaryHtml = `
            <table>
                <tr><td>Total Fare (PKR)</td><td>${nf_decimal(invoice.total_fare_pkr)}</td></tr>
                <tr><td>Service Fee (PKR)</td><td>${nf_decimal(invoice.service_fee_pkr)}</td></tr>
                ${invoice.discount_pkr > 0 ? `<tr class="discount-row"><td>Discount (PKR)</td><td>- ${nf_decimal(invoice.discount_pkr)}</td></tr>` : ''}
                <tr class="grand-total"><td>Grand Total (PKR)</td><td>${nf_decimal(invoice.grand_total_pkr)}</td></tr>
                ${getPaymentSummaryHtml(invoice)}
            </table>`;

                modalBody.innerHTML = `
        <div class="invoice-wrapper">
            ${getWatermarkHtml(invoice)}
            <div class="invoice-content">
                <header>
                    <table class="header-table">
                        <tr>
                            ${getAgentLogoHtml(invoice)}
                            <td style="width: 34%;" class="company-logo-container">
                                <img src="images/logo.png" alt="Logo">
                                <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>
                                Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                            </td>
                            <td style="width: 33%;">
                                <table class="meta-table">
                                    <tr><td>Invoice No:</td><td>${invoice_display_number}</td></tr>
                                    <tr><td>Guest Name:</td><td>${invoice.guest_name}</td></tr>
                                    <tr><td>Dated:</td><td>${formatDate(invoice.issue_date)}</td></tr>
                                </table>
                            </td>
                        </tr>
                    </table>
                </header>
                <main>${passengersHtml}${flightsHtml}${fareBreakdownHtml}</main>
                <footer class="footer-container">
                    <div class="footer-notes">${invoice.notes ? `<h4>Terms & Conditions:</h4><p>${invoice.notes.replace(/\n/g, '<br>')}</p>` : ''}</div>
                    <div class="summary-totals">${summaryHtml}</div>
                </footer>
                <div class="final-warning">All fares are subject to change without prior notice. Please verify all passenger names match their passports.</div>
            </div>
        </div>`;
            }

            function populateUmrahModal(invoice) {
                const invoice_display_number = invoice.invoice_number || invoice.id;

                const pilgrim_total = invoice.pilgrims.reduce((sum, p) => sum + parseFloat(p.visa_price_sar), 0);
                const hotel_total = invoice.hotels.reduce((sum, h) => sum + parseFloat(h.total_sar), 0);
                const transport_total = invoice.transports.reduce((sum, t) => sum + parseFloat(t.total_amount), 0);
                const services_total = invoice.other_services.reduce((sum, s) => sum + parseFloat(s.total_amount), 0);
                const tickets_total = invoice.airline_tickets.reduce((sum, t) => sum + parseFloat(t.total_amount), 0);

                let pilgrimsHtml = '',
                    hotelsHtml = '',
                    transportsHtml = '',
                    servicesHtml = '',
                    ticketsHtml = '';

                if (invoice.pilgrims && invoice.pilgrims.length > 0) {
                    pilgrimsHtml = `<div class="section-title">Pilgrims Detail</div><table class="detail-table"><thead><tr><th>PASSPORT NO</th><th>PASSANGER DETAILS</th><th>DOB</th><th>DOX</th><th>PAX</th><th>BED</th><th>Visa</th></tr></thead><tbody>
            ${invoice.pilgrims.map(p => `<tr><td>${p.passport_no}</td><td class="text-left">${p.passenger_details}</td><td>${formatDate(p.dob)}</td><td>${formatDate(p.dox)}</td><td>${p.pax}</td><td>${p.bed}</td><td>${nf(p.visa_price_sar)}</td></tr>`).join('')}
            <tr class="total-row"><td colspan="6" class="total-label">Total Visa Charges:</td><td class="total-value">${nf(pilgrim_total)}</td></tr></tbody></table>`;
                }
                if (invoice.hotels && invoice.hotels.length > 0) {
                    hotelsHtml = `<div class="section-title">Accommodation</div><table class="detail-table"><thead><tr><th rowspan="2">City</th><th rowspan="2">Hotel Name</th><th rowspan="2">Checkin</th><th rowspan="2">Nights</th><th rowspan="2">Checkout</th><th colspan="4">Room</th><th rowspan="2">Net Amount</th></tr><tr class="sub-header"><th>Type</th><th>Meal Plan</th><th>QTY</th><th>Rate</th></tr></thead><tbody>
            ${invoice.hotels.map(h => `<tr><td>${h.city}</td><td class="text-left">${h.hotel_name}</td><td>${formatDate(h.check_in)}</td><td>${h.nights}</td><td>${formatDate(h.check_out)}</td><td>${h.room_type}</td><td>${h.meal_plan}</td><td>${h.rooms}</td><td>${nf(h.rate_sar)}</td><td>${nf(h.total_sar)}</td></tr>`).join('')}
            <tr class="total-row"><td colspan="9" class="total-label">Total Accommodation:</td><td class="total-value">${nf(hotel_total)}</td></tr></tbody></table>`;
                }
                if (invoice.transports && invoice.transports.length > 0) {
                    transportsHtml = `<div class="section-title">Transportation</div><table class="detail-table"><thead><tr><th>Vehical Type</th><th>Route</th><th>Rate</th><th>Qty</th><th>Adult</th><th>Child</th><th>Net Amount</th></tr></thead><tbody>
            ${invoice.transports.map(t => `<tr><td class="text-left">${t.vehicle_type}</td><td class="text-left">${t.route}</td><td>${nf(t.rate)}</td><td>${t.qty}</td><td>${t.adult_qty}</td><td>${t.child_qty}</td><td>${nf(t.total_amount)}</td></tr>`).join('')}
            <tr class="total-row"><td colspan="6" class="total-label">Total Transportation:</td><td class="total-value">${nf(transport_total)}</td></tr></tbody></table>`;
                }
                if (invoice.other_services && invoice.other_services.length > 0) {
                    servicesHtml = `<div class="section-title">Other Services</div><table class="detail-table"><thead><tr><th rowspan="2">Service Name</th><th colspan="2">Adult</th><th colspan="2">Child</th><th colspan="2">Infant</th><th rowspan="2">Net Amount</th></tr><tr class="sub-header"><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th></tr></thead><tbody>
            ${invoice.other_services.map(s => `<tr><td class="text-left">${s.service_name}</td><td>${nf(s.adult_rate)}</td><td>${s.adult_qty}</td><td>${nf(s.child_rate)}</td><td>${s.child_qty}</td><td>${nf(s.infant_rate)}</td><td>${s.infant_qty}</td><td>${nf(s.total_amount)}</td></tr>`).join('')}
            <tr class="total-row"><td colspan="7" class="total-label">Total Services:</td><td class="total-value">${nf(services_total)}</td></tr></tbody></table>`;
                }
                if (invoice.airline_tickets && invoice.airline_tickets.length > 0) {
                    ticketsHtml = `<div class="section-title">Airline Tickets</div><table class="detail-table"><thead><tr><th rowspan="2">Airline</th><th rowspan="2">Sector</th><th colspan="2">Adult</th><th colspan="2">Child</th><th colspan="2">Infant</th><th rowspan="2">Net Amount</th></tr><tr class="sub-header"><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th></tr></thead><tbody>
            ${invoice.airline_tickets.map(t => `<tr><td class="text-left">${t.airline}</td><td>${t.sector}</td><td>${t.adult_qty}</td><td>${nf(t.adult_rate)}</td><td>${t.child_qty}</td><td>${nf(t.child_rate)}</td><td>${t.infant_qty}</td><td>${nf(t.infant_rate)}</td><td>${nf(t.total_amount)}</td></tr>`).join('')}
            <tr class="total-row"><td colspan="8" class="total-label">Total Airline Charges:</td><td class="total-value">${nf(tickets_total)}</td></tr></tbody></table>`;
                }

                modalBody.innerHTML = `
        <div class="invoice-wrapper">
            ${getWatermarkHtml(invoice)}
            <div class="invoice-content">
                <header>
                    <table class="header-table">
                        <tr>
                            ${getAgentLogoHtml(invoice)}
                            <td style="width: 34%;" class="company-logo-container">
                                <img src="images/logo.png" alt="Logo">
                               <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>
                                Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                            </td>
                            <td style="width: 33%;">
                                <table class="meta-table">
                                    <tr><td>Invoice No:</td><td>${invoice_display_number}</td></tr>
                                    <tr><td>Guest Name:</td><td>${invoice.guest_name}</td></tr>
                                    <tr><td>Dated:</td><td>${formatDate(invoice.issue_date)}</td></tr>
                                </table>
                            </td>
                        </tr>
                    </table>
                </header>
                <main>${pilgrimsHtml}${hotelsHtml}${transportsHtml}${servicesHtml}${ticketsHtml}</main>
                <footer class="footer-container">
                    <div class="footer-notes">${invoice.notes ? `<h4>Terms & Conditions:</h4><p>${invoice.notes.replace(/\n/g, '<br>')}</p>` : ''}</div>
                    <div class="summary-totals">
                        <table>
                            <tr><td>Total Amount (SAR)</td><td>${nf(invoice.total_sar)}</td></tr>
                            <tr><td>Exchange Rate:</td><td>${nf_decimal(invoice.exchange_rate)}</td></tr>
                            <tr><td>Total Amount (Pak Rs)</td><td>${nf(invoice.total_pkr_without_ticket)}</td></tr>
                            <tr><td>Ticket Amount (Pak Rs)</td><td>${nf(invoice.total_ticket_pkr)}</td></tr>
                            <tr class="grand-total"><td>Grand Total Including Ticket</td><td>${nf(invoice.grand_total_pkr)}</td></tr>
                            ${getPaymentSummaryHtml(invoice)}
                        </table>
                    </div>
                </footer>
                <div class="final-warning">SUBJECT TO AVAILABILITY AND EX RATES APPLIED AS PER DATE OF FULL PAYMENT</div>
            </div>
        </div>`;
            }
        });
    </script>

</body>

</html>