<?php
// This file assumes db-config.php has been included by the parent page (group-fares.php)
// And that session_start() has been called.

// --- CONFIGURATION ---
$whatsapp_number = "923052394810"; // Your WhatsApp number including country code, without '+' or '00'

// --- Check User Type ---
$is_agent = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');

// ======================================================================
//  2. BUILD & EXECUTE THE DATABASE QUERY
// ======================================================================
$fares_result = null;
$error_message = '';

// Base SQL query to get all active fares, ordered by the newest departure date first.
$sql = "SELECT gf.*, a.airline_name, a.logo_url 
        FROM group_fares gf 
        LEFT JOIN airlines a ON gf.airline_id = a.id
        WHERE gf.is_active = 1
        ORDER BY gf.departure_date DESC";

try {
    $stmt = $conn->prepare($sql);
    if ($stmt === false) {
        throw new Exception("Database prepare failed: " . $conn->error);
    }
    $stmt->execute();
    $fares_result = $stmt->get_result();
} catch (Exception $e) {
    $error_message = "Error fetching group fares: " . $e->getMessage();
    error_log($error_message);
}
?>

<style>
/* STYLES FOR THE GROUP FARE CARDS */
.group-fares-section {
    padding: 60px 20px;
    background-color: #f8f9fa;
}

/* Mobile Responsive (for screens below 768px) */
@media (max-width: 768px) {
    .group-fares-section {
        padding: 30px 15px; /* less padding for mobile */
        margin-top: 200px;
    }
}

/* Extra small screens (below 480px) */
@media (max-width: 480px) {
    .group-fares-section {
        padding: 20px 10px; /* even smaller for very small devices */
        margin-top: 200px;
    }
}

.listings-container {
    max-width: 1200px;
    margin: 0 auto;
}
.listings-header {
    text-align: center;
    margin-bottom: 50px;
}
.listings-title {
    font-size: 2.8rem;
    font-weight: 700;
    color: #2c3e50;
}
.listings-subtitle {
    font-size: 1.1rem;
    color: #7f8c8d;
    margin-top: 10px;
}
.fare-grid {
    display: grid;
    grid-template-columns: 1fr; /* Single column layout */
    gap: 30px;
}
.fare-card {
    background-color: #fff;
    border-radius: 12px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.08);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}
.fare-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 12px 35px rgba(0,0,0,0.12);
}
.fare-card-main {
    display: flex;
    flex-wrap: wrap; /* Allows wrapping on small screens */
    align-items: stretch;
}
.fare-airline-info {
    flex: 0 0 220px;
    padding: 25px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    text-align: center;
    border-right: 1px solid #e9ecef;
}
.fare-airline-info img {
    max-width: 120px;
    max-height: 50px;
    margin-bottom: 15px;
}
.fare-airline-name {
    font-weight: 600;
    font-size: 1.1rem;
    color: #34495e;
}
.fare-ref-id {
    font-size: 0.85rem;
    color: #95a5a6;
    margin-top: 5px;
    word-break: break-all;
}
.fare-details {
    flex: 1;
    padding: 25px;
    display: grid;
    grid-template-columns: repeat(4, 1fr);
    gap: 20px;
    align-items: center;
}
.detail-item {
    text-align: center;
}
.detail-item .label {
    font-size: 0.8rem;
    color: #7f8c8d;
    text-transform: uppercase;
    margin-bottom: 5px;
}
.detail-item .value {
    font-size: 1.1rem;
    font-weight: 600;
    color: #2c3e50;
}
.detail-item .value i {
    margin-right: 8px;
    color: #3498db;
}
.fare-pricing-action {
    flex: 0 0 250px;
    padding: 25px;
    border-left: 1px solid #e9ecef;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    background-color: #f8f9fa;
}
.fare-price .label {
    font-size: 0.9rem;
    color: #7f8c8d;
    text-align: center;
}
.fare-price .amount {
    font-size: 1.8rem;
    font-weight: 700;
    color: #e67e22;
    text-align: center;
    margin-bottom: 15px;
}
.btn-whatsapp-enquire {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
    width: 100%;
    padding: 12px 20px;
    background-color: #25D366;
    color: #fff;
    border: none;
    border-radius: 8px;
    font-size: 1rem;
    font-weight: 600;
    text-decoration: none;
    transition: background-color 0.3s ease;
}
.btn-whatsapp-enquire:hover {
    background-color: #128C7E;
}
.btn-whatsapp-enquire i {
    font-size: 1.2rem;
}

/* Responsive adjustments */
@media (max-width: 1200px) {
    .fare-details {
        grid-template-columns: repeat(2, 1fr);
    }
}
@media (max-width: 992px) {
    .fare-card-main {
        flex-direction: column;
    }
    .fare-airline-info, .fare-pricing-action {
        flex-basis: auto;
        width: 100%;
        border-right: none;
        border-left: none;
        border-bottom: 1px solid #e9ecef;
    }
    .fare-pricing-action {
        border-bottom: none;
        border-top: 1px solid #e9ecef;
    }
}
@media (max-width: 576px) {
    .fare-details {
        grid-template-columns: 1fr;
        gap: 25px;
        text-align: left;
    }
    .detail-item {
        text-align: left;
        display: flex;
        justify-content: space-between;
        align-items: center;
        border-bottom: 1px dashed #e0e0e0;
        padding-bottom: 10px;
    }
    .detail-item:last-child {
        border-bottom: none;
    }
    .detail-item .value {
        text-align: right;
    }
    .fare-airline-info {
        padding: 20px;
    }
}
</style>

<section class="group-fares-section">
    <div class="listings-container">
        <div class="listings-header">
            <h2 class="listings-title">Available Group Fares</h2>
            <p class="listings-subtitle">Find the perfect group deal for your upcoming travels.</p>
        </div>

        <div class="fare-grid">
            <?php if ($error_message): ?>
                <p class="notice error"><?= htmlspecialchars($error_message) ?></p>
            <?php elseif ($fares_result && $fares_result->num_rows > 0): ?>
                <?php while ($fare = $fares_result->fetch_assoc()): ?>
                    <?php
                        // Determine which price to show
                        $price_adult = ($is_agent && !empty($fare['price_adult_agent'])) ? $fare['price_adult_agent'] : $fare['price_adult'];

                        // Prepare WhatsApp message
                        $whatsapp_message = "Hi, I'm interested in the group fare with reference ID: " . $fare['group_ref_id'] . ". Please provide more details.";
                        $whatsapp_url = "https://wa.me/" . $whatsapp_number . "?text=" . urlencode($whatsapp_message);
                    ?>
                    <div class="fare-card">
                        <div class="fare-card-main">
                            <div class="fare-airline-info">
                                <?php if (!empty($fare['logo_url'])): ?>
                                    <img src="<?= htmlspecialchars($fare['logo_url']) ?>" alt="<?= htmlspecialchars($fare['airline_name']) ?> Logo">
                                <?php endif; ?>
                                <div class="fare-airline-name"><?= htmlspecialchars($fare['airline_name']) ?></div>
                                <div class="fare-ref-id">Ref: <?= htmlspecialchars($fare['group_ref_id']) ?></div>
                            </div>
                            <div class="fare-details">
                                <div class="detail-item">
                                    <div class="label">Route</div>
                                    <div class="value"><i class="fas fa-plane-departure"></i> <?= htmlspecialchars($fare['route']) ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="label">Departure</div>
                                    <div class="value"><i class="fas fa-calendar-alt"></i> <?= date("d M, Y", strtotime($fare['departure_date'])) ?></div>
                                </div>
                                <div class="detail-item">
                                    <div class="label">Duration</div>
                                    <div class="value"><i class="fas fa-clock"></i> <?= htmlspecialchars($fare['duration_days']) ?> Days</div>
                                </div>
                                <div class="detail-item">
                                    <div class="label">Seats Left</div>
                                    <div class="value"><i class="fas fa-chair"></i> <?= htmlspecialchars($fare['remaining_seats']) ?></div>
                                </div>
                            </div>
                            <div class="fare-pricing-action">
                                <div class="fare-price">
                                    <div class="label">Adult Price Starts From</div>
                                    <div class="amount">PKR <?= number_format($price_adult) ?></div>
                                </div>
                                <a href="<?= $whatsapp_url ?>" class="btn-whatsapp-enquire" target="_blank">
                                    <i class="fab fa-whatsapp"></i> Enquire Now
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <div class="content-card empty-state" style="text-align:center; padding: 4rem;">
                    <h3>No Group Fares Found</h3>
                    <p>There are currently no active group fares available. Please check back later.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</section>