<?php
// This is the main page for displaying ALL Umrah packages.
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
// Ensure the DB connection is available.
if (!isset($conn)) {
    require_once 'db-config.php';
}

// ======================================================================
//  EASY CONFIGURATION: SET YOUR EXCHANGE RATE HERE
// ======================================================================
define('SAR_TO_PKR_RATE', 75.50); // <--- UPDATE THIS VALUE AS NEEDED
// ======================================================================

// Check if the logged-in user is an agent
$is_agent = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');

// --- PREPARE AGENT/COMPANY DATA FOR THE POST FOOTER ---
$footer_logo = 'images/logo.png'; // Fallback
$footer_address = 'Al Quresh, Railway Phatak, Opposite Al Quresh Housing, Sher Shah Road, Multan'; // Fallback
$footer_phone_1 = '+92 305 2394810'; // Fallback
$footer_phone_2 = ''; // Fallback for a second number

if ($is_agent && isset($_SESSION['user_id'])) {
    // Fetch the logged-in agent's specific details from the database
    $agent_id = $_SESSION['user_id'];
    $stmt = $conn->prepare("SELECT logo_path, company_address, company_phone_1, company_phone_2 FROM users WHERE id = ?");
    $stmt->bind_param("i", $agent_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($agent_data = $result->fetch_assoc()) {
        if (!empty($agent_data['logo_path'])) {
            $footer_logo = $agent_data['logo_path'];
        }
        if (!empty($agent_data['company_address'])) {
            $footer_address = $agent_data['company_address'];
        }
        if (!empty($agent_data['company_phone_1'])) {
            $footer_phone_1 = $agent_data['company_phone_1'];
        }
        if (!empty($agent_data['company_phone_2'])) {
            $footer_phone_2 = $agent_data['company_phone_2'];
        }
    }
    $stmt->close();
}

// Fetch ALL active Umrah packages for this page
$packages_for_grid = $conn->query("
    SELECT up.*, a.airline_name, a.logo_url 
    FROM umrah_packages up 
    LEFT JOIN airlines a ON up.airline_id = a.id 
    WHERE up.is_active = 1 
    ORDER BY up.last_updated DESC
");

// Helper function to get the correct price (in PKR) based on user type
function get_price_for_user($package, $price_type, $is_agent) {
    $agent_price_key = $price_type . '_agent';
    $sar_price = null;

    if ($is_agent && isset($package[$agent_price_key]) && $package[$agent_price_key] !== null) {
        $sar_price = $package[$agent_price_key];
    } else {
        $sar_price = $package[$price_type] ?? null;
    }

    if ($sar_price !== null) {
        return number_format($sar_price * SAR_TO_PKR_RATE);
    }
    return null;
}

// FINAL FIX: Safe htmlspecialchars function to prevent errors with null values
function safe_htmlspecialchars($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// New helper function to determine the correct logo source path.
// It checks if the path is a full URL or a local file path.
function get_airline_logo_src($logo_path) {
    if (empty($logo_path)) {
        return ''; // Return empty string if no path is provided
    }
    // Check if it's an absolute URL (starts with http, https, or //)
    if (preg_match('/^(https?:)?\/\//', $logo_path)) {
        return safe_htmlspecialchars($logo_path); // It's a URL, use it directly
    }
    // Otherwise, it's a local path from the admin upload system, add the prefix
    return 'admin/' . safe_htmlspecialchars($logo_path);
}

?>

<!-- ======================================================== -->
<!-- === FINAL V6 STYLES: ALL DETAILS & FIXED DOWNLOAD POST === -->
<!-- ======================================================== -->
<style>
    /* --- Main Grid Layout --- */
    .umrah-grid-v6 { display: grid; grid-template-columns: repeat(auto-fill, minmax(350px, 1fr)); gap: 2rem; }
    @media (min-width: 1200px) { .umrah-grid-v6 { grid-template-columns: repeat(3, 1fr); } }
    
    /* --- V6 Visible Card (Original Layout Restored) --- */
    .umrah-card-v6 { background: #fff; border-radius: 12px; box-shadow: 0 4px 25px rgba(0,0,0,0.08); overflow: hidden; display: flex; flex-direction: column; transition: transform 0.3s, box-shadow 0.3s; }
    .umrah-card-v6:hover { transform: translateY(-5px); box-shadow: 0 8px 30px rgba(0,0,0,0.12); }
    .umrah-card-v6-banner { position: relative; }
    .umrah-card-v6-image { display: block; width: 100%; height: 220px; object-fit: cover; }
    .card-v6-logo-overlay { position: absolute; top: 15px; left: 15px; background: rgba(255,255,255,0.85); backdrop-filter: blur(5px); border-radius: 50%; width: 50px; height: 50px; padding: 6px; display: flex; align-items: center; justify-content: center; box-shadow: 0 1px 5px rgba(0,0,0,0.15); }
    .card-v6-logo-overlay.transparent-logo { background: none; backdrop-filter: none; box-shadow: none; }
    .card-v6-logo-overlay img { max-width: 100%; max-height: 100%; object-fit: contain; }
    .card-v6-days-badge { position: absolute; bottom: 15px; right: 15px; background: var(--primary-color); color: #fff; padding: 5px 12px; border-radius: 20px; font-size: 0.8rem; font-weight: 600; box-shadow: 0 2px 6px rgba(0,0,0,0.25); }
    .card-v6-airline-logo { position: absolute; top: 15px; right: 15px; background: rgba(255,255,255,0.85); backdrop-filter: blur(5px); border-radius: 8px; padding: 5px 8px; max-height: 35px; box-shadow: 0 1px 5px rgba(0,0,0,0.15); display:flex; align-items: center; justify-content: center; }
    .card-v6-airline-logo img { max-height: 25px; display: block; }
    .card-v6-flight-info { position: absolute; bottom: 15px; left: 15px; background: rgba(252, 252, 252, 0.9); backdrop-filter: blur(5px); border-radius: 8px; padding: 6px 10px; box-shadow: 0 1px 5px rgba(0,0,0,0.15); font-size: 0.7rem; color: #333; font-weight: 500; display: flex; flex-direction: column; align-items: flex-start; gap: 4px; max-width: calc(100% - 130px); /* Adjusted to avoid overlap with days badge */ }
    .card-v6-flight-info p { margin: 0; display: flex; align-items: center; gap: 5px; }
    .card-v6-flight-info p i { font-size: 0.8rem; flex-shrink: 0; }
    .card-v6-flight-info p span { white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
    .umrah-card-v6-content { padding: 1.25rem; display: flex; flex-direction: column; flex-grow: 1; }
    .umrah-name-v6 { font-size: 1.25rem; font-weight: 600; color: var(--text-dark); margin: 0 0 1rem 0; line-height: 1.3; }
    .umrah-card-v6-details-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; font-size: 0.8rem; color: #555; margin-bottom: 1rem; border-bottom: 1px solid #f0f0f0; padding-bottom: 1rem; }
    .detail-col h4 { font-size: 0.8rem; font-weight: 600; margin: 0 0 0.5rem 0; color: #333; }
    .detail-col ul { margin: 0; padding: 0; list-style: none; display: flex; flex-direction: column; gap: 0.5rem; }
    .detail-col ul li { display: flex; align-items: flex-start; gap: 8px; line-height: 1.4; }
    .detail-col ul i { color: var(--primary-gold); margin-top: 3px; }
    .pricing-grid-umrah-v6 { display: grid; grid-template-columns: repeat(auto-fit, minmax(120px, 1fr)); gap: 8px; margin-top: auto; }
    .price-item-umrah-v6 { text-align: center; }
    .price-item-umrah-v6 span { font-size: 0.7rem; color: #777; text-transform: uppercase; display: block; }
    .price-item-umrah-v6 strong { font-size: 1.2rem; color: var(--text-dark); font-weight: 700; }
    .card-v6-actions { display: grid; grid-template-columns: 1fr 1fr; gap: 0.75rem; margin-top: 1rem; border-top: 1px solid #f0f0f0; padding-top: 1rem; }
    .btn-download-card { width: 100%; padding: 12px; font-size: 0.9rem; font-weight: 600; text-align: center; border-radius: 8px; cursor: pointer; transition: background-color 0.3s, color 0.3s; background-color: #f0f0f0; color: #333; border: 1px solid #ddd; }
    .btn-download-card:hover:not(:disabled) { background-color: #e0e0e0; border-color: #ccc; }
    .btn-download-card:disabled { cursor: not-allowed; opacity: 0.7; } .btn-download-card i { margin-right: 8px; }

    /* --- V6 Downloadable Post (FIXED LAYOUT) --- */
    .downloadable-post-v6 { display: none; width: 1080px; height: 1080px; font-family: 'Poppins', sans-serif; position: relative; color: #fff; background: #1a237e; }
    .post-v6-bg-image { position: absolute; top: 0; left: 0; width: 100%; object-fit: cover; z-index: 1; }
    .post-v6-overlay { position: relative; z-index: 2; width: 100%; height: 100%; display: flex; flex-direction: column; background: linear-gradient(to top, rgba(10, 25, 41, 1) 0%, rgba(10, 25, 41, 0.98) 55%, transparent 80%); }
    .post-v6-header { display: flex; justify-content: space-between; align-items: flex-start; padding: 40px 40px 0 40px; }
    .post-v6-logo.white-bg-logo { background-color: rgba(255, 255, 255, 0.9); border-radius: 50%; width: 130px; height: 130px; display: flex; align-items: center; justify-content: center; box-shadow: 0 4px 10px rgba(0,0,0,0.4); }
    .post-v6-logo.white-bg-logo img { max-height: 100px; max-width: 100px; }
    .post-v6-logo.transparent-logo img { max-height: 110px; filter: drop-shadow(0px 4px 6px rgba(0,0,0,0.6)); }
    .post-v6-days-badge { background: #12a7ee; color: #fff; padding: 12px 25px; border-radius: 50px; text-align: center; box-shadow: 0 5px 15px rgba(0,0,0,0.3); }
    .post-v6-days-badge strong { font-size: 48px; display: block; line-height: 1; font-weight: 700; }
    .post-v6-days-badge span { font-size: 16px; font-weight: 500; letter-spacing: 1px; }
    .post-v6-body { margin-top: auto; padding: 0 40px; width: 100%; }
    .post-v6-title { text-align: center; margin-bottom: 25px; }
    .post-v6-title h1 { font-size: 78px; margin: 0; line-height: 1.1; font-weight: 800; text-transform: uppercase; text-shadow: 2px 2px 8px rgba(0,0,0,0.8); }
    .post-v6-title h2 { font-size: 32px; margin: 5px 0 0; font-weight: 400; opacity: 0.9; text-shadow: 1px 1px 5px rgba(0,0,0,0.8); letter-spacing: 2px;}
    .post-v6-flight-details { display: flex; flex-direction: column; align-items: flex-start; gap: 15px; margin-bottom: 25px; }
    .post-v6-flight-details .flight-airline-logo { max-height: 45px; filter: brightness(0) invert(1); }
    .post-v6-flight-details .flight-routes { display: flex; flex-direction: column; align-items: flex-start; gap: 10px; }
    .post-v6-flight-details .flight-route-item { font-size: 18px; font-weight: 500; margin: 0; opacity: 0.95; white-space: nowrap; }
    .post-v6-flight-details .flight-route-item i { margin-right: 10px; color: #12a7ee; }
    .post-v6-main-content { display: flex; gap: 25px; margin-bottom: 25px; align-items: stretch; }
    .post-v6-details-col, .post-v6-info-col { background: rgba(0, 0, 0, 0.55); backdrop-filter: blur(18px); -webkit-backdrop-filter: blur(18px); padding: 25px; border-radius: 20px; border: 1px solid rgba(255,255,255,0.15); flex: 1; display: flex; flex-direction: column; }
    .col-title { font-size: 24px; font-weight: 600; text-transform: uppercase; letter-spacing: 1px; border-bottom: 2px solid #12a7ee; padding-bottom: 8px; margin: 0 0 15px 0; flex-shrink: 0; }
    .post-v6-hotel-card { display: flex; align-items: center; gap: 15px; }
    .post-v6-hotel-card:not(:last-child) { margin-bottom: 15px; }
    .post-v6-hotel-card img.hotel-icon { height: 45px; flex-shrink: 0; }
    .post-v6-hotel-info h3 { font-size: 22px; margin: 0 0 2px; font-weight: 600; line-height: 1.2; }
    .post-v6-hotel-info p { font-size: 15px; margin: 0; opacity: 0.85; }
    .post-v6-info-list { list-style: none; padding: 0; margin: 0; font-size: 16px; overflow-y: auto; }
    .post-v6-info-list li { display: flex; align-items: flex-start; gap: 10px; margin-bottom: 8px; line-height: 1.4; }
    .post-v6-info-list i { color: #12a7ee; margin-top: 4px; flex-shrink: 0; }
    .post-v6-footer { padding: 0 40px 15px 40px; }
    .post-v6-pricing-grid { display: flex; justify-content: center; flex-wrap: wrap; gap: 15px; margin-bottom: 15px; }
    .post-v6-price-item { background: rgba(255, 255, 255, 0.15); backdrop-filter: blur(10px); border: 1px solid rgba(255,255,255,0.2); padding: 10px 20px; border-radius: 12px; text-align: center; min-width: 170px; }
    .post-v6-price-item span { display: block; font-size: 18px; font-weight: 500; opacity: 0.8; text-transform: uppercase; }
    .post-v6-price-item strong { display: block; font-size: 38px; font-weight: 700; line-height: 1.1; }
    .post-v6-footer-bar { background: #12a7ee; padding: 12px 40px; border-radius: 15px; display: flex; justify-content: space-between; align-items: center; font-size: 17px; font-weight: 500; flex-wrap: nowrap; white-space: nowrap; }
    .post-v6-footer-bar i { margin-right: 8px; } .post-v6-footer-bar span.separator { margin: 0 10px; }
    .post-v6-footer-bar, .post-v6-footer-bar * { text-decoration: none !important; }

    /* --- MOBILE RESPONSIVE IMPROVEMENTS --- */
    @media (max-width: 768px) {
        .umrah-listings-section .listings-container { padding-left: 0; padding-right: 0; }
        .umrah-grid-v6 { grid-template-columns: 1fr; gap: 1rem; padding: 0 1rem; }

        /* Card content adjustments for smaller screens */
        .umrah-card-v6-content { padding: 0.75rem; }
        .umrah-name-v6 { font-size: 1.1rem; margin-bottom: 0.75rem; }
        .umrah-card-v6-details-grid { 
            grid-template-columns: 1fr; /* Stack columns */
            gap: 0.5rem; 
            padding-bottom: 0.75rem;
            margin-bottom: 0.75rem;
        }
        .price-item-umrah-v6 strong { font-size: 1.1rem; }
        .card-v6-actions { padding-top: 0.75rem; gap: 0.5rem; }
        .btn-view-deal, .btn-download-card { padding: 10px; font-size: 0.8rem; }
    }
    @media (max-width: 480px) {
        .umrah-grid-v6 { padding: 0 0.5rem; }
    }
</style>

<section class="umrah-listings-section">
    <div class="listings-container">
        <div class="listings-header">
            <div class="listings-header-text">
                <h2 class="listings-title">All Umrah Packages</h2>
                <p class="listings-subtitle">Choose from our curated packages for a blessed journey.</p>
            </div>
        </div>

        <div class="umrah-grid-v6">
            <?php if ($packages_for_grid && $packages_for_grid->num_rows > 0): ?>
                <?php while ($pkg = $packages_for_grid->fetch_assoc()): ?>
                    <!-- VISIBLE CARD ON THE WEBSITE (MODIFIED LAYOUT) -->
                    <div class="umrah-card-v6">
                        <div class="umrah-card-v6-banner">
                            <!-- Agent/Company Logo (Top Left) -->
                            <?php $logo_overlay_class = 'card-v6-logo-overlay' . ((isset($pkg['logo_has_background']) && $pkg['logo_has_background'] == 0) ? ' transparent-logo' : ''); ?>
                            <div class="<?= $logo_overlay_class ?>">
                                <img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous">
                            </div>
                            
                            <!-- Airline Logo (Top Right) -->
                            <?php 
                                $airline_logo_src = get_airline_logo_src($pkg['logo_url']);
                                if (!empty($airline_logo_src)): 
                            ?>
                                <div class="card-v6-airline-logo">
                                    <img src="<?= $airline_logo_src ?>" alt="<?= safe_htmlspecialchars($pkg['airline_name']) ?>" crossorigin="anonymous">
                                </div>
                            <?php endif; ?>

                            <!-- Stacked Flight Info (Bottom Left) -->
                            <?php if (!empty($pkg['outbound_flight_details']) || !empty($pkg['inbound_flight_details'])): ?>
                                <div class="card-v6-flight-info">
                                    <?php if (!empty($pkg['outbound_flight_details'])): ?>
                                        <p><i class="fas fa-plane-departure"></i> <span><?= safe_htmlspecialchars($pkg['outbound_flight_details']) ?></span></p>
                                    <?php endif; ?>
                                    <?php if (!empty($pkg['inbound_flight_details'])): ?>
                                        <p><i class="fas fa-plane-arrival"></i> <span><?= safe_htmlspecialchars($pkg['inbound_flight_details']) ?></span></p>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Days Badge (Bottom Right) -->
                            <?php if (!empty($pkg['days'])): ?>
                                <div class="card-v6-days-badge"><?= safe_htmlspecialchars($pkg['days']) ?> DAYS</div>
                            <?php endif; ?>

                            <img src="<?= safe_htmlspecialchars($pkg['main_image_link'] ?: 'images/default-package.jpg') ?>" alt="<?= safe_htmlspecialchars($pkg['package_name']) ?>" class="umrah-card-v6-image" loading="lazy" crossorigin="anonymous">
                        </div>
                        <div class="umrah-card-v6-content">
                            <h3 class="umrah-name-v6"><?= safe_htmlspecialchars($pkg['package_name']) ?></h3>
                            <div class="umrah-card-v6-details-grid">
                                <div class="detail-col">
                                    <h4>Accommodation</h4>
                                    <ul>
                                        <?php if (!empty($pkg['makkah_hotel'])): ?>
                                            <li><i class="fas fa-mosque"></i> <div class="detail-item-text"><?= safe_htmlspecialchars($pkg['makkah_hotel']) ?> <span>Makkah: <?= safe_htmlspecialchars($pkg['makkah_hotel_distance']) ?> | <?= safe_htmlspecialchars($pkg['makkah_nights']) ?>N</span></div></li>
                                        <?php endif; ?>
                                        <?php if (!empty($pkg['madinah_hotel'])): ?>
                                            <li><i class="fas fa-mosque"></i> <div class="detail-item-text"><?= safe_htmlspecialchars($pkg['madinah_hotel']) ?> <span>Madinah: <?= safe_htmlspecialchars($pkg['madinah_hotel_distance']) ?> | <?= safe_htmlspecialchars($pkg['madinah_nights']) ?>N</span></div></li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                                <div class="detail-col">
                                    <h4>Package Details</h4>
                                    <ul>
                                        <?php if (!empty($pkg['whats_included'])):
                                            $includes_array = preg_split('/,*\R+/', $pkg['whats_included'], -1, PREG_SPLIT_NO_EMPTY);
                                            $first_inclusion = $includes_array[0] ?? null;
                                            if ($first_inclusion):
                                        ?>
                                            <li><i class="fas fa-check-circle"></i> <div class="detail-item-text"><?= safe_htmlspecialchars(trim($first_inclusion)) ?></div></li>
                                        <?php endif; endif; ?>
                                    </ul>
                                </div>
                            </div>
                            <div class="pricing-grid-umrah-v6">
                                <?php 
                                    $price_quint = get_price_for_user($pkg, 'price_quint', $is_agent);
                                    $price_quad = get_price_for_user($pkg, 'price_quad', $is_agent);
                                    $price_triple = get_price_for_user($pkg, 'price_triple', $is_agent);
                                    $price_double = get_price_for_user($pkg, 'price_double', $is_agent);
                                ?>
                                <?php if ($price_quint): ?><div class="price-item-umrah-v6"><span>Sharing</span><strong><?= $price_quint ?>/-</strong></div><?php endif; ?>
                                <?php if ($price_quad): ?><div class="price-item-umrah-v6"><span>Quad</span><strong><?= $price_quad ?>/-</strong></div><?php endif; ?>
                                <?php if ($price_triple): ?><div class="price-item-umrah-v6"><span>Triple</span><strong><?= $price_triple ?>/-</strong></div><?php endif; ?>
                                <?php if ($price_double): ?><div class="price-item-umrah-v6"><span>Double</span><strong><?= $price_double ?>/-</strong></div><?php endif; ?>
                            </div>
                            <div class="card-v6-actions">
                                <a href="umrah-package-template.php?id=<?= urlencode($pkg['package_id']) ?>" class="btn-view-deal">View Details</a>
                                <button class="btn-download-card" data-package-id="<?= $pkg['package_id'] ?>" data-package-name="<?= safe_htmlspecialchars($pkg['package_name']) ?>"><i class="fa-solid fa-download"></i> Download JPG</button>
                            </div>
                        </div>
                    </div>

                    <!-- HIDDEN V6 POST FOR DOWNLOADING (MODIFIED LAYOUT) -->
                    <div class="downloadable-post-v6" id="download-post-<?= $pkg['package_id'] ?>">
                        <img src="<?= safe_htmlspecialchars($pkg['main_image_link'] ?: 'images/default-package.jpg') ?>" class="post-v6-bg-image" crossorigin="anonymous" alt="">
                        <div class="post-v6-overlay">
                            <header class="post-v6-header">
                                <?php $logo_bg_class = (isset($pkg['logo_has_background']) && $pkg['logo_has_background'] == 0) ? 'post-v6-logo transparent-logo' : 'post-v6-logo white-bg-logo'; ?>
                                <div class="<?= $logo_bg_class ?>"><img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous"></div>
                                <?php if (!empty($pkg['days'])): ?>
                                    <div class="post-v6-days-badge"><strong><?= safe_htmlspecialchars($pkg['days']) ?></strong><span>DAYS</span></div>
                                <?php endif; ?>
                            </header>
                            <main class="post-v6-body">
                                <div class="post-v6-title">
                                    <h1><?= safe_htmlspecialchars($pkg['package_name']) ?></h1>
                                    <h2>UMRAH PACKAGE</h2>
                                </div>
                                
                                <div class="post-v6-main-content">
                                    <div class="post-v6-details-col">
                                        <?php if (!empty($pkg['logo_url']) || !empty($pkg['outbound_flight_details']) || !empty($pkg['inbound_flight_details'])): ?>
                                        <div class="post-v6-flight-details">
                                            <?php 
                                                $airline_logo_src_post = get_airline_logo_src($pkg['logo_url']);
                                                if (!empty($airline_logo_src_post)): 
                                            ?>
                                                <img src="<?= $airline_logo_src_post ?>" class="flight-airline-logo" alt="<?= safe_htmlspecialchars($pkg['airline_name']) ?>" crossorigin="anonymous">
                                            <?php endif; ?>
                                            <div class="flight-routes">
                                                <?php if (!empty($pkg['outbound_flight_details'])): ?>
                                                    <p class="flight-route-item"><i class="fas fa-plane-departure"></i> <?= safe_htmlspecialchars($pkg['outbound_flight_details']) ?></p>
                                                <?php endif; ?>
                                                <?php if (!empty($pkg['inbound_flight_details'])): ?>
                                                    <p class="flight-route-item"><i class="fas fa-plane-arrival"></i> <?= safe_htmlspecialchars($pkg['inbound_flight_details']) ?></p>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>

                                        <h3 class="col-title">Accommodation</h3>
                                        <?php if (!empty($pkg['makkah_hotel'])): ?>
                                        <div class="post-v6-hotel-card">
                                            <img src="images/makkah-icon.png" class="hotel-icon" alt="Makkah" crossorigin="anonymous">
                                            <div class="post-v6-hotel-info">
                                                <h3><?= safe_htmlspecialchars($pkg['makkah_hotel']) ?></h3>
                                                <p><?= safe_htmlspecialchars($pkg['makkah_hotel_distance']) ?> | <?= safe_htmlspecialchars($pkg['makkah_nights']) ?> Nights</p>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                        <?php if (!empty($pkg['madinah_hotel'])): ?>
                                        <div class="post-v6-hotel-card">
                                            <img src="images/madinah-icon.png" class="hotel-icon" alt="Madinah" crossorigin="anonymous">
                                            <div class="post-v6-hotel-info">
                                                <h3><?= safe_htmlspecialchars($pkg['madinah_hotel']) ?></h3>
                                                <p><?= safe_htmlspecialchars($pkg['madinah_hotel_distance']) ?> | <?= safe_htmlspecialchars($pkg['madinah_nights']) ?> Nights</p>
                                            </div>
                                        </div>
                                        <?php endif; ?>
                                    </div>
                                    <div class="post-v6-info-col">
                                        <h3 class="col-title">Package Includes</h3>
                                        <ul class="post-v6-info-list">
                                        <?php
                                            if (!empty($pkg['whats_included'])) {
                                                $includes = preg_split('/,*\R+/', $pkg['whats_included'], -1, PREG_SPLIT_NO_EMPTY);
                                                foreach ($includes as $item) {
                                                    echo '<li><i class="fas fa-check-circle"></i>' . safe_htmlspecialchars(trim($item)) . '</li>';
                                                }
                                            }
                                        ?>
                                        </ul>
                                    </div>
                                </div>
                            </main>
                            <footer class="post-v6-footer">
                                <div class="post-v6-pricing-grid">
                                    <?php if ($price_quint): ?><div class="post-v6-price-item"><span>SHARING</span><strong><?= $price_quint ?></strong></div><?php endif; ?>
                                    <?php if ($price_quad): ?><div class="post-v6-price-item"><span>QUAD</span><strong><?= $price_quad ?></strong></div><?php endif; ?>
                                    <?php if ($price_triple): ?><div class="post-v6-price-item"><span>TRIPLE</span><strong><?= $price_triple ?></strong></div><?php endif; ?>
                                    <?php if ($price_double): ?><div class="post-v6-price-item"><span>DOUBLE</span><strong><?= $price_double ?></strong></div><?php endif; ?>
                                </div>
                                <div class="post-v6-footer-bar">
                                    <div><i class="fa-solid fa-location-dot"></i> <?= safe_htmlspecialchars($footer_address) ?></div>
                                    <div>
                                        <i class="fa-brands fa-whatsapp"></i> <?= safe_htmlspecialchars($footer_phone_1) ?>
                                        <?php if (!empty($footer_phone_2)): ?>
                                            <span class="separator">|</span><i class="fa-solid fa-phone"></i> <?= safe_htmlspecialchars($footer_phone_2) ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </footer>
                        </div>
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <p>No Umrah packages are available at the moment. Please check back later.</p>
            <?php endif; ?>
        </div>
    </div>
</section>

<script>
let html2canvasLoadedV6 = false;
function loadHtml2CanvasV6(callback) {
    if (html2canvasLoadedV6) { if (callback) callback(); return; }
    const script = document.createElement('script');
    script.src = 'js/html2canvas.min.js'; 
    script.onload = function() { html2canvasLoadedV6 = true; if (callback) callback(); };
    script.onerror = function() {
        console.error("Failed to load html2canvas.min.js from the '/js/' folder.");
        document.querySelectorAll('.btn-download-card:disabled').forEach(button => {
            button.innerHTML = '<i class="fa-solid fa-download"></i> Download';
            button.disabled = false;
        });
    };
    document.head.appendChild(script);
}

document.addEventListener('DOMContentLoaded', function () {
    document.querySelectorAll('.btn-download-card').forEach(button => {
        button.addEventListener('click', function () {
            const originalButtonHTML = this.innerHTML;
            this.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Please Wait';
            this.disabled = true;

            loadHtml2CanvasV6(() => {
                const packageId = this.dataset.packageId;
                const postToDownload = document.getElementById(`download-post-${packageId}`);
                if (!postToDownload) {
                    console.error('Downloadable post element not found for package ID:', packageId);
                    this.innerHTML = originalButtonHTML; this.disabled = false; return;
                }
                const packageName = this.dataset.packageName || 'umrah-package';
                const sanitizedFileName = packageName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
                
                document.body.appendChild(postToDownload);
                postToDownload.style.cssText = 'position: absolute; left: -9999px; top: 0; display: block;';

                html2canvas(postToDownload, { useCORS: true, scale: 1.5, logging: false, removeContainer: true })
                .then(canvas => {
                    const link = document.createElement('a');
                    link.href = canvas.toDataURL('image/jpeg', 0.9);
                    link.download = `${sanitizedFileName}.jpg`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                }).catch(err => {
                    console.error('Download failed!', err);
                    alert("Download Failed. This can happen if an image is linked from another website. Please ensure all package images are uploaded to your own server.");
                }).finally(() => {
                    if (document.body.contains(postToDownload)) {
                        postToDownload.style.display = 'none';
                    }
                    this.innerHTML = originalButtonHTML;
                    this.disabled = false;
                });
            });
        });
    });
});
</script>