<?php
session_start(); // Ensure session is started at the very top
include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user a CUSTOMER? ---
// If they are an agent, redirect them to their own dashboard.
if (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent') {
    header("location: agent-dashboard.php");
    exit;
}

// Get the customer's name and ID from the session
$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// --- FETCH FLIGHT BOOKINGS ---
$sql_flights = "SELECT * FROM bookings WHERE user_id = ? AND booking_type = 'flight' ORDER BY created_at DESC";
$stmt_flights = $conn->prepare($sql_flights);
$stmt_flights->bind_param("i", $user_id);
$stmt_flights->execute();
$flights_result = $stmt_flights->get_result();
$stmt_flights->close();

// --- FETCH GROUP BOOKINGS ---
$sql_groups = "SELECT * FROM bookings WHERE user_id = ? AND booking_type = 'group' ORDER BY created_at DESC";
$stmt_groups = $conn->prepare($sql_groups);
$stmt_groups->bind_param("i", $user_id);
$stmt_groups->execute();
$groups_result = $stmt_groups->get_result();
$stmt_groups->close();

?>
<!DOCTYPE html>
<html>

<head>
    <title>My Account</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">

    <style>
        .section-header {
            margin-top: 2rem;
            margin-bottom: 1rem;
            font-size: 1.4rem;
            color: #333;
            border-bottom: 2px solid #f0f0f0;
            padding-bottom: 0.5rem;
        }

        .section-header:first-of-type {
            margin-top: 0;
        }

        .actions-cell {
            text-align: center;
        }

        .actions-cell a {
            text-decoration: none;
        }

        /* === NEW & IMPROVED RESPONSIVE TABLE CSS === */
        @media screen and (max-width: 992px) {
            .bookings-table thead {
                display: none;
            }

            .bookings-table,
            .bookings-table tbody,
            .bookings-table tr,
            .bookings-table td {
                display: block;
                width: 100%;
            }

            .bookings-table tr {
                margin-bottom: 1.5rem; /* More space between cards */
                border: 1px solid var(--border-color);
                border-radius: 8px;
                overflow: hidden;
                background-color: #fff;
            }

            .bookings-table td {
                padding: 10px 15px;
                text-align: left; /* Align all text to the left */
                border: none;
                border-bottom: 1px solid var(--border-color-light);
            }
            
            .bookings-table td:last-child {
                border-bottom: 0;
            }

            .bookings-table td::before {
                content: attr(data-label);
                display: block; /* Creates the stacked layout */
                font-weight: 600;
                font-size: 0.8rem;
                color: #6c757d;
                margin-bottom: 5px;
                text-transform: uppercase;
            }

            /* Specific styling for more clarity */
            .bookings-table td[data-label="Booking Ref"] strong {
                font-size: 1.1em;
                color: var(--primary-color, #31a7e2);
            }
            
            .bookings-table td[data-label="Total Price"] {
                font-weight: bold;
            }

            .actions-cell {
                background-color: #f8f9fa;
                text-align: center !important; /* Center the button */
                padding-top: 12px;
                padding-bottom: 12px;
            }

            .actions-cell a {
                text-decoration: none;
                color: var(--primary-color, #31a7e2);
                font-weight: 500;
                transition: opacity 0.2s;
            }
            
            .actions-cell a:hover {
                opacity: 0.8;
            }

            .actions-cell a i {
                margin-right: 6px;
            }
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">

            <?php include 'customer-sidebar.php'; ?>

            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-ticket"></i> My Bookings</h2>
                    <p class="content-description">Here is a list of your bookings. Click the <i class="fas fa-eye"></i> icon to view the voucher for any booking.</p>

                    <!-- FLIGHT BOOKINGS SECTION -->
                    <h3 class="section-header"><i class="fa-solid fa-plane"></i> Flight Bookings</h3>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Booking Ref</th>
                                    <th>Primary Passenger</th>
                                    <th>Route</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Total Price</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($flights_result && $flights_result->num_rows > 0): ?>
                                    <?php while ($booking = $flights_result->fetch_assoc()): ?>
                                        <?php
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $passenger_details = json_decode($booking['passenger_details'], true);

                                        $primary_passenger_name = 'N/A';
                                        if (isset($passenger_details['adults'])) {
                                            $first_adult = reset($passenger_details['adults']);
                                            if ($first_adult) {
                                                $primary_passenger_name = htmlspecialchars(($first_adult['title'] ?? '') . '. ' . ($first_adult['firstName'] ?? '') . ' ' . ($first_adult['lastName'] ?? ''));
                                            }
                                        }

                                        $route = 'N/A';
                                        $date = 'N/A';
                                        if (isset($flight_details['itineraries'][0])) {
                                            $first_itinerary = $flight_details['itineraries'][0];
                                            $route = htmlspecialchars($first_itinerary['origin'] ?? '') . ' - ' . htmlspecialchars($first_itinerary['destination'] ?? '');
                                            if (!empty($first_itinerary['departureDate'])) {
                                                $date = date('M j, Y', strtotime($first_itinerary['departureDate']));
                                            }
                                        }

                                        $status_class = 'status-' . strtolower($booking['status']);
                                        ?>
                                        <tr>
                                            <td data-label="Booking Ref"><strong><?php echo htmlspecialchars($booking['booking_ref']); ?></strong></td>
                                            <td data-label="Passenger"><?php echo $primary_passenger_name; ?></td>
                                            <td data-label="Route"><?php echo $route; ?></td>
                                            <td data-label="Date"><?php echo $date; ?></td>
                                            <td data-label="Status"><span class="status-badge <?php echo $status_class; ?>"><?php echo ucfirst($booking['status']); ?></span></td>
                                            <td data-label="Total Price"><?php echo htmlspecialchars($booking['price_currency']) . ' ' . number_format($booking['total_price'], 0); ?></td>
                                            <td data-label="Actions" class="actions-cell">
                                                <a href="ticket-voucher.php?booking_id=<?php echo $booking['id']; ?>" target="_blank" rel="noopener noreferrer" title="View Voucher">
                                                    <i class="fas fa-eye"></i> View Voucher
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="no-bookings-found">You have no flight bookings.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- GROUP BOOKINGS SECTION -->
                    <h3 class="section-header"><i class="fa-solid fa-users"></i> Group Bookings</h3>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Booking Ref</th>
                                    <th>Primary Passenger</th>
                                    <th>Route</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Total Price</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($groups_result && $groups_result->num_rows > 0): ?>
                                    <?php while ($booking = $groups_result->fetch_assoc()): ?>
                                        <?php
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $passenger_details = json_decode($booking['passenger_details'], true);

                                        $primary_passenger_name = 'N/A';
                                        if (isset($passenger_details['adults'])) {
                                            $first_adult = reset($passenger_details['adults']);
                                            if ($first_adult) {
                                                $primary_passenger_name = htmlspecialchars(($first_adult['title'] ?? '') . '. ' . ($first_adult['given_name'] ?? '') . ' ' . ($first_adult['surname'] ?? ''));
                                            }
                                        }

                                        $route = htmlspecialchars($flight_details['route'] ?? 'Group Fare');

                                        $date = 'N/A';
                                        if (!empty($flight_details['departure_date'])) {
                                            $date = date('M j, Y', strtotime($flight_details['departure_date']));
                                        }

                                        $status_class = 'status-' . strtolower($booking['status']);
                                        ?>
                                        <tr>
                                            <td data-label="Booking Ref"><strong><?php echo htmlspecialchars($booking['booking_ref']); ?></strong></td>
                                            <td data-label="Passenger"><?php echo $primary_passenger_name; ?></td>
                                            <td data-label="Route"><?php echo $route; ?></td>
                                            <td data-label="Date"><?php echo $date; ?></td>
                                            <td data-label="Status"><span class="status-badge <?php echo $status_class; ?>"><?php echo ucfirst($booking['status']); ?></span></td>
                                            <td data-label="Total Price"><?php echo htmlspecialchars($booking['price_currency']) . ' ' . number_format($booking['total_price'], 0); ?></td>
                                            <td data-label="Actions" class="actions-cell">
                                                <a href="ticket-voucher.php?booking_id=<?php echo $booking['id']; ?>" target="_blank" rel="noopener noreferrer" title="View Voucher">
                                                    <i class="fas fa-eye"></i> View Voucher
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="no-bookings-found">You have no group bookings.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'floating-icon.php'; ?>
    <?php include 'footer.php'; ?>

</body>

</html>