<?php
session_start(); // Ensure session is started at the very top
include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user a CUSTOMER? ---
if (isset($_SESSION['user_type']) && $_SESSION['user_type'] !== 'customer') {
    $redirect_url = ($_SESSION['user_type'] === 'agent') ? 'agent-dashboard.php' : 'admin/index.php';
    header("location: $redirect_url");
    exit;
}

// Get the customer's name and ID from the session
$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name']; // For the sidebar

function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- DATA FETCHING FOR STANDARD UMRAH INQUIRIES ---
$standard_inquiries_result = $conn->execute_query("SELECT * FROM umrah_inquiries WHERE user_id = ? ORDER BY created_at DESC", [$user_id]);

// --- NEW: DATA FETCHING FOR CUSTOM PACKAGE INQUIRIES ---
$custom_inquiries_result = $conn->execute_query("SELECT * FROM custom_package_inquiries WHERE user_id = ? ORDER BY created_at DESC", [$user_id]);

?>
<!DOCTYPE html>
<html>
<head>
    <title>My Inquiries - Customer Dashboard</title>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css"> 
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .inquiry-list { border: 1px solid var(--border-color); border-radius: 8px; overflow: hidden; }
        .inquiry-item { padding: 15px 20px; border-bottom: 1px solid var(--border-color); background-color: #fff; transition: background-color 0.2s; }
        .inquiry-list .inquiry-item:last-child { border-bottom: none; }
        .inquiry-summary { display: flex; align-items: center; gap: 15px; cursor: pointer; }
        .inquiry-icon { font-size: 1.8rem; color: var(--primary-color); width: 30px; text-align: center; }
        .inquiry-primary-info { flex-grow: 1; }
        .inquiry-primary-info strong { font-size: 1.1rem; color: var(--text-dark); }
        .inquiry-primary-info span { font-size: 0.85rem; color: #6c757d; }
        .inquiry-meta { display: flex; align-items: center; gap: 15px; }
        .expand-arrow { font-size: 0.9rem; color: #6c757d; transition: transform 0.3s ease; }
        .inquiry-item.expanded .expand-arrow { transform: rotate(180deg); }

        .status-badge { padding: 5px 12px; border-radius: 20px; font-size: 0.75rem; font-weight: 700; text-transform: uppercase; color: #fff; min-width: 90px; text-align: center; }
        .status-pending { background-color: #ffc107; color: #000; }
        .status-in-progress { background-color: #0dcaf0; color: #000; }
        .status-completed { background-color: #198754; }
        .status-rejected { background-color: #dc3545; }

        .inquiry-details { max-height: 0; overflow: hidden; transition: all 0.4s ease-in-out; }
        .inquiry-item.expanded .inquiry-details { max-height: 1500px; padding-top: 20px; margin-top: 20px; border-top: 1px solid #e9ecef; }
        .details-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px 30px; font-size: 0.95rem; }
        .detail-item { color: #555; }
        .detail-item strong { color: var(--text-dark); display: block; margin-bottom: 3px; }
        
        .empty-state { text-align: center; padding: 50px 20px; color: #6c757d; }
        .empty-state i { font-size: 3.5rem; opacity: 0.4; margin-bottom: 15px; }

        /* NEW: Styles for the custom package details table */
        .package-details-table { width: 100%; margin-top: 1rem; border-collapse: collapse; }
        .package-details-table th, .package-details-table td { text-align: left; padding: 10px; border-bottom: 1px solid #eee; }
        .package-details-table th { background-color: #f8f9fa; font-weight: 600; font-size: 0.85rem; }
        .package-details-table .total-row td { font-weight: bold; border-top: 2px solid #ddd; }
        .package-details-table .grand-total-row td { background-color: #e9f5fd; color: #0353a4; font-weight: bold; font-size: 1.1rem; }
    </style>
</head>
<body>

<?php include 'header.php'; ?>

<main class="account-page-wrapper">
    <div class="account-container">
        
        <?php include 'customer-sidebar.php'; ?>

        <div class="account-content">
            <!-- SECTION 1: Standard Umrah Package Inquiries -->
            <div class="content-card">
                <h2><i class="fa-solid fa-kaaba"></i> My Umrah Package Inquiries</h2>
                <p class="content-description">Here is a list of all standard Umrah package inquiries you have submitted.</p>
                <div class="inquiry-list" id="standard-inquiry-list">
                    <?php if ($standard_inquiries_result && $standard_inquiries_result->num_rows > 0): ?>
                        <?php while ($inquiry = $standard_inquiries_result->fetch_assoc()): ?>
                            <div class="inquiry-item">
                                <div class="inquiry-summary">
                                    <div class="inquiry-icon"><i class="fas fa-kaaba"></i></div>
                                    <div class="inquiry-primary-info">
                                        <strong>Package: <?php echo e($inquiry['package_name']); ?></strong>
                                        <span>Submitted: <?php echo date('d M Y, g:ia', strtotime($inquiry['created_at'])); ?></span>
                                    </div>
                                    <div class="inquiry-meta">
                                        <span class="status-badge status-<?php echo strtolower(str_replace(' ', '-', e($inquiry['status']))); ?>"><?php echo e($inquiry['status']); ?></span>
                                        <div class="expand-arrow"><i class="fas fa-chevron-down"></i></div>
                                    </div>
                                </div>
                                <div class="inquiry-details">
                                    <div class="details-grid">
                                        <div class="detail-item"><strong>Inquiry ID:</strong> #<?php echo e($inquiry['id']); ?></div>
                                        <div class="detail-item"><strong>Package ID:</strong> <?php echo e($inquiry['package_id']); ?></div>
                                        <div class="detail-item"><strong>Room Type:</strong> <?php echo e($inquiry['room_type']); ?></div>
                                        <div class="detail-item"><strong>Total Passengers:</strong> <?php echo e($inquiry['pax']); ?></div>
                                    </div>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <div class="empty-state"><i class="fas fa-folder-open"></i><p>You have not submitted any standard Umrah inquiries yet.</p></div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- NEW SECTION 2: Custom Package Inquiries -->
            <div class="content-card">
                <h2><i class="fas fa-calculator"></i> My Custom Package Inquiries</h2>
                <p class="content-description">These are the packages you have built using our custom package calculator.</p>
                <div class="inquiry-list" id="custom-inquiry-list">
                    <?php if ($custom_inquiries_result && $custom_inquiries_result->num_rows > 0): ?>
                        <?php while ($inquiry = $custom_inquiries_result->fetch_assoc()): 
                            $details = json_decode($inquiry['package_details_json'], true);
                            $summary = $details['summary'] ?? [];
                            $pax = $details['pax'] ?? [];
                        ?>
                            <div class="inquiry-item">
                                <div class="inquiry-summary">
                                    <div class="inquiry-icon"><i class="fas fa-calculator"></i></div>
                                    <div class="inquiry-primary-info">
                                        <strong>Flight Ref: <?php echo e($inquiry['booking_ref']); ?></strong>
                                        <span>Submitted: <?php echo date('d M Y, g:ia', strtotime($inquiry['created_at'])); ?></span>
                                    </div>
                                    <div class="inquiry-meta">
                                        <span class="status-badge status-<?php echo strtolower(str_replace(' ', '-', e($inquiry['status']))); ?>"><?php echo e($inquiry['status']); ?></span>
                                        <div class="expand-arrow"><i class="fas fa-chevron-down"></i></div>
                                    </div>
                                </div>
                                <div class="inquiry-details">
                                    <div class="details-grid">
                                        <div class="detail-item"><strong>Inquiry ID:</strong> #<?= e($inquiry['id']) ?></div>
                                        <div class="detail-item"><strong>Passengers:</strong> Adults: <?= e($pax['adults'] ?? 0) ?>, Children: <?= e($pax['children'] ?? 0) ?>, Infants: <?= e($pax['infants'] ?? 0) ?></div>
                                    </div>
                                    <h4 style="margin-top: 1.5rem; font-size: 1rem;">Package Breakdown</h4>
                                    <table class="package-details-table">
                                        <thead>
                                            <tr><th>Item</th><th>Details</th><th>Cost</th></tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach($details['components'] as $key => $comp): if(empty($comp['cost_sar']) && empty($comp['cost_pkr'])) continue; ?>
                                            <tr>
                                                <td><?= e(ucfirst(str_replace('_', ' ', $key))) ?></td>
                                                <td><?= e($comp['details']) ?></td>
                                                <td><?= !empty($comp['cost_pkr']) ? 'PKR '.e(number_format($comp['cost_pkr'])) : 'SAR '.e(number_format($comp['cost_sar'], 2)) ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                            <tr class="total-row">
                                                <td colspan="2">Total (SAR)</td>
                                                <td><?= e($summary['grand_total_sar'] ?? 'SAR 0.00') ?></td>
                                            </tr>
                                            <tr class="total-row">
                                                <td colspan="2">Grand Total (PKR)</td>
                                                <td><?= e($summary['grand_total_pkr'] ?? 'PKR 0') ?></td>
                                            </tr>
                                            <tr class="grand-total-row">
                                                <td colspan="2">Total Per Person</td>
                                                <td><?= e($summary['total_per_person'] ?? 'PKR 0') ?></td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <div class="empty-state"><i class="fas fa-folder-open"></i><p>You have not submitted any custom package inquiries yet.</p></div>
                    <?php endif; ?>
                </div>
            </div>
            
        </div>
    </div>
</main>

<?php include 'floating-icon.php'; ?>
<?php include 'footer.php'; ?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // This single function will handle accordions for both lists
    function setupAccordion(listId) {
        const inquiryList = document.getElementById(listId);
        if (inquiryList) {
            inquiryList.addEventListener('click', function(event) {
                const inquiryItem = event.target.closest('.inquiry-item');
                if (!inquiryItem) return;
                // Prevent toggling if a link is clicked inside the details
                if (!event.target.closest('a')) {
                    inquiryItem.classList.toggle('expanded');
                }
            });
        }
    }
    
    setupAccordion('standard-inquiry-list');
    setupAccordion('custom-inquiry-list');
});
</script>

</body>
</html>