<?php
// process-booking.php
include 'db-config.php';
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Security Check 1: Must be a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: index.php");
    exit();
}

// Security Check 2: User must be logged in
if (!isset($_SESSION['user_id'])) {
    $_SESSION['error_message'] = "Your session expired. Please log in again to complete your booking.";
    header("Location: login.php");
    exit();
}

$flight_id = $_POST['flight_id'] ?? null;

// Security Check 3: Flight data must exist in the session
if ($flight_id === null || !isset($_SESSION['flight_search_results'][$flight_id]) || !isset($_SESSION['flight_search_params'])) {
    $_SESSION['error_message_search'] = "Your flight selection seems to have expired. Please search for a flight again.";
    header("Location: index.php");
    exit();
}

// All checks passed, let's process the booking.
try {
    // 1. Get all necessary data
    $user_id = $_SESSION['user_id'];
    $flight = $_SESSION['flight_search_results'][$flight_id];
    $search_params = $_SESSION['flight_search_params'];
    $passengers = $_POST['passengers'] ?? [];
    $contact_details = $_POST['contact_details'] ?? [];

    // Encode data for database storage
    $flight_json = json_encode($flight);
    $passengers_json = json_encode($passengers);
    $contact_details_json = json_encode($contact_details);
    
    // Financial and reference data
    $total_price = $flight['price'];
    $currency = $flight['currency'];
    $booking_ref = 'SPT-' . strtoupper(substr(md5(uniqid()), 0, 8));

    // 2. Insert into database using PREPARED STATEMENTS
    $sql = "INSERT INTO bookings (user_id, booking_ref, flight_details, passenger_details, contact_details, total_price, price_currency, status, booking_type) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', 'flight')";
            
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("issssds", $user_id, $booking_ref, $flight_json, $passengers_json, $contact_details_json, $total_price, $currency);

    // 3. Execute and check for success
    if ($stmt->execute()) {
        
        // --- PREPARE DATA FOR WHATSAPP REDIRECT ---
        $first_itinerary = $flight['itineraries'][0] ?? null;
        $last_itinerary = end($flight['itineraries']) ?? null;
        $flight_route = ($first_itinerary && $last_itinerary) ? $first_itinerary['origin'] . ' to ' . $last_itinerary['destination'] : 'N/A';
        $flight_date = $first_itinerary ? 'Departing on: ' . date('D, M j, Y', strtotime($first_itinerary['departureDate'])) : 'N/A';
        $total_passengers = (int)($search_params['adults'] ?? 0) + (int)($search_params['children'] ?? 0) + (int)($search_params['infants'] ?? 0);
        $customer_name = $passengers['adults'][1]['firstName'] . ' ' . $passengers['adults'][1]['lastName'] ?? 'Valued Customer';

        // Cleanup session *after* getting all needed data
        unset($_SESSION['flight_search_results']);
        unset($_SESSION['flight_search_params']);
        
        // --- SET SESSION FLAGS FOR THANK YOU PAGE ---
        $_SESSION['submission_success'] = true;
        $_SESSION['booking_type'] = 'flight'; // <-- This makes it dynamic
        $_SESSION['success_context_name'] = $customer_name;
        $_SESSION['success_booking_ref'] = $booking_ref;
        $_SESSION['success_flight_route'] = $flight_route;
        $_SESSION['success_flight_date'] = $flight_date;
        $_SESSION['success_passenger_count'] = $total_passengers;
        
        // --- REDIRECT TO THANK YOU PAGE ---
        header("Location: thank-you.php");
        exit();
    } else {
        throw new Exception("Database insertion failed.");
    }

} catch (Exception $e) {
    error_log("Booking processing error: " . $e->getMessage());
    $_SESSION['checkout_error'] = "A server error occurred while processing your booking. Please try again.";
    header("Location: checkout.php?flight_id=" . urlencode($flight_id));
    exit();
}
?>