<?php
// ======================================================================
//  0. START SESSION & INITIALIZE
// ======================================================================
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
require_once 'db-config.php';

// --- CONFIGURATION ---
$admin_email = "rftravelsandtours@gmail.com";
$whatsapp_number = "923052394810";

// --- Check User Type ---
$is_agent = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');

// ======================================================================
//  2. HANDLE AGENT REFERRAL LINK
// ======================================================================
if (isset($_GET['ref']) && !empty($_GET['ref'])) {
    $_SESSION['referring_agent_id'] = (int)$_GET['ref'];
}

// ======================================================================
//  3. HANDLE BOOKING FORM SUBMISSION
// ======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['request_booking'])) {
    $customer_name = trim($_POST['customer_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone_number = trim($_POST['phone'] ?? '');
    $room_type = $_POST['room_type'] ?? 'Sharing';
    $pax = (int)($_POST['party_size'] ?? 1);
    
    // Get slug from the current file name to re-fetch package info
    $current_page_filename = basename($_SERVER['PHP_SELF']);
    $page_slug_for_redirect = str_replace('.php', '', $current_page_filename);
    
    // Fetch package details again to ensure data integrity
    $package_info_stmt = $conn->prepare("SELECT package_id, package_name FROM umrah_packages WHERE page_link = ?");
    $package_info_stmt->bind_param("s", $page_slug_for_redirect);
    $package_info_stmt->execute();
    $package_info_result = $package_info_stmt->get_result();
    $package_info = $package_info_result->fetch_assoc();
    $package_id_from_db = $package_info['package_id'] ?? 'Unknown';
    $package_name_from_db = $package_info['package_name'] ?? 'Unknown';
    $package_info_stmt->close();

    if (empty($customer_name) || empty($email) || empty($phone_number) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $_SESSION['form_message'] = "Please fill in all required fields with valid information.";
        $_SESSION['form_msg_type'] = "error";
        header("Location: " . basename(__FILE__) . "#booking-form-anchor");
        exit();
    }

    $user_id_to_save = isset($_SESSION['user_id']) ? (int)$_SESSION['user_id'] : null;
    $referred_by_agent_id = isset($_SESSION['referring_agent_id']) ? (int)$_SESSION['referring_agent_id'] : null;
    $status_to_save = 'Pending';

    $conn->begin_transaction();
    try {
        $sql1 = "INSERT INTO umrah_inquiries (package_id, package_name, customer_name, customer_phone, customer_email, room_type, pax, user_id, referred_by_agent_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt1 = $conn->prepare($sql1);
        $stmt1->bind_param("ssssssiiis", $package_id_from_db, $package_name_from_db, $customer_name, $phone_number, $email, $room_type, $pax, $user_id_to_save, $referred_by_agent_id, $status_to_save);
        $stmt1->execute();
        $stmt1->close();

        $notification_message = "Inquiry for " . htmlspecialchars($package_name_from_db) . " from " . htmlspecialchars($customer_name) . ".";
        if ($referred_by_agent_id) {
            $notification_message .= " (Referred by Agent ID: " . $referred_by_agent_id . ")";
        }
        $sql2 = "INSERT INTO notifications (type, message, link) VALUES ('New Umrah Inquiry', ?, 'admin/view-inquiries.php')";
        $stmt2 = $conn->prepare($sql2);
        $stmt2->bind_param("s", $notification_message);
        $stmt2->execute();
        $stmt2->close();

        $conn->commit();

        if (isset($_SESSION['referring_agent_id'])) {
            unset($_SESSION['referring_agent_id']);
        }
        
        $_SESSION['submission_success'] = true;
        $_SESSION['success_context_name'] = $customer_name;
        header("Location: thank-you.php");
        exit();

    } catch (mysqli_sql_exception $exception) {
        $conn->rollback();
        error_log("Booking Form DB Error: " . $exception->getMessage());
        $_SESSION['form_message'] = "A server error occurred. Our team has been notified. Please try again later.";
        $_SESSION['form_msg_type'] = "error";
        header("Location: " . basename(__FILE__) . "#booking-form-anchor");
        exit();
    }
}

// ======================================================================
//  4. FETCH PACKAGE DATA FOR DISPLAY
// ======================================================================
$package = null;
$error_message = '';

$current_page_filename = basename($_SERVER['PHP_SELF']);
$page_slug = str_replace('.php', '', $current_page_filename);

if (!empty($page_slug)) {
    try {
        $sql = "SELECT up.*, a.airline_name, a.logo_url FROM umrah_packages up LEFT JOIN airlines a ON up.airline_id = a.id WHERE up.page_link = ? AND up.is_active = 1";
        $stmt = $conn->prepare($sql);
        if ($stmt === false) { throw new Exception("DB prepare failed: " . $conn->error); }
        $stmt->bind_param("s", $page_slug);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $package = $result->fetch_assoc();
        } else {
            $error_message = "This Umrah package could not be found or is currently unavailable.";
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("Package Fetch Error: " . $e->getMessage());
        $error_message = "A server error occurred while retrieving package details.";
    }
} else {
    $error_message = "Invalid page request.";
}

if ($package === null) {
    http_response_code(404);
}

// Helper functions
function echo_as_list_items($text) {
    if (empty(trim($text))) return;
    $items = explode("\n", trim($text));
    echo "<ul>";
    foreach ($items as $item) {
        $trimmed_item = trim(str_replace(['- ','* '], '', $item));
        if (!empty($trimmed_item)) {
            echo "<li><i class='fas fa-check'></i>" . htmlspecialchars($trimmed_item) . "</li>\n";
        }
    }
    echo "</ul>";
}
function echo_as_paragraphs($text) {
    if (empty(trim($text))) return;
    $items = explode("\n", trim($text));
    foreach ($items as $item) {
        $trimmed_item = trim($item);
        if (!empty($trimmed_item)) {
            echo "<p>" . htmlspecialchars($trimmed_item) . "</p>\n";
        }
    }
}
function get_sidebar_price($package, $price_type, $is_agent) {
    $agent_price_key = $price_type . '_agent';
    if ($is_agent && isset($package[$agent_price_key]) && $package[$agent_price_key] > 0) {
        return $package[$agent_price_key];
    }
    return $package[$price_type] ?? 0;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $package ? htmlspecialchars($package['package_name']) : 'Package Not Found' ?> - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <style>
        .tour-detail-page { padding: 40px 0; background-color: #fff; }
        .container { max-width: 1700px; margin: 0 auto; padding: 0 2rem; }
        .tour-layout-grid { display: grid; grid-template-columns: minmax(0, 2fr) minmax(0, 1fr); gap: 50px; align-items: flex-start; }
        .tour-header { margin-bottom: 30px; }
        .tour-header h1 { font-size: 2.5rem; font-weight: 700; margin-bottom: 10px; color: var(--text-dark); }
        .tour-meta-info { display: flex; align-items: center; gap: 20px; color: var(--text-light); }
        .image-gallery { display: grid; grid-template-columns: 1fr; gap: 10px; }
        .gallery-main-image { aspect-ratio: 16 / 9; border-radius: 12px; overflow: hidden; }
        .gallery-main-image img { width: 100%; height: 100%; object-fit: cover; }
        
        .contact-card { display: flex; align-items: center; gap: 20px; background: var(--light-bg); padding: 25px; border-radius: 12px; margin: 40px 0; }
        .contact-logo { flex-shrink: 0; width: 60px; height: 60px; }
        .contact-logo img { width: 100%; height: 100%; object-fit: contain; }
        .contact-details { flex-grow: 1; }
        .contact-details span { color: var(--text-light); }
        .contact-details h3 { font-size: 1.3rem; margin: 5px 0; color: var(--text-dark); }
        .btn-message { background: #e7f0f7; color: var(--primary-dark); border: none; padding: 12px 20px; border-radius: 8px; font-weight: 600; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; }
        .btn-message i { margin-right: 8px; }

        .tour-section { padding: 30px 0; border-bottom: 1px solid var(--border-color); }
        .tour-section:last-of-type { border-bottom: none; }
        .tour-section h2 { font-size: 1.5rem; font-weight: 700; margin-bottom: 15px; color: var(--text-dark); }
        .tour-section p, .tour-section ul li { line-height: 1.7; color: var(--text-light); }
        .tour-section ul { list-style-type: none; padding-left: 0; }
        
        .flight-includes-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 30px; }
        .flight-details-box h3, .includes-box h3 { font-size: 1.2rem; font-weight: 600; margin-bottom: 15px; color: var(--text-dark); }
        .flight-item { display: flex; gap: 15px; align-items: flex-start; }
        .flight-item:not(:last-child) { margin-bottom: 20px; }
        .flight-item i { color: var(--primary-gold); font-size: 1.2rem; margin-top: 3px; }
        .flight-info-text strong { display: block; font-weight: 600; color: var(--text-dark); }
        .flight-info-text span { color: var(--text-light); font-size: 0.9rem; }
        .includes-box ul li { display: flex; align-items: center; gap: 10px; margin-bottom: 8px; color: var(--text-light); }
        .includes-box ul li i { color: #27ae60; }
        
        .tour-booking-sidebar { position: sticky; top: 120px; }
        .booking-card { border: 1px solid var(--border-color); border-radius: 16px; box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1); overflow: hidden; }
        .booking-card-header { padding: 20px; background-color: #f8f9fa; }
        .booking-card-header .price-label { font-size: 1rem; color: var(--text-light); }
        .booking-card-header .price-amount { font-size: 2.2rem; font-weight: 700; color: var(--text-dark); }
        .booking-card .cta-content { padding: 25px; }
        .btn-booking { width: 100%; background: var(--primary-gold); color: var(--white); border: none; padding: 15px; border-radius: 8px; font-size: 1.1rem; font-weight: 600; cursor: pointer; transition: filter 0.3s; text-decoration: none; text-align: center; display: inline-flex; align-items: center; justify-content: center; gap: 10px; }
        .btn-booking:hover { filter: brightness(110%); }
        .sidebar-price-grid { padding: 0 25px 25px; display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
        .price-grid-item { text-align: center; }
        .price-grid-item .label { font-size: 0.8rem; color: var(--text-light); text-transform: uppercase; }
        .price-grid-item .price { font-size: 1.2rem; font-weight: 600; color: var(--text-dark); }

        @media (max-width: 992px) {
            .tour-layout-grid { grid-template-columns: 1fr; }
            .tour-booking-sidebar { position: static; margin-top: 40px; }
        }
        @media (max-width: 768px) {
            .container { padding: 0 1rem; }
            .tour-header h1 { font-size: 2rem; }
            .contact-card { flex-direction: column; text-align: center; }
            .flight-includes-grid { grid-template-columns: 1fr; }
        }
    </style>
</head>

<body>
    <?php include 'header.php'; ?>

    <main class="tour-detail-page">
        <div class="container">
            <?php if ($package): ?>
                <div class="tour-layout-grid">
                    <div class="tour-main-content">
                        <div class="tour-header">
                            <h1><?= htmlspecialchars($package['package_name']) ?></h1>
                            <div class="tour-meta-info">
                                <span><i class="fas fa-clock"></i> <?= htmlspecialchars($package['days']) ?> Days</span>
                                <?php if(!empty($package['airline_name'])): ?>
                                    <span><i class="fas fa-plane"></i> <?= htmlspecialchars($package['airline_name']) ?></span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div class="image-gallery">
                            <div class="gallery-main-image">
                                <img src="<?= htmlspecialchars($package['main_image_link']) ?>" alt="<?= htmlspecialchars($package['package_name']) ?>">
                            </div>
                        </div>

                        <section class="contact-card">
                            <div class="contact-logo">
                                <?php
                                    $logo_src = 'images/logo.png';
                                    if ($is_agent && !empty($_SESSION['logo_path'])) {
                                        $agent_logo_path = 'admin/' . ltrim($_SESSION['logo_path'], '/');
                                        if (file_exists($agent_logo_path)) {
                                           $logo_src = $agent_logo_path;
                                        }
                                    }
                                ?>
                                <img src="<?= htmlspecialchars($logo_src) ?>" alt="Agency Logo">
                            </div>
                            <div class="contact-details">
                                <span>Have Questions?</span>
                                <h3>Talk to our Umrah Experts for assistance.</h3>
                            </div>
                            <div class="contact-message">
                                <?php $whatsapp_message = urlencode("Hello, I'm interested in the " . $package['package_name'] . " (" . $package['package_id'] . ")."); ?>
                                <a href="https://wa.me/<?= $whatsapp_number ?>?text=<?= $whatsapp_message ?>" target="_blank" class="btn-message">
                                    <i class="fa-brands fa-whatsapp"></i> Message Us
                                </a>
                            </div>
                        </section>

                        <?php if (!empty($package['overview'])): ?>
                        <section class="tour-section">
                            <h2>Package Overview</h2>
                            <?= echo_as_paragraphs($package['overview']) ?>
                        </section>
                        <?php endif; ?>

                        <section class="tour-section">
                            <div class="flight-includes-grid">
                                <div class="flight-details-box">
                                    <h3>Flight & Transport</h3>
                                    <?php if (!empty($package['outbound_flight_details'])): ?>
                                    <div class="flight-item">
                                        <i class="fas fa-plane-departure"></i>
                                        <div class="flight-info-text">
                                            <strong>Outbound</strong>
                                            <span><?= htmlspecialchars($package['outbound_flight_details']) ?></span>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    <?php if (!empty($package['inbound_flight_details'])): ?>
                                    <div class="flight-item">
                                        <i class="fas fa-plane-arrival"></i>
                                        <div class="flight-info-text">
                                            <strong>Inbound</strong>
                                            <span><?= htmlspecialchars($package['inbound_flight_details']) ?></span>
                                        </div>
                                    </div>
                                    <?php endif; ?>
                                    <div class="flight-item">
                                        <i class="fas fa-bus"></i>
                                        <div class="flight-info-text">
                                            <strong>Transportation</strong>
                                            <span><?= htmlspecialchars($package['transportation']) ?> Transport Included</span>
                                        </div>
                                    </div>
                                </div>
                                <div class="includes-box">
                                    <h3>What's Included</h3>
                                    <?= echo_as_list_items($package['whats_included']); ?>
                                </div>
                            </div>
                        </section>

                        <?php if (!empty($package['itinerary'])): ?>
                        <section class="tour-section">
                            <h2>Package Itinerary</h2>
                            <?= echo_as_list_items($package['itinerary']) ?>
                        </section>
                        <?php endif; ?>
                        
                        <?php if (!empty($package['whats_extra'])): ?>
                        <section class="tour-section">
                            <h2>What's Extra</h2>
                            <?= echo_as_list_items($package['whats_extra']) ?>
                        </section>
                        <?php endif; ?>
                    </div>

                    <aside class="tour-booking-sidebar">
                        <div class="booking-card">
                            <div class="booking-card-header">
                                <div class="price-label">Starts From (per person)</div>
                                <div class="price-amount">
                                    <?php
                                    $sidebar_prices = [
                                        get_sidebar_price($package, 'price_quint', $is_agent),
                                        get_sidebar_price($package, 'price_quad', $is_agent),
                                        get_sidebar_price($package, 'price_triple', $is_agent),
                                        get_sidebar_price($package, 'price_double', $is_agent)
                                    ];
                                    $valid_prices = array_filter($sidebar_prices, function($price){ return $price > 0; });
                                    $starting_price = !empty($valid_prices) ? min($valid_prices) : 0;
                                    echo $starting_price > 0 ? 'PKR ' . number_format($starting_price) : 'On Request';
                                    ?>
                                </div>
                            </div>
                            <div class="sidebar-price-grid">
                                <?php if($price = get_sidebar_price($package, 'price_double', $is_agent)): ?><div class="price-grid-item"><div class="label">Double</div><div class="price"><?= number_format($price) ?></div></div><?php endif; ?>
                                <?php if($price = get_sidebar_price($package, 'price_triple', $is_agent)): ?><div class="price-grid-item"><div class="label">Triple</div><div class="price"><?= number_format($price) ?></div></div><?php endif; ?>
                                <?php if($price = get_sidebar_price($package, 'price_quad', $is_agent)): ?><div class="price-grid-item"><div class="label">Quad</div><div class="price"><?= number_format($price) ?></div></div><?php endif; ?>
                                <?php if($price = get_sidebar_price($package, 'price_quint', $is_agent)): ?><div class="price-grid-item"><div class="label">Sharing</div><div class="price"><?= number_format($price) ?></div></div><?php endif; ?>
                            </div>
                            <div class="cta-content" id="booking-form-anchor">
                                <form action="<?= basename(__FILE__) . '?id=' . urlencode($package_code_to_display) ?>" method="POST">
                                    <input type="hidden" name="request_booking" value="1">
                                    <button type="submit" class="btn-booking">
                                        <i class="fa-solid fa-paper-plane"></i> Send Booking Request
                                    </button>
                                </form>
                            </div>
                        </div>
                    </aside>
                </div>
            <?php else: ?>
                <div class="error-container" style="text-align: center; padding: 5rem 1rem;">
                    <h1>Package Not Found</h1>
                    <p><?= htmlspecialchars($error_message) ?></p>
                    <p><a href="index.php" class="btn-primary">Return to Homepage</a></p>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <?php include 'footer.php'; ?>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const menuToggle = document.querySelector('.menu-toggle');
            const mobileSidebar = document.querySelector('.mobile-sidebar');
            const sidebarOverlay = document.querySelector('.sidebar-overlay');
            if (menuToggle && mobileSidebar && sidebarOverlay) {
                menuToggle.addEventListener('click', () => {
                    mobileSidebar.classList.add('active');
                    sidebarOverlay.classList.add('active');
                });
                const closeSidebar = () => {
                    mobileSidebar.classList.remove('active');
                    sidebarOverlay.classList.remove('active');
                };
                document.querySelector('.sidebar-close-btn').addEventListener('click', closeSidebar);
                sidebarOverlay.addEventListener('click', closeSidebar);
            }
        });
    </script>
</body>
</html>