<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- HANDLE FORM SUBMISSION ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {
    
    // --- ADD PAYMENT ACTION ---
    if ($_POST['action'] === 'add_payment') {
        $payment_type = $_POST['payment_type']; // 'credit' or 'debit'
        $amount = (float)$_POST['payment_amount'];
        $user_id = !empty($_POST['link_to_user']) ? (int)$_POST['link_to_user'] : null;
        $vendor_id = !empty($_POST['link_to_vendor']) ? (int)$_POST['link_to_vendor'] : null;
        $invoice_ref = trim($_POST['invoice_reference'] ?? '');
        
        // ** THE FIX IS HERE **
        // Determine invoice_id and invoice_type, ensuring NULL if not set
        $invoice_id = null;
        $invoice_type = null;
        if (!empty($_POST['link_to_invoice'])) {
            list($type, $id) = explode('-', $_POST['link_to_invoice']);
            if (is_numeric($id) && $id > 0) {
                $invoice_id = (int)$id;
                $invoice_type = ($type === 'pkg') ? 'package' : 'ticket';
            }
        }

        $credit_amount = ($payment_type === 'credit') ? $amount : 0.00;
        $debit_amount = ($payment_type === 'debit') ? $amount : 0.00;

        $sql = "INSERT INTO payments (invoice_id, invoice_type, user_id, vendor_id, payment_date, payment_amount, credit_amount, debit_amount, payment_method, invoice_reference, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param("isiisddssss", $invoice_id, $invoice_type, $user_id, $vendor_id, $_POST['payment_date'], $amount, $credit_amount, $debit_amount, $_POST['payment_method'], $invoice_ref, $_POST['notes']);
        
        if ($stmt->execute()) {
            $_SESSION['success_message'] = "Payment of " . number_format($amount) . " recorded successfully.";
        } else {
            $_SESSION['error_message'] = "Database Error: " . $stmt->error;
        }
        header("Location: add-payment.php");
        exit();
    }
}

// --- FETCH DATA FOR FORM DROPDOWNS ---
$users = $conn->query("SELECT id, name, user_type FROM users ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$recent_payments = $conn->query("
    SELECT p.*, u.name as user_name, v.name as vendor_name 
    FROM payments p 
    LEFT JOIN users u ON p.user_id = u.id 
    LEFT JOIN vendors v ON p.vendor_id = v.id
    ORDER BY p.payment_date DESC, p.id DESC LIMIT 10
")->fetch_all(MYSQLI_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Record a Payment</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style> .main-content { min-width: 0; } .content-body { padding: 1.5rem; } </style>
</head>
<body>
<div class="dashboard-wrapper">
    <?php include 'sidebar.php'; ?>
    <div class="main-content">
        <header class="main-header">
            <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
            <div class="user-info"><span>Record a Payment</span></div>
        </header>
        <main class="content-body">
            <h1 class="page-title">Record a Payment</h1>

            <?php if(isset($_SESSION['success_message'])): ?><div class="alert alert-success"><?= e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div><?php endif; ?>
            <?php if(isset($_SESSION['error_message'])): ?><div class="alert alert-danger"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div><?php endif; ?>

            <div class="card">
                <div class="card-body">
                    <form action="add-payment.php" method="POST">
                        <input type="hidden" name="action" value="add_payment">
                        <div class="row g-3">
                            <div class="col-md-3"><label class="form-label">Payment Type*</label>
                                <select name="payment_type" class="form-select" required>
                                    <option value="credit">Payment Received (Credit)</option>
                                    <option value="debit">Payment Made / Expense (Debit)</option>
                                </select>
                            </div>
                            <div class="col-md-3"><label class="form-label">Amount (PKR)*</label><input type="number" name="payment_amount" class="form-control" step="0.01" required></div>
                            <div class="col-md-3"><label class="form-label">Payment Date*</label><input type="date" name="payment_date" class="form-control" value="<?= date('Y-m-d') ?>" required></div>
                            <div class="col-md-3"><label class="form-label">Payment Method*</label>
                                <select name="payment_method" class="form-select" required>
                                    <option>Bank Transfer</option><option>Cash</option><option>Card</option><option>Cheque</option><option>Other</option>
                                </select>
                            </div>

                            <hr class="my-4">
                            <h5 class="mb-3">Link Payment To (Optional)</h5>

                            <div class="col-md-6"><label class="form-label">Link to User (Customer/Agent)</label>
                                <select name="link_to_user" id="user-filter" class="form-select">
                                    <option value="">-- No User --</option>
                                    <?php foreach($users as $user): ?><option value="<?= $user['id'] ?>"><?= e($user['name']) ?> [<?= e(ucfirst($user['user_type'])) ?>]</option><?php endforeach; ?>
                                </select>
                            </div>
                             <div class="col-md-6"><label class="form-label">Link to Vendor</label>
                                <select name="link_to_vendor" class="form-select">
                                    <option value="">-- No Vendor --</option>
                                    <?php foreach($vendors as $vendor): ?><option value="<?= $vendor['id'] ?>"><?= e($vendor['name']) ?></option><?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6"><label class="form-label">Link to Specific Invoice</label>
                                <select name="link_to_invoice" id="invoice-dropdown" class="form-select">
                                    <option value="">-- Do not link to an invoice --</option>
                                    <!-- Options will be loaded by JavaScript -->
                                </select>
                            </div>
                             <div class="col-md-6"><label class="form-label">Manual Reference</label><input type="text" name="invoice_reference" class="form-control" placeholder="e.g., TKT-1234, cash expense..."></div>
                            <div class="col-12"><label class="form-label">Notes</label><textarea name="notes" class="form-control" rows="3"></textarea></div>

                            <div class="col-12 text-center mt-4">
                                <button type="submit" class="btn btn-primary btn-lg">Record Payment</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
            
            <div class="card mt-4">
                <div class="card-header"><h2 class="h5 mb-0">Recent Transactions</h2></div>
                <div class="table-responsive">
                    <table class="table table-hover align-middle mb-0">
                        <thead class="table-light"><tr><th>Date</th><th>Linked To</th><th>Reference</th><th>Credit</th><th>Debit</th><th>Method</th></tr></thead>
                        <tbody>
                            <?php foreach($recent_payments as $p): ?>
                            <tr>
                                <td><?= date('d M Y', strtotime($p['payment_date'])) ?></td>
                                <td>
                                    <?php if($p['user_name']) echo e($p['user_name']); elseif($p['vendor_name']) echo e($p['vendor_name']); else echo 'N/A'; ?>
                                </td>
                                <td><?= e($p['invoice_reference']) ?></td>
                                <td class="text-success fw-bold"><?= $p['credit_amount'] > 0 ? number_format($p['credit_amount'], 2) : '-' ?></td>
                                <td class="text-danger fw-bold"><?= $p['debit_amount'] > 0 ? number_format($p['debit_amount'], 2) : '-' ?></td>
                                <td><?= e($p['payment_method']) ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </main>
    </div>
</div>
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script>
$(document).ready(function() {
    $('#user-filter').change(function() {
        const userId = $(this).val();
        const invoiceDropdown = $('#invoice-dropdown');
        invoiceDropdown.html('<option value="">Loading...</option>');

        if (userId) {
            $.ajax({
                url: 'ajax_get_invoices.php', // We will create this new file
                type: 'GET',
                data: { user_id: userId },
                dataType: 'json',
                success: function(response) {
                    invoiceDropdown.html('<option value="">-- Do not link to an invoice --</option>');
                    if (response.package_invoices) {
                        response.package_invoices.forEach(inv => {
                            invoiceDropdown.append(`<option value="pkg-${inv.id}">Package Invoice #${inv.invoice_number || inv.id}</option>`);
                        });
                    }
                    if (response.ticket_invoices) {
                        response.ticket_invoices.forEach(inv => {
                            invoiceDropdown.append(`<option value="tkt-${inv.id}">Ticket Invoice #${inv.invoice_number || inv.id}</option>`);
                        });
                    }
                },
                error: function() {
                    invoiceDropdown.html('<option value="">Could not load invoices</option>');
                }
            });
        } else {
            invoiceDropdown.html('<option value="">-- Select a user to see invoices --</option>');
        }
    });
});
</script>
</body>
</html>