<?php
// db-config.php should handle session_start()
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// Helper function for secure output
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// Fetch all agents for the dropdown
$agents_result = $conn->query("SELECT id, name FROM users WHERE user_type = 'agent' ORDER BY name ASC");

// Initialize variables
$selected_agent_id = null;
$start_date = null;
$end_date = null;
$incentive_data = [];
$summary = ['total_bookings' => 0, 'total_incentive' => 0];

// Handle form submission
if ($_SERVER["REQUEST_METHOD"] == "GET" && isset($_GET['filter'])) {
    $selected_agent_id = !empty($_GET['agent_id']) ? (int)$_GET['agent_id'] : null;
    $start_date = !empty($_GET['start_date']) ? $_GET['start_date'] : null;
    $end_date = !empty($_GET['end_date']) ? $_GET['end_date'] : null;

    if ($selected_agent_id && $start_date && $end_date) {
        // Prepare the SQL query
        // *** FIX: Changed ui.inquiry_date to ui.created_at ***
        $sql = "SELECT
                    ui.id as inquiry_id,
                    ui.customer_name,
                    ui.package_name,
                    ui.created_at as inquiry_date,
                    up.agent_incentive
                FROM
                    umrah_inquiries ui
                JOIN
                    umrah_packages up ON ui.package_id = up.package_id
                WHERE
                    ui.referred_by_agent_id = ?
                    AND DATE(ui.created_at) BETWEEN ? AND ?
                    AND up.agent_incentive > 0
                ORDER BY
                    ui.created_at DESC";
        
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("iss", $selected_agent_id, $start_date, $end_date);
            $stmt->execute();
            $result = $stmt->get_result();
            while ($row = $result->fetch_assoc()) {
                $incentive_data[] = $row;
                $summary['total_bookings']++;
                $summary['total_incentive'] += (float)$row['agent_incentive'];
            }
            $stmt->close();
        } else {
            $error_message = "Database query failed to prepare.";
        }
    } else {
        $error_message = "Please select an agent and both a start and end date.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agent Incentives Report</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <!-- Select2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .filter-form .form-grid {
            display: grid;
            grid-template-columns: 2fr 1fr 1fr auto;
            gap: 1rem;
            align-items: end;
        }
        .report-summary {
            margin-top: 2rem;
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 1.5rem;
            display: flex;
            justify-content: space-around;
            text-align: center;
        }
        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
        }
        .summary-item .value {
            font-size: 1.8em;
            font-weight: 600;
            color: #0d2d4c;
        }
        @media (max-width: 992px) {
            .filter-form .form-grid {
                grid-template-columns: 1fr;
            }
            .report-summary {
                flex-direction: column;
                gap: 1rem;
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo e(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body">
                <div class="page-header">
                    <h1 class="page-title">Agent Incentives Report</h1>
                </div>

                <?php if (isset($error_message)): ?><div class="notice error"><?= e($error_message) ?></div><?php endif; ?>

                <div class="content-card">
                    <h2 class="card-title">Filter Report</h2>
                    <form action="agent-incentives.php" method="GET" class="filter-form">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="agent_id">Select Agent</label>
                                <select name="agent_id" id="agent_id" class="select2-field" required>
                                    <option value="">-- Choose an Agent --</option>
                                    <?php while($agent = $agents_result->fetch_assoc()): ?>
                                        <option value="<?= e($agent['id']) ?>" <?= ($selected_agent_id == $agent['id']) ? 'selected' : '' ?>>
                                            <?= e($agent['name']) ?>
                                        </option>
                                    <?php endwhile; ?>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="start_date">Start Date</label>
                                <input type="date" name="start_date" id="start_date" value="<?= e($start_date) ?>" required>
                            </div>
                            <div class="form-group">
                                <label for="end_date">End Date</label>
                                <input type="date" name="end_date" id="end_date" value="<?= e($end_date) ?>" required>
                            </div>
                            <div class="form-actions">
                                <button type="submit" name="filter" value="1" class="btn btn-primary"><i class="fas fa-filter"></i> Filter</button>
                            </div>
                        </div>
                    </form>
                </div>

                <?php if (!empty($incentive_data)): ?>
                <div class="content-card">
                    <h2 class="card-title">Incentive Details</h2>
                    <div class="table-responsive">
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Inquiry ID</th>
                                    <th>Customer Name</th>
                                    <th>Package Name</th>
                                    <th>Booking Date</th>
                                    <th>Incentive (SAR)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($incentive_data as $row): ?>
                                <tr>
                                    <td><?= e($row['inquiry_id']) ?></td>
                                    <td><?= e($row['customer_name']) ?></td>
                                    <td><?= e($row['package_name']) ?></td>
                                    <td><?= e(date('d M, Y H:i', strtotime($row['inquiry_date']))) ?></td>
                                    <td><?= e(number_format($row['agent_incentive'], 2)) ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="report-summary">
                        <div class="summary-item">
                            <div class="label">Total Referred Bookings</div>
                            <div class="value"><?= e($summary['total_bookings']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Total Incentive Payable</div>
                            <div class="value"><?= e(number_format($summary['total_incentive'], 2)) ?> SAR</div>
                        </div>
                    </div>
                </div>
                <?php elseif (isset($_GET['filter'])): ?>
                <div class="content-card">
                    <p class="empty-state">No incentive records found for the selected agent and date range.</p>
                </div>
                <?php endif; ?>

            </main>
        </div>
    </div>

    <!-- JQuery and Select2 JS -->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize Select2
            $('.select2-field').select2({
                width: '100%'
            });
            // Standard sidebar toggle script
            const menuToggle = document.getElementById('menu-toggle');
            if (menuToggle) {
                menuToggle.addEventListener('click', function() {
                    document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
                });
            }
        });
    </script>
</body>
</html>