<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- GENERATE NEW INVOICE NUMBER ---
$result = $conn->query("SELECT MAX(CAST(invoice_number AS UNSIGNED)) as max_inv FROM invoices");
$max_inv_row = $result->fetch_assoc();
$last_inv_num = $max_inv_row['max_inv'] ?? 0;
$start_inv_num = 92157201;
$new_invoice_number = ($last_inv_num >= $start_inv_num) ? $last_inv_num + 1 : $start_inv_num;


// --- FETCH DATA FOR DROPDOWNS ---
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendor_options_html = '<option value="0">-- Optional --</option>';
foreach ($vendors_list as $vendor) {
    $vendor_options_html .= '<option value="' . $vendor['id'] . '">' . e($vendor['name']) . '</option>';
}
$makkah_hotels = []; $madinah_hotels = [];
$hotel_result = $conn->query("SELECT hotel_name, city FROM rate_sheets ORDER BY hotel_name ASC");
if ($hotel_result) {
    while ($row = $hotel_result->fetch_assoc()) {
        if (strtolower($row['city']) === 'makkah') $makkah_hotels[] = $row['hotel_name'];
        elseif (strtolower($row['city']) === 'madinah') $madinah_hotels[] = $row['hotel_name'];
    }
}

// --- HANDLE FORM SUBMISSION ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    try {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $user_id_to_save = ($user_id > 0) ? $user_id : null;
        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $vendor_id_to_save = ($vendor_id > 0) ? $vendor_id : null;
        $guest_name = trim($_POST['guest_name']);
        $invoice_number = trim($_POST['invoice_number']);
        $issue_date = $_POST['issue_date'];
        $status_to_save = in_array($_POST['status'], ['Tentative', 'Approved']) ? $_POST['status'] : 'Tentative';
        $exchange_rate = (float)($_POST['exchange_rate'] ?? 0);
        $notes = trim($_POST['notes'] ?? '');
        $discount_pkr = (float)($_POST['discount_pkr'] ?? 0);
        $pilgrims_vendor_id_to_save = !empty($_POST['pilgrims_vendor_id']) ? (int)$_POST['pilgrims_vendor_id'] : null;
        $transport_vendor_id_to_save = !empty($_POST['transport_vendor_id']) ? (int)$_POST['transport_vendor_id'] : null;
        $tickets_vendor_id_to_save = !empty($_POST['tickets_vendor_id']) ? (int)$_POST['tickets_vendor_id'] : null;

        if (empty($guest_name) || empty($issue_date) || !DateTime::createFromFormat('Y-m-d', $issue_date)) throw new Exception("Guest Name and a valid Issue Date are required.");
        if ($exchange_rate <= 0) throw new Exception("A valid Exchange Rate is required.");

        // Server-side recalculation to ensure data integrity
        $total_sar_calc = 0;
        $total_sar_cost_calc = 0;
        $total_ticket_pkr_calc = 0;
        $total_ticket_pkr_cost_calc = 0;

        // Pilgrims
        if (!empty($_POST['pilgrim_passport_no']) && is_array($_POST['pilgrim_passport_no'])) {
            foreach ($_POST['pilgrim_visa_price_sar'] as $key => $val) {
                $total_sar_calc += (float)($_POST['pilgrim_visa_price_sar'][$key] ?? 0);
                $total_sar_cost_calc += (float)($_POST['pilgrim_visa_price_sar_cost'][$key] ?? 0);
            }
        }
        // Hotels
        if (!empty($_POST['hotel_city']) && is_array($_POST['hotel_city'])) {
            foreach ($_POST['hotel_city'] as $key => $val) {
                $qty = (float)($_POST['hotel_qty'][$key] ?? 1);
                $rate = (float)($_POST['hotel_rate'][$key] ?? 0);
                $nights = (float)($_POST['hotel_nights'][$key] ?? 1);
                $rate_cost = (float)($_POST['hotel_rate_cost'][$key] ?? 0);
                $total_sar_calc += $qty * $rate * $nights;
                $total_sar_cost_calc += $qty * $rate_cost * $nights;
            }
        }
        // Transport
        if (!empty($_POST['transport_vehicle']) && is_array($_POST['transport_vehicle'])) {
            foreach ($_POST['transport_vehicle'] as $key => $val) {
                $qty = (float)($_POST['transport_qty'][$key] ?? 1);
                $rate = (float)($_POST['transport_rate'][$key] ?? 0);
                $rate_cost = (float)($_POST['transport_rate_cost'][$key] ?? 0);
                $total_sar_calc += $qty * $rate;
                $total_sar_cost_calc += $qty * $rate_cost;
            }
        }
        // Services
        if (!empty($_POST['service_name']) && is_array($_POST['service_name'])) {
            foreach ($_POST['service_name'] as $key => $val) {
                $adult_qty = (float)($_POST['service_adult_qty'][$key] ?? 0);
                $adult_rate = (float)($_POST['service_adult_rate'][$key] ?? 0);
                $child_qty = (float)($_POST['service_child_qty'][$key] ?? 0);
                $child_rate = (float)($_POST['service_child_rate'][$key] ?? 0);
                $infant_qty = (float)($_POST['service_infant_qty'][$key] ?? 0);
                $infant_rate = (float)($_POST['service_infant_rate'][$key] ?? 0);
                $total_cost = (float)($_POST['service_total_cost'][$key] ?? 0);
                $total_sar_calc += ($adult_qty * $adult_rate) + ($child_qty * $child_rate) + ($infant_qty * $infant_rate);
                $total_sar_cost_calc += $total_cost;
            }
        }
        // Tickets
        if (!empty($_POST['ticket_airline']) && is_array($_POST['ticket_airline'])) {
            foreach ($_POST['ticket_airline'] as $key => $val) {
                $adult_qty = (float)($_POST['ticket_adult_qty'][$key] ?? 0);
                $adult_rate = (float)($_POST['ticket_adult_rate'][$key] ?? 0);
                $child_qty = (float)($_POST['ticket_child_qty'][$key] ?? 0);
                $child_rate = (float)($_POST['ticket_child_rate'][$key] ?? 0);
                $infant_qty = (float)($_POST['ticket_infant_qty'][$key] ?? 0);
                $infant_rate = (float)($_POST['ticket_infant_rate'][$key] ?? 0);
                $total_cost = (float)($_POST['ticket_total_cost'][$key] ?? 0);
                $total_ticket_pkr_calc += ($adult_qty * $adult_rate) + ($child_qty * $child_rate) + ($infant_qty * $infant_rate);
                $total_ticket_pkr_cost_calc += $total_cost;
            }
        }

        $total_pkr_without_ticket_calc = $total_sar_calc * $exchange_rate;
        $grand_total_pkr_calc = $total_pkr_without_ticket_calc + $total_ticket_pkr_calc - $discount_pkr;
        $total_pkr_cost_without_ticket_calc = $total_sar_cost_calc * $exchange_rate;
        $grand_total_pkr_cost_calc = $total_pkr_cost_without_ticket_calc + $total_ticket_pkr_cost_calc;

        $sql_invoice = "INSERT INTO invoices (user_id, vendor_id, guest_name, invoice_number, status, issue_date, exchange_rate, total_sar, total_pkr_without_ticket, total_ticket_pkr, discount_pkr, grand_total_pkr, grand_total_pkr_cost, notes, pilgrims_vendor_id, transport_vendor_id, tickets_vendor_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt_invoice = $conn->prepare($sql_invoice);
        if (!$stmt_invoice) throw new Exception("Prepare statement failed: " . $conn->error);
        
        $stmt_invoice->bind_param("iissssdddddddsiii", $user_id_to_save, $vendor_id_to_save, $guest_name, $invoice_number, $status_to_save, $issue_date, $exchange_rate, $total_sar_calc, $total_pkr_without_ticket_calc, $total_ticket_pkr_calc, $discount_pkr, $grand_total_pkr_calc, $grand_total_pkr_cost_calc, $notes, $pilgrims_vendor_id_to_save, $transport_vendor_id_to_save, $tickets_vendor_id_to_save);
        $stmt_invoice->execute();
        $invoice_id = $conn->insert_id;
        if ($invoice_id <= 0) throw new Exception("Failed to create main invoice record.");

        // Insert Pilgrims
        if (!empty($_POST['pilgrim_passport_no']) && is_array($_POST['pilgrim_passport_no'])) {
            $sql_pilgrim = "INSERT INTO invoice_pilgrims (invoice_id, passport_no, passenger_details, dob, dox, pax, bed, visa_price_sar, visa_price_sar_cost) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_pilgrim = $conn->prepare($sql_pilgrim);
            foreach ($_POST['pilgrim_passport_no'] as $key => $val) {
                if (empty(trim($val)) && empty(trim($_POST['pilgrim_passenger_details'][$key]))) continue;
                $stmt_pilgrim->bind_param("issssssdd", $invoice_id, $_POST['pilgrim_passport_no'][$key], $_POST['pilgrim_passenger_details'][$key], $_POST['pilgrim_dob'][$key], $_POST['pilgrim_dox'][$key], $_POST['pilgrim_pax'][$key], $_POST['pilgrim_bed'][$key], $_POST['pilgrim_visa_price_sar'][$key], $_POST['pilgrim_visa_price_sar_cost'][$key]);
                $stmt_pilgrim->execute();
            }
            $stmt_pilgrim->close();
        }

        // Insert Hotels
        if (!empty($_POST['hotel_city']) && is_array($_POST['hotel_city'])) {
            $sql_hotel = "INSERT INTO invoice_hotels (invoice_id, city, hotel_name, vendor_id, check_in, nights, check_out, room_type, meal_plan, rooms, rate_sar, total_sar, rate_sar_cost, helpline) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_hotel = $conn->prepare($sql_hotel);
            foreach ($_POST['hotel_city'] as $key => $city) {
                if (empty($_POST['hotel_name'][$key])) continue;
                $vendor = (int)$_POST['hotel_vendor_id'][$key] > 0 ? (int)$_POST['hotel_vendor_id'][$key] : null;
                $nights = (int)$_POST['hotel_nights'][$key];
                $rate_sar = (float)$_POST['hotel_rate'][$key];
                $rooms = (int)$_POST['hotel_qty'][$key];
                $total_sar = $nights * $rate_sar * $rooms;
                $helpline = trim($_POST['hotel_helpline'][$key] ?? '');
                $stmt_hotel->bind_param("issisissssidds", $invoice_id, $city, $_POST['hotel_name'][$key], $vendor, $_POST['hotel_check_in'][$key], $nights, $_POST['hotel_check_out'][$key], $_POST['hotel_type'][$key], $_POST['hotel_meal_plan'][$key], $rooms, $rate_sar, $total_sar, $_POST['hotel_rate_cost'][$key], $helpline);
                $stmt_hotel->execute();
            }
            $stmt_hotel->close();
        }
        
        // Insert Transport
        if (!empty($_POST['transport_vehicle']) && is_array($_POST['transport_vehicle'])) {
            $sql_transport = "INSERT INTO invoice_transports (invoice_id, vehicle_type, route, rate, qty, adult_qty, child_qty, total_amount, rate_cost) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_transport = $conn->prepare($sql_transport);
            foreach($_POST['transport_vehicle'] as $key => $vehicle) {
                if(empty(trim($vehicle))) continue;
                $rate = (float)$_POST['transport_rate'][$key];
                $qty = (int)$_POST['transport_qty'][$key];
                $total = $rate * $qty;
                $stmt_transport->bind_param("issdiiidd", $invoice_id, $vehicle, $_POST['transport_route'][$key], $rate, $qty, $_POST['transport_adult_qty'][$key], $_POST['transport_child_qty'][$key], $total, $_POST['transport_rate_cost'][$key]);
                $stmt_transport->execute();
            }
            $stmt_transport->close();
        }

        // Insert Other Services
        if (!empty($_POST['service_name']) && is_array($_POST['service_name'])) {
            $sql_service = "INSERT INTO invoice_other_services (invoice_id, service_name, vendor_id, adult_rate, adult_qty, child_rate, child_qty, infant_rate, infant_qty, total_amount, total_cost) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_service = $conn->prepare($sql_service);
            foreach($_POST['service_name'] as $key => $service) {
                if(empty(trim($service))) continue;
                $vendor = (int)$_POST['service_vendor_id'][$key] > 0 ? (int)$_POST['service_vendor_id'][$key] : null;
                $adult_rate = (float)$_POST['service_adult_rate'][$key];
                $adult_qty = (int)$_POST['service_adult_qty'][$key];
                $child_rate = (float)$_POST['service_child_rate'][$key];
                $child_qty = (int)$_POST['service_child_qty'][$key];
                $infant_rate = (float)$_POST['service_infant_rate'][$key];
                $infant_qty = (int)$_POST['service_infant_qty'][$key];
                $total = ($adult_rate * $adult_qty) + ($child_rate * $child_qty) + ($infant_rate * $infant_qty);
                $stmt_service->bind_param("isididididd", $invoice_id, $service, $vendor, $adult_rate, $adult_qty, $child_rate, $child_qty, $infant_rate, $infant_qty, $total, $_POST['service_total_cost'][$key]);
                $stmt_service->execute();
            }
            $stmt_service->close();
        }

        // Insert Airline Tickets
        if (!empty($_POST['ticket_airline']) && is_array($_POST['ticket_airline'])) {
            $sql_ticket = "INSERT INTO invoice_airline_tickets (invoice_id, airline, sector, adult_qty, adult_rate, child_qty, child_rate, infant_qty, infant_rate, total_amount, total_cost) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_ticket = $conn->prepare($sql_ticket);
            foreach($_POST['ticket_airline'] as $key => $airline) {
                if(empty(trim($airline))) continue;
                $adult_rate = (float)$_POST['ticket_adult_rate'][$key];
                $adult_qty = (int)$_POST['ticket_adult_qty'][$key];
                $child_rate = (float)$_POST['ticket_child_rate'][$key];
                $child_qty = (int)$_POST['ticket_child_qty'][$key];
                $infant_rate = (float)$_POST['ticket_infant_rate'][$key];
                $infant_qty = (int)$_POST['ticket_infant_qty'][$key];
                $total = ($adult_rate * $adult_qty) + ($child_rate * $child_qty) + ($infant_rate * $infant_qty);
                $stmt_ticket->bind_param("issidididdd", $invoice_id, $airline, $_POST['ticket_sector'][$key], $adult_qty, $adult_rate, $child_qty, $child_rate, $infant_qty, $infant_rate, $total, $_POST['ticket_total_cost'][$key]);
                $stmt_ticket->execute();
            }
            $stmt_ticket->close();
        }

        $conn->commit();
        $_SESSION['success_message'] = "Invoice #" . $invoice_number . " created successfully!";
        header("Location: manage-invoices.php");
        exit();

    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error Creating Invoice: " . $e->getMessage();
        header("Location: create-invoice.php");
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Invoice</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .select2-container .select2-selection--single { height: calc(1.5em + .75rem + 2px); padding: .375rem .75rem; border: 1px solid #ced4da; }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 1.5; padding-left: 0; }
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: calc(1.5em + .75rem); }
        .dynamic-table th, .dynamic-table td { vertical-align: middle; text-align: center; }
        .summary-grid { display: grid; grid-template-columns: 1fr auto; gap: 5px 20px; align-items: center; }
        .summary-label { font-weight: bold; text-align: right; }
        .grand-total { font-size: 1.1em; font-weight: bold; }
        .profit-label { color: var(--success-color); }
        .cost-label { color: var(--danger-color); }
        
        /* --- FINAL FIX FOR TABLE SCROLLBAR & FIELD WIDTHS --- */
        .main-content {
            min-width: 0;
        }

        .table-responsive > .table {
            min-width: 1500px; /* Generous width to give all columns breathing room */
        }

        .dynamic-table th {
            white-space: nowrap;
        }

        .dynamic-table .form-control,
        .dynamic-table .form-select {
            width: 100%; /* Make input fill the cell */
        }
        
        /* Set min-width on the table cells (td/th) for precise column width control */
        #pilgrims-tbody th:nth-child(1), #pilgrims-tbody td:nth-child(1) { min-width: 160px; } /* Passport */
        #pilgrims-tbody th:nth-child(2), #pilgrims-tbody td:nth-child(2) { min-width: 220px; } /* Passenger */
        #pilgrims-tbody th:nth-child(3), #pilgrims-tbody td:nth-child(3) { min-width: 150px; } /* DOB */
        #pilgrims-tbody th:nth-child(4), #pilgrims-tbody td:nth-child(4) { min-width: 150px; } /* DOX */
        #pilgrims-tbody th:nth-child(5), #pilgrims-tbody td:nth-child(5) { min-width: 110px; } /* Pax */
        #pilgrims-tbody th:nth-child(6), #pilgrims-tbody td:nth-child(6) { min-width: 110px; } /* Bed */
        
        #hotel-tbody th:nth-child(1), #hotel-tbody td:nth-child(1) { min-width: 150px; } /* City */
        #hotel-tbody th:nth-child(2), #hotel-tbody td:nth-child(2) { min-width: 250px; } /* Hotel (Select2) */
        #hotel-tbody th:nth-child(3), #hotel-tbody td:nth-child(3) { min-width: 160px; } /* Vendor */
        #hotel-tbody th:nth-child(4), #hotel-tbody td:nth-child(4) { min-width: 150px; } /* Helpline */
        #hotel-tbody th:nth-child(5), #hotel-tbody td:nth-child(5) { min-width: 150px; } /* Check-in */
        #hotel-tbody th:nth-child(6), #hotel-tbody td:nth-child(6) { min-width: 90px;  } /* Nights */
        #hotel-tbody th:nth-child(7), #hotel-tbody td:nth-child(7) { min-width: 150px; } /* Check-out */
        #hotel-tbody th:nth-child(8), #hotel-tbody td:nth-child(8) { min-width: 110px; } /* Type */
        #hotel-tbody th:nth-child(9), #hotel-tbody td:nth-child(9) { min-width: 110px; } /* Meal */
        #hotel-tbody th:nth-child(10), #hotel-tbody td:nth-child(10){ min-width: 90px;  } /* Qty */
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Create New Invoice</span></div>
            </header>
            <main class="content-body">
                <div class="content-card">
                    <div class="card-body">
                        <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="alert alert-danger"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                        <?php endif; ?>

                        <form action="create-invoice.php" method="POST" id="invoice-main-form">
                            <fieldset class="mb-4"><legend class="h5 fw-bold"><i class="fas fa-file-invoice"></i> Main Details</legend>
                                <div class="row g-3">
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="guest_name" class="form-label">Guest Name*</label><input type="text" id="guest_name" name="guest_name" class="form-control" required></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="invoice_number" class="form-label">Invoice No</label><input type="text" id="invoice_number" name="invoice_number" class="form-control" value="<?= $new_invoice_number ?>" readonly></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="issue_date" class="form-label">Dated*</label><input type="date" id="issue_date" name="issue_date" class="form-control" required value="<?= date('Y-m-d') ?>"></div>
                                    <div class="col-xl-3 col-lg-6 col-md-6 col-12"><label for="exchange_rate" class="form-label">Exchange Rate (SAR to PKR)*</label><input type="number" id="exchange_rate" name="exchange_rate" class="form-control" step="0.01" required oninput="calculateTotals()"></div>
                                </div>
                            </fieldset>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-users"></i> Pilgrims Detail</h3>
                                <div class="mb-3" style="max-width: 300px;"><label class="form-label">Assign Vendor for Pilgrims</label><select name="pilgrims_vendor_id" class="form-select"><?= $vendor_options_html ?></select></div>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead class="table-light"><tr><th>Passport No</th><th>Passenger</th><th>DOB</th><th>DOX</th><th>Pax</th><th>Bed</th><th>Visa Sale (SAR)</th><th>Visa Cost (SAR)</th><th>Action</th></tr></thead><tbody id="pilgrims-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-sm btn-secondary mt-2" onclick="addPilgrimRow()">+ Add Pilgrim</button>
                            </section>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-hotel"></i> Accommodation</h3>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead class="table-light"><tr><th>City</th><th>Hotel</th><th>Vendor</th><th>Helpline</th><th>Check-In</th><th>Nights</th><th>Check-Out</th><th>Type</th><th>Meal</th><th>Qty</th><th>Sale Rate</th><th>Cost Rate</th><th>Net (SAR)</th><th>Action</th></tr></thead><tbody id="hotel-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-sm btn-secondary mt-2" onclick="addHotelRow()">+ Add Accommodation</button>
                            </section>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-bus"></i> Transportation</h3>
                                <div class="mb-3" style="max-width: 300px;"><label class="form-label">Assign Vendor for Transport</label><select name="transport_vendor_id" class="form-select"><?= $vendor_options_html ?></select></div>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead class="table-light"><tr><th>Vehicle</th><th>Route</th><th>Sale Rate</th><th>Cost Rate</th><th>Qty</th><th>Adults</th><th>Children</th><th>Net (SAR)</th><th>Action</th></tr></thead><tbody id="transport-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-sm btn-secondary mt-2" onclick="addTransportRow()">+ Add Transport</button>
                            </section>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-concierge-bell"></i> Other Services</h3>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead class="table-light"><tr><th rowspan="2">Service</th><th rowspan="2">Vendor</th><th colspan="2">Adult</th><th colspan="2">Child</th><th colspan="2">Infant</th><th rowspan="2">Total Cost (SAR)</th><th rowspan="2">Net (SAR)</th><th rowspan="2">Action</th></tr><tr><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th></tr></thead><tbody id="service-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-sm btn-secondary mt-2" onclick="addServiceRow()">+ Add Service</button>
                            </section>

                            <section class="mb-4"><h3 class="h5 fw-bold"><i class="fas fa-plane-departure"></i> Airline Tickets</h3>
                                <div class="mb-3" style="max-width: 300px;"><label class="form-label">Assign Vendor for Tickets</label><select name="tickets_vendor_id" class="form-select"><?= $vendor_options_html ?></select></div>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead class="table-light"><tr><th rowspan="2">Airline</th><th rowspan="2">Sector</th><th colspan="2">Adult</th><th colspan="2">Child</th><th colspan="2">Infant</th><th rowspan="2">Total Cost (PKR)</th><th rowspan="2">Net (PKR)</th><th rowspan="2">Action</th></tr><tr><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th><th>Qty</th><th>Rate</th></tr></thead><tbody id="ticket-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-sm btn-secondary mt-2" onclick="addTicketRow()">+ Add Ticket</button>
                            </section>

                            <fieldset class="mb-4"><legend class="h5 fw-bold"><i class="fas fa-cogs"></i> Finalization</legend>
                                <div class="row g-4">
                                    <div class="col-lg-7">
                                        <div class="mb-3"><label for="notes" class="form-label">Terms & Conditions / Notes</label><textarea id="notes" name="notes" class="form-control" rows="5">1. Please provide your approval in written to proceed booking...</textarea></div>
                                        <div class="row g-3">
                                            <div class="col-md-6"><label for="user_selector" class="form-label">Assign to User/Agent</label>
                                                <select id="user_selector" name="user_id" class="form-select"><option value="0" data-name="" selected>-- No Assignment --</option><?php foreach ($users_list as $user): ?><option value="<?= $user['id'] ?>" data-name="<?= e($user['name']) ?>"><?= e($user['name']) ?> - [<?= ($user['user_type'] === 'agent') ? 'A' : 'C' ?>]</option><?php endforeach; ?></select>
                                            </div>
                                            <div class="col-md-6"><label for="vendor_selector" class="form-label">Assign to Main Vendor</label><select id="vendor_selector" name="vendor_id" class="form-select"><?= $vendor_options_html ?></select></div>
                                            <div class="col-12 mt-3"><label class="form-label d-block">Status</label><div class="form-check form-check-inline"><input class="form-check-input" type="radio" name="status" id="statusTentative" value="Tentative" checked><label class="form-check-label" for="statusTentative">Tentative</label></div><div class="form-check form-check-inline"><input class="form-check-input" type="radio" name="status" id="statusApproved" value="Approved"><label class="form-check-label" for="statusApproved">Approved</label></div></div>
                                        </div>
                                    </div>
                                    <div class="col-lg-5">
                                        <div class="bg-light p-3 border rounded">
                                            <h4 class="h6 text-center mb-3">Invoice Summary</h4>
                                            <div class="summary-grid">
                                                <div class="summary-label">Total Amount (SAR):</div><div id="summary-total-sar">0.00</div>
                                                <div class="summary-label">Total Amount (PKR):</div><div id="summary-total-pkr-no-ticket">0.00</div>
                                                <div class="summary-label">Ticket Amount (PKR):</div><div id="summary-total-ticket-pkr">0.00</div>
                                                <div class="summary-label">Discount (PKR):</div><div><input type="number" name="discount_pkr" id="discount-pkr-input" class="form-control form-control-sm text-end" value="0.00" step="0.01" oninput="calculateTotals()"></div>
                                                <hr style="grid-column: 1 / -1; margin: 5px 0;"><div class="summary-label grand-total">Grand Total:</div><div class="grand-total text-end" id="summary-grand-total">0.00</div>
                                                <hr style="grid-column: 1 / -1; margin: 10px 0; border-style: dashed;"><div class="summary-label cost-label">Total Cost (PKR):</div><div class="cost-label text-end" id="summary-total-cost">0.00</div>
                                                <div class="summary-label profit-label grand-total">Net Profit:</div><div class="profit-label grand-total text-end" id="summary-net-profit">0.00</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </fieldset>

                            <div class="text-center mt-4"><button type="submit" class="btn btn-primary btn-lg"><i class="fas fa-check-circle"></i> Create Invoice</button></div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        const makkahHotels = <?= json_encode(array_values(array_unique($makkah_hotels))) ?>;
        const madinahHotels = <?= json_encode(array_values(array_unique($madinah_hotels))) ?>;
        const vendorSelectOptions = '<?php echo addslashes($vendor_options_html); ?>';
        const addRow = (tbodyId, rowHTML) => {
            const tbody = document.getElementById(tbodyId);
            if(tbody) tbody.insertAdjacentHTML('beforeend', rowHTML);
        }
        const removeRow = (btn) => { btn.closest('tr').remove(); calculateTotals(); };
        const calc = () => calculateTotals();

        function addPilgrimRow() { addRow('pilgrims-tbody', `<tr><td><input type="text" name="pilgrim_passport_no[]" class="form-control"></td><td><input type="text" name="pilgrim_passenger_details[]" class="form-control"></td><td><input type="date" name="pilgrim_dob[]" class="form-control"></td><td><input type="date" name="pilgrim_dox[]" class="form-control"></td><td><select name="pilgrim_pax[]" class="form-select"><option>Adult</option><option>Child</option></select></td><td><select name="pilgrim_bed[]" class="form-select"><option>TRUE</option><option>FALSE</option></select></td><td><input type="number" name="pilgrim_visa_price_sar[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="pilgrim_visa_price_sar_cost[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }
        function addHotelRow() {
            const newRowHTML = `<tr>
                <td><select name="hotel_city[]" class="form-select" onchange="updateHotelDropdown(this)"><option selected>Makkah</option><option>Madinah</option></select></td>
                <td><select name="hotel_name[]" class="hotel-select2"></select></td>
                <td><select name="hotel_vendor_id[]" class="form-select">${vendorSelectOptions}</select></td>
                <td><input type="text" name="hotel_helpline[]" class="form-control"></td>
                <td><input type="date" name="hotel_check_in[]" class="form-control" oninput="calculateNights(this)"></td>
                <td><input type="number" class="form-control" name="hotel_nights[]" value="0" oninput="calculateCheckoutDate(this)"></td>
                <td><input type="date" name="hotel_check_out[]" class="form-control" oninput="calculateNights(this)"></td>
                <td><input type="text" name="hotel_type[]" class="form-control" value="Quad"></td>
                <td><input type="text" name="hotel_meal_plan[]" class="form-control" value="RO"></td>
                <td><input type="number" name="hotel_qty[]" class="form-control" value="1" oninput="calc()"></td>
                <td><input type="number" name="hotel_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td>
                <td><input type="number" name="hotel_rate_cost[]" class="form-control" value="0" step="0.01" oninput="calc()"></td>
                <td class="table-light fw-bold"><input name="hotel_net[]" class="form-control text-end bg-light" readonly></td>
                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td>
            </tr>`;
            addRow('hotel-tbody', newRowHTML);
            const newRow = document.getElementById('hotel-tbody').lastElementChild;
            $(newRow.querySelector('.hotel-select2')).select2({ placeholder: 'Search hotel', tags: true, allowClear: true, width: '100%' });
            updateHotelDropdown(newRow.querySelector('select[name="hotel_city[]"]'));
        }
        function addTransportRow() { addRow('transport-tbody', `<tr><td><input type="text" name="transport_vehicle[]" class="form-control"></td><td><input type="text" name="transport_route[]" class="form-control"></td><td><input type="number" name="transport_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="transport_rate_cost[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="transport_qty[]" class="form-control" value="1" oninput="calc()"></td><td><input type="number" name="transport_adult_qty[]" class="form-control" value="0"></td><td><input type="number" name="transport_child_qty[]" class="form-control" value="0"></td><td class="table-light fw-bold"><input name="transport_net[]" class="form-control text-end bg-light" readonly></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }
        function addServiceRow() { addRow('service-tbody', `<tr><td><input type="text" name="service_name[]" class="form-control" value="Ziarat"></td><td><select name="service_vendor_id[]" class="form-select">${vendorSelectOptions}</select></td><td><input type="number" name="service_adult_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="service_adult_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="service_child_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="service_child_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="service_infant_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="service_infant_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="service_total_cost[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td class="table-light fw-bold"><input name="service_net[]" class="form-control text-end bg-light" readonly></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }
        function addTicketRow() { addRow('ticket-tbody', `<tr><td><input type="text" name="ticket_airline[]" class="form-control"></td><td><input type="text" name="ticket_sector[]" class="form-control"></td><td><input type="number" name="ticket_adult_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="ticket_adult_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="ticket_child_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="ticket_child_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="ticket_infant_qty[]" class="form-control" value="0" oninput="calc()"></td><td><input type="number" name="ticket_infant_rate[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td><input type="number" name="ticket_total_cost[]" class="form-control" value="0" step="0.01" oninput="calc()"></td><td class="table-light fw-bold"><input name="ticket_net[]" class="form-control text-end bg-light" readonly></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }

        function updateHotelDropdown(citySelect) {
            const hotelSelect = citySelect.closest('tr').querySelector('.hotel-select2');
            const selectedCity = citySelect.value.toLowerCase();
            const hotels = selectedCity === 'makkah' ? makkahHotels : madinahHotels;
            $(hotelSelect).empty();
            hotels.forEach(hotel => $(hotelSelect).append(new Option(hotel, hotel, false, false)));
            $(hotelSelect).trigger('change');
        }

        function calculateCheckoutDate(nightsInput) {
            const tr = nightsInput.closest('tr');
            const checkInInput = tr.querySelector('input[name="hotel_check_in[]"]');
            const checkOutInput = tr.querySelector('input[name="hotel_check_out[]"]');
            const nights = parseInt(nightsInput.value, 10);
            if (checkInInput.value && nights >= 0) {
                const checkInDate = new Date(checkInInput.value);
                checkInDate.setDate(checkInDate.getDate() + nights);
                checkOutInput.value = checkInDate.toISOString().split('T')[0];
            }
        }
        
        function calculateNights(dateInput) {
            const tr = dateInput.closest('tr');
            const checkInInput = tr.querySelector('input[name="hotel_check_in[]"]');
            const checkOutInput = tr.querySelector('input[name="hotel_check_out[]"]');
            const nightsInput = tr.querySelector('input[name="hotel_nights[]"]');
            if(checkInInput.value && checkOutInput.value) {
                const checkInDate = new Date(checkInInput.value);
                const checkOutDate = new Date(checkOutInput.value);
                if (checkOutDate > checkInDate) {
                    const diffTime = checkOutDate - checkInDate;
                    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
                    nightsInput.value = diffDays;
                } else {
                    nightsInput.value = 0;
                }
            }
        }
        
        function calculateTotals() {
            let totalSAR = 0, totalCostSAR = 0, totalTicketPKR = 0, totalCostTicketPKR = 0;
            const exchangeRate = parseFloat(document.getElementById('exchange_rate').value) || 0;

            document.querySelectorAll('#pilgrims-tbody tr').forEach(row => {
                totalSAR += parseFloat(row.querySelector('input[name="pilgrim_visa_price_sar[]"]').value) || 0;
                totalCostSAR += parseFloat(row.querySelector('input[name="pilgrim_visa_price_sar_cost[]"]').value) || 0;
            });
            document.querySelectorAll('#hotel-tbody tr').forEach(row => {
                const qty = parseFloat(row.querySelector('input[name="hotel_qty[]"]').value) || 0;
                const nights = parseFloat(row.querySelector('input[name="hotel_nights[]"]').value) || 0;
                const rate = parseFloat(row.querySelector('input[name="hotel_rate[]"]').value) || 0;
                const costRate = parseFloat(row.querySelector('input[name="hotel_rate_cost[]"]').value) || 0;
                const net = qty * nights * rate;
                row.querySelector('input[name="hotel_net[]"]').value = net.toFixed(2);
                totalSAR += net;
                totalCostSAR += qty * nights * costRate;
            });
            document.querySelectorAll('#transport-tbody tr').forEach(row => {
                const qty = parseFloat(row.querySelector('input[name="transport_qty[]"]').value) || 0;
                const rate = parseFloat(row.querySelector('input[name="transport_rate[]"]').value) || 0;
                const costRate = parseFloat(row.querySelector('input[name="transport_rate_cost[]"]').value) || 0;
                const net = qty * rate;
                row.querySelector('input[name="transport_net[]"]').value = net.toFixed(2);
                totalSAR += net;
                totalCostSAR += qty * costRate;
            });
            document.querySelectorAll('#service-tbody tr').forEach(row => {
                const adultQty = parseFloat(row.querySelector('input[name="service_adult_qty[]"]').value) || 0;
                const adultRate = parseFloat(row.querySelector('input[name="service_adult_rate[]"]').value) || 0;
                const childQty = parseFloat(row.querySelector('input[name="service_child_qty[]"]').value) || 0;
                const childRate = parseFloat(row.querySelector('input[name="service_child_rate[]"]').value) || 0;
                const infantQty = parseFloat(row.querySelector('input[name="service_infant_qty[]"]').value) || 0;
                const infantRate = parseFloat(row.querySelector('input[name="service_infant_rate[]"]').value) || 0;
                const net = (adultQty * adultRate) + (childQty * childRate) + (infantQty * infantRate);
                row.querySelector('input[name="service_net[]"]').value = net.toFixed(2);
                totalSAR += net;
                totalCostSAR += parseFloat(row.querySelector('input[name="service_total_cost[]"]').value) || 0;
            });
            document.querySelectorAll('#ticket-tbody tr').forEach(row => {
                const adultQty = parseFloat(row.querySelector('input[name="ticket_adult_qty[]"]').value) || 0;
                const adultRate = parseFloat(row.querySelector('input[name="ticket_adult_rate[]"]').value) || 0;
                const childQty = parseFloat(row.querySelector('input[name="ticket_child_qty[]"]').value) || 0;
                const childRate = parseFloat(row.querySelector('input[name="ticket_child_rate[]"]').value) || 0;
                const infantQty = parseFloat(row.querySelector('input[name="ticket_infant_qty[]"]').value) || 0;
                const infantRate = parseFloat(row.querySelector('input[name="ticket_infant_rate[]"]').value) || 0;
                const net = (adultQty * adultRate) + (childQty * childRate) + (infantQty * infantRate);
                row.querySelector('input[name="ticket_net[]"]').value = net.toFixed(2);
                totalTicketPKR += net;
                totalCostTicketPKR += parseFloat(row.querySelector('input[name="ticket_total_cost[]"]').value) || 0;
            });

            const totalPkrNoTicket = totalSAR * exchangeRate;
            const discountPKR = parseFloat(document.getElementById('discount-pkr-input').value) || 0;
            const grandTotal = totalPkrNoTicket + totalTicketPKR - discountPKR;
            const totalCostPkrNoTicket = totalCostSAR * exchangeRate;
            const grandTotalCost = totalCostPkrNoTicket + totalCostTicketPKR;
            const netProfit = grandTotal - grandTotalCost;

            document.getElementById('summary-total-sar').textContent = totalSAR.toFixed(2);
            document.getElementById('summary-total-pkr-no-ticket').textContent = totalPkrNoTicket.toFixed(2);
            document.getElementById('summary-total-ticket-pkr').textContent = totalTicketPKR.toFixed(2);
            document.getElementById('summary-grand-total').textContent = grandTotal.toFixed(2);
            document.getElementById('summary-total-cost').textContent = grandTotalCost.toFixed(2);
            document.getElementById('summary-net-profit').textContent = netProfit.toFixed(2);
        }

        document.addEventListener('DOMContentLoaded', () => {
            $('#user_selector, #vendor_selector').select2({ width: '100%' });
            $('#user_selector').on('change', function() {
                const selected = $(this).find('option:selected');
                $('#guest_name').val(selected.data('name') || '');
            });
            addPilgrimRow(); 
            addHotelRow();
            calculateTotals();
        });
    </script>
</body>
</html>