<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK: Ensure user is a logged-in admin ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- FETCH USERS FOR THE DROPDOWN ---
$users_list = [];
$sql_users = "SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC";
$result_users = $conn->query($sql_users);
if ($result_users) {
    $users_list = $result_users->fetch_all(MYSQLI_ASSOC);
}

// --- HANDLE FORM SUBMISSION ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['create_quick_booking'])) {
    $conn->begin_transaction();
    try {
        $user_id = (int)($_POST['user_id'] ?? 0);
        $user_id_to_save = ($user_id > 0) ? $user_id : null;
        $customer_name = trim($_POST['customer_name']);
        $package_type = $_POST['package_type'];
        $room_type = $_POST['room_type'];
        $status = $_POST['status']; // Capture status from form

        if (empty($customer_name)) throw new Exception("Customer Name is a required field.");
        if (!isset($_POST['family_head'])) throw new Exception("Please select one passenger as the Family Head.");

        $stmt_get_max_id = $conn->prepare("SELECT MAX(id) as max_id FROM quick_bookings");
        $stmt_get_max_id->execute();
        $result_max_id = $stmt_get_max_id->get_result()->fetch_assoc();
        $last_id = $result_max_id['max_id'] ?? 0;
        $start_id = 100001;
        $booking_id = ($last_id >= $start_id) ? $last_id + 1 : $start_id;

        $sql_booking = "INSERT INTO quick_bookings (id, user_id, customer_name, package_type, room_type, status) VALUES (?, ?, ?, ?, ?, ?)";
        $stmt_booking = $conn->prepare($sql_booking);
        $stmt_booking->bind_param("iissss", $booking_id, $user_id_to_save, $customer_name, $package_type, $room_type, $status);
        $stmt_booking->execute();

        if ($stmt_booking->affected_rows <= 0) {
            throw new Exception("Failed to create the main booking record.");
        }

        if (!empty($_POST['passenger_name']) && is_array($_POST['passenger_name'])) {
            $sql_passenger = "INSERT INTO quick_booking_passengers (booking_id, is_family_head, full_name, passport_no, visa_no, dob, gender, passport_issue_date, passport_expiry_date, pax_type, bed_required, group_no, pnr_no) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_passenger = $conn->prepare($sql_passenger);
            $family_head_index = $_POST['family_head'];

            foreach ($_POST['passenger_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $is_head = ($key == $family_head_index) ? 1 : 0;
                $dob = !empty($_POST['dob'][$key]) ? $_POST['dob'][$key] : null;
                $issue_date = !empty($_POST['passport_issue_date'][$key]) ? $_POST['passport_issue_date'][$key] : null;
                $expiry_date = !empty($_POST['passport_expiry_date'][$key]) ? $_POST['passport_expiry_date'][$key] : null;

                $stmt_passenger->bind_param(
                    "iisssssssssss",
                    $booking_id,
                    $is_head,
                    $name,
                    $_POST['passport_no'][$key],
                    $_POST['visa_no'][$key],
                    $dob,
                    $_POST['gender'][$key],
                    $issue_date,
                    $expiry_date,
                    $_POST['pax_type'][$key],
                    $_POST['bed_required'][$key],
                    $_POST['group_no'][$key],
                    $_POST['pnr_no'][$key]
                );
                $stmt_passenger->execute();
            }
            $stmt_passenger->close();
        }

        $conn->commit();
        $_SESSION['success_message'] = "Quick Booking #" . $booking_id . " created successfully!";
        header("Location: create-quick-booking.php");
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        $_SESSION['error_message'] = "Error Creating Booking: " . $e->getMessage();
        header("Location: create-quick-booking.php");
        exit();
    }
}

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Quick Booking</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .select2-container--default .select2-selection--single {
            height: calc(1.5em + .75rem + 2px); padding: .375rem .75rem; border: 1px solid #ced4da;
        }
        .select2-container--default .select2-selection--single .select2-selection__rendered { line-height: 1.5; padding-left: 0;}
        .select2-container--default .select2-selection--single .select2-selection__arrow { height: calc(1.5em + .75rem); }
        .passenger-card {
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: .375rem;
            margin-bottom: 1rem;
            position: relative;
        }
        .passenger-card .remove-row-btn {
            position: absolute;
            top: 10px;
            right: 10px;
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header"><div class="user-info"><span>Create Quick Booking</span></div></header>
            <main class="content-body">
                <div class="content-card">

                    <?php if ($success_message): ?><div class="alert alert-success"><?= e($success_message); ?></div><?php endif; ?>
                    <?php if ($error_message): ?><div class="alert alert-danger"><?= e($error_message); ?></div><?php endif; ?>

                    <form action="create-quick-booking.php" method="POST" id="quick-booking-form">
                        
                        <fieldset class="p-3 mb-4 border rounded-3">
                            <legend class="fw-bold h5"><i class="fas fa-book"></i> Booking Details</legend>
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="user_selector" class="form-label">Select User (Optional)</label>
                                    <select id="user_selector" name="user_id" class="form-select">
                                        <option value="0" data-name="" selected>-- Manual Customer Entry --</option>
                                        <?php foreach ($users_list as $user): ?>
                                            <option value="<?= e($user['id']) ?>" data-name="<?= e($user['name']) ?>"><?= e($user['name']) ?> - [<?= e(strtoupper(substr($user['user_type'], 0, 1))) ?>]</option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="customer_name" class="form-label">Customer Name*</label>
                                    <input type="text" id="customer_name" name="customer_name" class="form-control" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="package_type" class="form-label">Package Type</label>
                                    <select id="package_type" name="package_type" class="form-select" required>
                                        <option value="Standard">Standard</option>
                                        <option value="Star">Star</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="room_type" class="form-label">Room Type</label>
                                    <select id="room_type" name="room_type" class="form-select" required>
                                        <option value="Sharing">Sharing</option>
                                        <option value="Quint">Quint</option>
                                        <option value="Quad">Quad</option>
                                        <option value="Triple">Triple</option>
                                        <option value="Double">Double</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="status" class="form-label">Status</label>
                                    <select id="status" name="status" class="form-select" required>
                                        <option value="Pending" selected>Pending</option>
                                        <option value="Confirmed">Confirmed</option>
                                        <option value="Cancelled">Cancelled</option>
                                    </select>
                                </div>
                            </div>
                        </fieldset>

                        <section>
                            <h3 class="h5 fw-bold mb-3"><i class="fas fa-users"></i> Passenger Details</h3>
                            <div id="passengers-container"></div>
                            <button type="button" class="btn btn-secondary" onclick="addPassengerRow()">
                                <i class="fas fa-plus"></i> Add Passenger
                            </button>
                        </section>

                        <div class="text-center mt-4">
                            <button type="submit" name="create_quick_booking" class="btn btn-primary btn-lg">Create Quick Booking</button>
                        </div>
                    </form>
                </div>
            </main>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
        let passengerIndex = 0;

        document.addEventListener('DOMContentLoaded', function() {
            $('#user_selector').select2({ width: '100%' });

            const userSelector = $('#user_selector');
            const customerNameInput = $('#customer_name');

            userSelector.on('change', function() {
                const selectedOption = $(this).find('option:selected');
                customerNameInput.val(selectedOption.data('name') || '');
            });

            addPassengerRow();
        });

        function addPassengerRow() {
            const container = document.getElementById('passengers-container');
            const isFirstRow = container.children.length === 0;
            const checkedAttribute = isFirstRow ? 'checked' : '';

            const card = document.createElement('div');
            card.className = 'passenger-card';
            card.id = `passenger-card-${passengerIndex}`;

            card.innerHTML = `
                <button type="button" class="btn btn-danger btn-sm remove-row-btn" onclick="removeRow(${passengerIndex})"><i class="fas fa-trash"></i></button>
                <div class="p-3">
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="radio" name="family_head" id="family_head_${passengerIndex}" value="${passengerIndex}" ${checkedAttribute}>
                        <label class="form-check-label fw-bold" for="family_head_${passengerIndex}">
                            This Passenger is the Family Head
                        </label>
                    </div>
                    <div class="row g-3">
                        <div class="col-lg-4 col-md-6"><label class="form-label">Full Name*</label><input type="text" name="passenger_name[${passengerIndex}]" class="form-control" required></div>
                        <div class="col-lg-4 col-md-6"><label class="form-label">Passport No.</label><input type="text" name="passport_no[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-4 col-md-6"><label class="form-label">Visa No.</label><input type="text" name="visa_no[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Pax Type</label><select name="pax_type[${passengerIndex}]" class="form-select"><option value="Adult">Adult</option><option value="Child">Child</option><option value="Infant">Infant</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Bed Required</label><select name="bed_required[${passengerIndex}]" class="form-select"><option value="Yes">Yes</option><option value="No">No</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Group #</label><input type="text" name="group_no[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">PNR #</label><input type="text" name="pnr_no[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Date of Birth</label><input type="date" name="dob[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Gender</label><select name="gender[${passengerIndex}]" class="form-select"><option value="Male">Male</option><option value="Female">Female</option><option value="Other">Other</option></select></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Passport Issue</label><input type="date" name="passport_issue_date[${passengerIndex}]" class="form-control"></div>
                        <div class="col-lg-3 col-md-6"><label class="form-label">Passport Expiry</label><input type="date" name="passport_expiry_date[${passengerIndex}]" class="form-control"></div>
                    </div>
                </div>
            `;
            container.appendChild(card);
            passengerIndex++;
        }

        function removeRow(index) {
            const cardToRemove = document.getElementById(`passenger-card-${index}`);
            const radio = cardToRemove.querySelector('input[type="radio"]');

            if (radio && radio.checked) {
                cardToRemove.remove();
                const firstRadio = document.querySelector('#passengers-container input[type="radio"]');
                if (firstRadio) {
                    firstRadio.checked = true;
                }
            } else {
                cardToRemove.remove();
            }
        }

        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>
</html>