<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- LOGIC TO PRE-FILL FORM FROM A QUICK BOOKING ---
$quick_booking_data = null;
$quick_booking_passengers = [];
$family_head_name_prefill = '';
$quick_booking_id = (int)($_GET['quick_booking_id'] ?? 0);

if ($quick_booking_id > 0) {
    $stmt_qb = $conn->prepare("SELECT * FROM quick_bookings WHERE id = ?");
    $stmt_qb->bind_param("i", $quick_booking_id);
    $stmt_qb->execute();
    $quick_booking_data = $stmt_qb->get_result()->fetch_assoc();

    $stmt_pass = $conn->prepare("SELECT * FROM quick_booking_passengers WHERE booking_id = ? ORDER BY is_family_head DESC, id ASC");
    $stmt_pass->bind_param("i", $quick_booking_id);
    $stmt_pass->execute();
    $quick_booking_passengers = $stmt_pass->get_result()->fetch_all(MYSQLI_ASSOC);

    foreach ($quick_booking_passengers as $p) {
        if ($p['is_family_head']) {
            $family_head_name_prefill = $p['full_name'];
            break;
        }
    }
    if (empty($family_head_name_prefill)) {
        $family_head_name_prefill = $quick_booking_data['customer_name'] ?? '';
    }
}

// --- FETCH DATA FOR DROPDOWNS ---
$users_list = $conn->query("SELECT id, name, user_type FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$makkah_hotels = [];
$madinah_hotels = [];
$hotel_result = $conn->query("SELECT hotel_name, city FROM rate_sheets ORDER BY hotel_name ASC");
if ($hotel_result) {
    while ($row = $hotel_result->fetch_assoc()) {
        if (strtolower(trim($row['city'])) === 'makkah') $makkah_hotels[] = $row['hotel_name'];
        elseif (strtolower(trim($row['city'])) === 'madinah') $madinah_hotels[] = $row['hotel_name'];
    }
}
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- HANDLE FORM SUBMISSION ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    try {
        $shirka_logo_path = null;
        if (isset($_FILES['shirka_logo']) && $_FILES['shirka_logo']['error'] == UPLOAD_ERR_OK) {
            $upload_dir = 'uploads/shirka_logos/';
            if (!is_dir($upload_dir)) mkdir($upload_dir, 0777, true);
            $unique_filename = uniqid('shirka_', true) . '.' . strtolower(pathinfo($_FILES['shirka_logo']['name'], PATHINFO_EXTENSION));
            $shirka_logo_path = $upload_dir . $unique_filename;
            if (!move_uploaded_file($_FILES['shirka_logo']['tmp_name'], $shirka_logo_path)) throw new Exception("Failed to upload Shirka logo.");
        }
        $user_id = (int)($_POST['user_id'] ?? 0);
        $user_id_to_save = ($user_id > 0) ? $user_id : null;
        $vendor_id = (int)($_POST['vendor_id'] ?? 0);
        $vendor_id_to_save = ($vendor_id > 0) ? $vendor_id : null;

        $status = $_POST['status'];
        $voucher_date = $_POST['voucher_date'];
        $family_head_name = $_POST['family_head_name'];
        $booking_ref_no = $_POST['booking_ref_no'];
        $manual_no = $_POST['manual_no'];
        $package_type = $_POST['package_type'];
        $package_duration_nights = (int)$_POST['package_duration_nights'];
        $pax_summary = $_POST['pax_summary'];
        $shirka_name = $_POST['shirka_name'];
        $transporter_name = $_POST['transporter_name'];
        $transport_type = $_POST['transport_type'];
        $transport_description = $_POST['transport_description'];
        $transport_brn = $_POST['transport_brn'];
        $transport_helpline_1 = $_POST['transport_helpline_1'];
        $transport_helpline_2 = $_POST['transport_helpline_2'];
        $hotel_checkin_time = $_POST['hotel_checkin_time'];
        $hotel_checkout_time = $_POST['hotel_checkout_time'];
        $notes_urdu = $_POST['notes_urdu'];

        $sql_voucher = "INSERT INTO vouchers (user_id, vendor_id, status, voucher_date, family_head_name, booking_ref_no, manual_no, package_type, package_duration_nights, pax_summary, shirka_logo_path, shirka_name, transporter_name, transport_type, transport_description, transport_brn, transport_helpline_1, transport_helpline_2, hotel_checkin_time, hotel_checkout_time, notes_urdu) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmt_voucher = $conn->prepare($sql_voucher);
        $stmt_voucher->bind_param("iissssssissssssssssss", $user_id_to_save, $vendor_id_to_save, $status, $voucher_date, $family_head_name, $booking_ref_no, $manual_no, $package_type, $package_duration_nights, $pax_summary, $shirka_logo_path, $shirka_name, $transporter_name, $transport_type, $transport_description, $transport_brn, $transport_helpline_1, $transport_helpline_2, $hotel_checkin_time, $hotel_checkout_time, $notes_urdu);
        $stmt_voucher->execute();
        $voucher_id = $conn->insert_id;
        if ($voucher_id <= 0) throw new Exception("Failed to create the main voucher record.");

        if (!empty($_POST['mutamer_name'])) {
            $stmt_mutamer = $conn->prepare("INSERT INTO voucher_mutamers (voucher_id, mutamer_name, passport_no, pax_type, bed_required, group_no, visa_no, pnr_no) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
            foreach ($_POST['mutamer_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $stmt_mutamer->bind_param("isssssss", $voucher_id, $name, $_POST['mutamer_passport_no'][$key], $_POST['mutamer_pax_type'][$key], $_POST['mutamer_bed_required'][$key], $_POST['mutamer_group_no'][$key], $_POST['mutamer_visa_no'][$key], $_POST['mutamer_pnr_no'][$key]);
                $stmt_mutamer->execute();
            }
        }
        
        // *** FIX STARTS HERE ***
        if (!empty($_POST['hotel_name'])) {
            // Corrected SQL statement to include the 'view_type' column
            $sql_accom = "INSERT INTO voucher_accommodations (voucher_id, city, hotel_name, room_type, check_in_date, nights, check_out_date, view_type, meal_plan, confirmation_no, person_name, person_call_number, person_whatsapp_number) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_accom = $conn->prepare($sql_accom);

            // Check if prepare() failed
            if ($stmt_accom === false) {
                throw new Exception("Database prepare failed for accommodation: " . $conn->error);
            }

            foreach ($_POST['hotel_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $checkInDateStr = !empty($_POST['hotel_check_in'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_in'][$key]))) : null;
                $checkOutDateStr = !empty($_POST['hotel_check_out'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_out'][$key]))) : null;
                $nights = (int)($_POST['hotel_nights'][$key] ?? 0);
                $view_type = 'Standard'; // Provide a value for the view_type column

                // Corrected bind_param with the right number of variables and types
                $stmt_accom->bind_param("issssisssssss", $voucher_id, $_POST['hotel_city'][$key], $name, $_POST['hotel_room_type'][$key], $checkInDateStr, $nights, $checkOutDateStr, $view_type, $_POST['hotel_meal_plan'][$key], $_POST['hotel_confirmation_no'][$key], $_POST['hotel_person_name'][$key], $_POST['hotel_call_number'][$key], $_POST['hotel_whatsapp_number'][$key]);
                $stmt_accom->execute();
            }
        }
        // *** FIX ENDS HERE ***

        if (!empty(trim($_POST['flight_no_dep']))) {
            $stmt_flight = $conn->prepare("INSERT INTO voucher_flights (voucher_id, direction, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)");
            $direction_dep = 'Pakistan To KSA';
            $stmt_flight->bind_param("isssss", $voucher_id, $direction_dep, $_POST['flight_no_dep'], $_POST['sector_dep'], $_POST['departure_datetime_dep'], $_POST['arrival_datetime_dep']);
            $stmt_flight->execute();
        }
        if (!empty(trim($_POST['flight_no_arr']))) {
            $stmt_flight = $conn->prepare("INSERT INTO voucher_flights (voucher_id, direction, flight_no, sector, departure_datetime, arrival_datetime) VALUES (?, ?, ?, ?, ?, ?)");
            $direction_arr = 'KSA To Pakistan';
            $stmt_flight->bind_param("isssss", $voucher_id, $direction_arr, $_POST['flight_no_arr'], $_POST['sector_arr'], $_POST['departure_datetime_arr'], $_POST['arrival_datetime_arr']);
            $stmt_flight->execute();
        }

        $invoice_number_from_voucher = preg_replace('/^hb-\s*/i', '', $booking_ref_no);
        $invoice_notes = "1. Please provide your approval in written to proceed booking.\n2. Rate and Dates are subject to hotel/transport availability and reconfirmation from both parties.\n3. Rate are Net and Non-Commisionable.\nSUBJECT TO AVAILABILITY AND EX RATES APPLIED AS PER DATE OF FULL PAYMENT";

        $sql_invoice = "INSERT INTO invoices (user_id, vendor_id, guest_name, invoice_number, status, issue_date, notes, exchange_rate) VALUES (?, ?, ?, ?, 'Tentative', ?, ?, 0.00)";
        $stmt_invoice = $conn->prepare($sql_invoice);
        $stmt_invoice->bind_param("iissss", $user_id_to_save, $vendor_id_to_save, $family_head_name, $invoice_number_from_voucher, $voucher_date, $invoice_notes);
        $stmt_invoice->execute();
        $invoice_id = $conn->insert_id;
        if ($invoice_id <= 0) throw new Exception("Voucher created, but failed to auto-create the invoice.");

        $stmt_link_voucher = $conn->prepare("UPDATE vouchers SET invoice_id = ? WHERE id = ?");
        $stmt_link_voucher->bind_param("ii", $invoice_id, $voucher_id);
        $stmt_link_voucher->execute();

        if (!empty($_POST['mutamer_name'])) {
            $stmt_pilgrim = $conn->prepare("INSERT INTO invoice_pilgrims (invoice_id, passport_no, passenger_details, dob, dox, pax, bed) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $passenger_details_source = [];
            if ($quick_booking_id > 0) { foreach ($quick_booking_passengers as $p) { $passenger_details_source[$p['passport_no']] = $p; } }
            foreach ($_POST['mutamer_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $passport_no = $_POST['mutamer_passport_no'][$key];
                $dob = $passenger_details_source[$passport_no]['dob'] ?? null;
                $dox = $passenger_details_source[$passport_no]['passport_expiry_date'] ?? null;
                $stmt_pilgrim->bind_param("issssss", $invoice_id, $passport_no, $name, $dob, $dox, $_POST['mutamer_pax_type'][$key], $_POST['mutamer_bed_required'][$key]);
                $stmt_pilgrim->execute();
            }
        }
        if (!empty($_POST['hotel_name'])) {
            $stmt_inv_accom = $conn->prepare("INSERT INTO invoice_hotels (invoice_id, city, hotel_name, room_type, check_in, check_out, nights, meal_plan, rooms) VALUES (?, ?, ?, ?, ?, ?, ?, ?, 1)");
            foreach ($_POST['hotel_name'] as $key => $name) {
                if (empty(trim($name))) continue;
                $checkInDateStr = !empty($_POST['hotel_check_in'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_in'][$key]))) : null;
                $checkOutDateStr = !empty($_POST['hotel_check_out'][$key]) ? date('Y-m-d', strtotime(str_replace('/', '-', $_POST['hotel_check_out'][$key]))) : null;
                $nights = (int)($_POST['hotel_nights'][$key] ?? 0);
                $stmt_inv_accom->bind_param("isssssis", $invoice_id, $_POST['hotel_city'][$key], $name, $_POST['hotel_room_type'][$key], $checkInDateStr, $checkOutDateStr, $nights, $_POST['hotel_meal_plan'][$key]);
                $stmt_inv_accom->execute();
            }
        }

        $conn->commit();
        $_SESSION['success_message'] = "Voucher #" . $voucher_id . " & Invoice #" . $invoice_id . " created! Please add prices to the invoice.";
        header("Location: edit-invoice.php?id=" . $invoice_id);
        exit();
    } catch (Exception $e) {
        $conn->rollback();
        if (isset($shirka_logo_path) && file_exists($shirka_logo_path)) { unlink($shirka_logo_path); }
        $_SESSION['error_message'] = "Error Creating Records: " . $e->getMessage();
        header("Location: create-voucher.php" . ($quick_booking_id > 0 ? "?quick_booking_id=$quick_booking_id" : ""));
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Travel Voucher</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/css/bootstrap-datepicker.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Noto+Nastaliq+Urdu:wght@400;700&display=swap" rel="stylesheet">
    <style>
        :root { --form-grid-gap: 1.25rem; }
        .voucher-form fieldset {
            border: 1px solid var(--border-color, #e0e0e0);
            padding: 1.5rem;
            border-radius: 8px;
            margin-bottom: 2rem;
            background-color: #fdfdfd;
        }
        .voucher-form legend {
            font-weight: 600;
            padding: 0 10px;
            font-size: 1.1rem;
            color: var(--primary-color, #007bff);
        }
        .form-grid {
            display: grid;
            gap: var(--form-grid-gap);
        }
        .grid-cols-2 { grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); }
        .grid-cols-3 { grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); }
        
        .select2-container .select2-selection--single { height: calc(1.5em + .75rem + 2px); padding: .375rem .75rem; border: 1px solid #ced4da; }
        .dynamic-table th, .dynamic-table td { vertical-align: middle; text-align: center; font-size: 0.9rem;}
        .dynamic-table .form-select, .dynamic-table .form-control { font-size: 0.9rem; min-width: 100px; }
        textarea[name="notes_urdu"] { direction: rtl; font-family: 'Noto Nastaliq Urdu', sans-serif; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Create New Voucher</span></div>
            </header>
            <main class="content-body">
                <div class="content-card">
                    <div class="card-body">
                        <div class="alert alert-info">Creating a voucher will also automatically generate a corresponding price-less invoice. You will be redirected to the invoice to add pricing.</div>
                        <?php if (isset($_SESSION['error_message'])): ?>
                        <div class="notice error"><?= e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
                        <?php endif; ?>

                        <form action="create-voucher.php<?= $quick_booking_id > 0 ? '?quick_booking_id=' . $quick_booking_id : '' ?>" method="POST" enctype="multipart/form-data" class="voucher-form">

                           <!-- All form fields remain the same -->
                           
                           <fieldset><legend>Voucher Details</legend>
                                <div class="form-grid grid-cols-3">
                                    <div class="form-group"><label>Manual No</label><input type="text" class="form-control" name="manual_no" value="<?= e($quick_booking_data['id'] ?? '') ?>" <?= $quick_booking_data ? 'readonly' : '' ?>></div>
                                    <div class="form-group"><label>Booking Ref #</label><input type="text" class="form-control" name="booking_ref_no"></div>
                                    <div class="form-group"><label>Voucher Date*</label><input type="date" class="form-control" name="voucher_date" required value="<?= date('Y-m-d') ?>"></div>
                                    <div class="form-group"><label>Family Head Name*</label><input type="text" class="form-control" id="family_head_name" name="family_head_name" value="<?= e($family_head_name_prefill) ?>" required></div>
                                    <div class="form-group"><label>Pkg Type</label><input type="text" class="form-control" name="package_type" value="<?= e($quick_booking_data['package_type'] ?? 'Standard') ?>"></div>
                                    <div class="form-group"><label>Duration (Nights)</label><input type="number" class="form-control" name="package_duration_nights"></div>
                                    <div class="form-group"><label>Pax Summary</label><input type="text" class="form-control" name="pax_summary" value="<?= count($quick_booking_passengers) > 0 ? '(A-' . count($quick_booking_passengers) . ':C-0:I-0) Beds(' . count($quick_booking_passengers) . ')' : '' ?>" placeholder="(A-4:C-0:I-0) Beds(4)"></div>
                                    <div class="form-group"><label>Shirka Name</label><input type="text" class="form-control" name="shirka_name"></div>
                                    <div class="form-group"><label>Shirka Logo</label><input type="file" class="form-control" name="shirka_logo" accept="image/*"></div>
                                </div>
                            </fieldset>

                            <fieldset><legend><i class="fas fa-users"></i> Mutamers</legend>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead><tr><th>Name</th><th>Passport No</th><th>Pax</th><th>Bed</th><th>Group #</th><th>Visa #</th><th>PNR #</th><th>Action</th></tr></thead><tbody id="mutamers-tbody"><?php foreach ($quick_booking_passengers as $p): ?><tr><td><input type="text" name="mutamer_name[]" class="form-control" value="<?= e($p['full_name']) ?>"></td><td><input type="text" name="mutamer_passport_no[]" class="form-control" value="<?= e($p['passport_no']) ?>"></td><td><select name="mutamer_pax_type[]" class="form-select"><option value="Adult" <?= e($p['pax_type']) == 'Adult' ? 'selected' : '' ?>>Adult</option><option value="Child" <?= e($p['pax_type']) == 'Child' ? 'selected' : '' ?>>Child</option><option value="Infant" <?= e($p['pax_type']) == 'Infant' ? 'selected' : '' ?>>Infant</option></select></td><td><select name="mutamer_bed_required[]" class="form-select"><option value="Yes" <?= e($p['bed_required']) == 'Yes' ? 'selected' : '' ?>>Yes</option><option value="No" <?= e($p['bed_required']) == 'No' ? 'selected' : '' ?>>No</option></select></td><td><input type="text" name="mutamer_group_no[]" class="form-control" value="<?= e($p['group_no']) ?>"></td><td><input type="text" name="mutamer_visa_no[]" class="form-control" value="<?= e($p['visa_no']) ?>"></td><td><input type="text" name="mutamer_pnr_no[]" class="form-control" value="<?= e($p['pnr_no']) ?>"></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr><?php endforeach; ?></tbody></table></div>
                                <button type="button" class="btn btn-secondary btn-sm mt-2" onclick="addMutamerRow()">+ Add Mutamer</button>
                            </fieldset>

                            <fieldset><legend><i class="fas fa-hotel"></i> Accommodation</legend>
                                <div class="table-responsive"><table class="table table-bordered dynamic-table"><thead><tr><th>City</th><th style="width: 25%;">Hotel</th><th>Check-In (dd/mm/yyyy)</th><th>Nights</th><th>Check-Out (dd/mm/yyyy)</th><th>Room Type</th><th>Meal</th><th>Conf #</th><th>Action</th></tr></thead><tbody id="hotel-tbody"></tbody></table></div>
                                <button type="button" class="btn btn-secondary btn-sm mt-2" onclick="addHotelRow()">+ Add Accommodation</button>
                            </fieldset>

                            <fieldset><legend>Makkah Hotel Contact Details</legend>
                                <div class="table-responsive">
                                    <table class="table table-bordered dynamic-table">
                                        <thead><tr><th style="width: 25%;">Hotel Name (Ref)</th><th>Person Name</th><th>Person call number</th><th>Person WhatsApp Number</th></tr></thead>
                                        <tbody id="makkah-contacts-tbody"></tbody>
                                    </table>
                                </div>
                            </fieldset>

                            <fieldset><legend>Madinah Shareef Hotel Contact Details</legend>
                                <div class="table-responsive">
                                    <table class="table table-bordered dynamic-table">
                                         <thead><tr><th style="width: 25%;">Hotel Name (Ref)</th><th>Person Name</th><th>Person call number</th><th>Person WhatsApp Number</th></tr></thead>
                                        <tbody id="madinah-contacts-tbody"></tbody>
                                    </table>
                                </div>
                            </fieldset>

                            <fieldset><legend><i class="fas fa-bus"></i> Transport</legend>
                                <div class="form-grid grid-cols-3">
                                    <div class="form-group"><label>Transporter</label>
                                        <select name="transporter_name" class="form-select">
                                            <option value="Company Transport" selected>Company Transport</option>
                                            <option value="Private">Private</option>
                                        </select>
                                    </div>
                                    <div class="form-group"><label>Type</label>
                                        <select name="transport_type" class="form-select">
                                            <option value="Economy By Bus" selected>Economy By Bus</option>
                                            <option value="Luxury By Bus">Luxury By Bus</option>
                                            <option value="Sedan Car">Sedan Car</option>
                                            <option value="SUV / GMC">SUV / GMC</option>
                                        </select>
                                    </div>
                                    <div class="form-group"><label>BRN</label><input type="text" class="form-control" name="transport_brn"></div>
                                    <div class="form-group"><label>Description</label>
                                        <select name="transport_description" class="form-select">
                                            <option value="Round Trip (Full Package)" selected>Round Trip (Full Package)</option>
                                            <option value="Jeddah Airport -> Makkah Hotel">Jeddah Airport -> Makkah Hotel</option>
                                            <option value="Makkah Hotel -> Madinah Hotel">Makkah Hotel -> Madinah Hotel</option>
                                            <option value="Madinah Hotel -> Madinah Airport">Madinah Hotel -> Madinah Airport</option>
                                            <option value="Madinah Hotel -> Jeddah Airport">Madinah Hotel -> Jeddah Airport</option>
                                            <option value="Makkah Hotel -> Jeddah Airport">Makkah Hotel -> Jeddah Airport</option>
                                            <option value="Ziarat Makkah Only">Ziarat Makkah Only</option>
                                            <option value="Ziarat Madinah Only">Ziarat Madinah Only</option>
                                            <option value="Ziarat Makkah & Madinah">Ziarat Makkah & Madinah</option>
                                        </select>
                                    </div>
                                    <div class="form-group"><label>Helpline 1</label><input type="text" class="form-control" name="transport_helpline_1" value="00092-3007763000"></div>
                                    <div class="form-group"><label>Helpline 2</label><input type="text" class="form-control" name="transport_helpline_2" value="0092-3001119170"></div>
                                </div>
                            </fieldset>

                            <fieldset><legend><i class="fas fa-plane-departure"></i> Flight Details</legend>
                                <div class="form-grid grid-cols-2">
                                    <div><h6>Departure Flight (Pakistan to KSA)</h6>
                                        <div class="form-grid grid-cols-2">
                                            <div class="form-group"><label>Flight No</label><input type="text" name="flight_no_dep" class="form-control"></div>
                                            <div class="form-group"><label>Sector</label><input type="text" name="sector_dep" class="form-control"></div>
                                            <div class="form-group"><label>Departure</label><input type="datetime-local" name="departure_datetime_dep" class="form-control"></div>
                                            <div class="form-group"><label>Arrival</label><input type="datetime-local" name="arrival_datetime_dep" class="form-control"></div>
                                        </div>
                                    </div>
                                    <div><h6>Return Flight (KSA to Pakistan)</h6>
                                        <div class="form-grid grid-cols-2">
                                            <div class="form-group"><label>Flight No</label><input type="text" name="flight_no_arr" class="form-control"></div>
                                            <div class="form-group"><label>Sector</label><input type="text" name="sector_arr" class="form-control"></div>
                                            <div class="form-group"><label>Departure</label><input type="datetime-local" name="departure_datetime_arr" class="form-control"></div>
                                            <div class="form-group"><label>Arrival</label><input type="datetime-local" name="arrival_datetime_arr" class="form-control"></div>
                                        </div>
                                    </div>
                                </div>
                            </fieldset>
                            
                            <fieldset><legend><i class="fas fa-cogs"></i> Final Settings</legend>
                                <div class="form-grid grid-cols-3">
                                    <div class="form-group">
                                        <label class="d-block">Status*</label>
                                        <div style="padding-top: 8px;">
                                            <div class="form-check form-check-inline">
                                                <input class="form-check-input" type="radio" name="status" id="statusTentative" value="Tentative" checked>
                                                <label class="form-check-label" for="statusTentative">Tentative</label>
                                            </div>
                                            <div class="form-check form-check-inline">
                                                <input class="form-check-input" type="radio" name="status" id="statusConfirmed" value="Confirmed">
                                                <label class="form-check-label" for="statusConfirmed">Confirmed</label>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="form-group"><label>Assign to User/Agent</label><select class="form-control" id="user_id" name="user_id"><option value="0" data-name="" selected>-- No Assignment --</option><?php foreach ($users_list as $user): ?><option value="<?= $user['id'] ?>" data-name="<?= e($user['name']) ?>" <?= ($quick_booking_data && $quick_booking_data['user_id'] == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?> - [<?= e(ucfirst($user['user_type'])) ?>]</option><?php endforeach; ?></select></div>
                                    <div class="form-group"><label>Assign to Vendor</label><select class="form-control" id="vendor_id" name="vendor_id"><option value="0" selected>-- No Assignment --</option><?php foreach ($vendors_list as $vendor): ?><option value="<?= $vendor['id'] ?>"><?= e($vendor['name']) ?></option><?php endforeach; ?></select></div>
                                </div>
                                <hr class="my-4">
                                <div class="form-grid grid-cols-2">
                                    <div class="form-group"><label>Hotel Check In Time</label><input type="text" name="hotel_checkin_time" class="form-control" value="04:00 PM"></div>
                                    <div class="form-group"><label>Hotel Check Out Time</label><input type="text" name="hotel_checkout_time" class="form-control" value="02:00 PM"></div>
                                </div>
                                <div class="form-group mt-3"><label>Notes / Terms & Conditions (Urdu)</label><textarea name="notes_urdu" class="form-control" rows="8">آپ کا ہوٹل اور پیکج اس دستاویز میں لکھ دیا گیا ہے۔ اس کے مطابق آپ کو رہائش اور دیگر سہولیات فراہم کی جائیں گی۔
سفری سامان حرمین شریفین کی طرف لے جانا سعودی انتظاميہ کی طرف سے ممنوع ہے، خلاف ورزی کی صورت میں حکومت جرمانہ کر سکتی ہے جس کی آدائيگی آپ کے ذمہ ہو گی۔
نشہ آور اشياء شرعا اور قانونا ممنوع ہيں۔ سعودی عرب ميں منشيات لے جانے کی سزا موت ہے۔
حرمین شریفین کے اندر زمين پر گری پڑی چيز (پرس، موبائل، یا کوئی اور قيمتی چيز) ہرگز نہ اٹھائيں ورنہ آپ مشکل میں پڑ سکتے ہيں۔
جدہ ائير پورٹ پر اميگريشن، سعودی وزارت و سعودی کمپنی کے انتظامات ميں 3 سے 5 گھنٹے لگ سکتے ہيں۔ اس لئيے جدہ ائير پورٹ سے سعودی سم خریديں اور اپنے گھر والوں کو خيريت کی اطلاع ديں۔
عمرہ زائيرین جدہ ائیر پورٹ سے مکہ جانے کیلئے صرف اپنے ويزہ پر موجود شرکہ کی بسوں میں سفر کريں۔
اگر بس سٹاف آپکا پاسپورٹ اپنے پاس رکھ رہا ہو تو بس کی فرنٹ تصویر نمبر پلیٹ کے ساتھ لے کر ابن یوسف گروپ کے سٹاف کو لازمی آگاہ کریں۔
ہوٹل چیک ان ٹائم 4:00 بجے اور ہوٹل چیک آؤٹ ٹائم 2:00 بجے ہے۔ اس کے بعد اگلی Night چارج ہو گی۔ ہوٹل واؤچر کی 4 عدد کاپياں اپنے ہمراہ رکھيں۔
واپسی کی فلائٹ سے 10 گھنٹے قبل مسافر اپنے سامان کے ساتھ ريسيپشن پر موجود رہيں۔ واپسی کی فلائٹ کی معلومات معتمر کی اپنی ذمہ داری ہے۔ مشکل کی صورت ميں سٹاف سے رابطہ کريں۔
سعودی قانون کے مطابق غير مصنف ہوٹلز میں قيام جرم ہے۔ اس ليئے کمپنی کے تصنيف شدہ ہوٹلز کا استعمال کريں۔
(Only Visa Cases) Self Accommodation اور غير تصنيف شدہ ہوٹلز کے استعمال کی وجہ سے اگر وزارت جرمانہ کرتی ہے تو وہ متعلقہ ایجنٹ سے وصول کیا جاۓ گا۔
مدینہ ائیر پورٹ پر اترنے والی فلائٹس مدینہ ہوٹل سے واپس مدینہ ایئر پورٹ کے لیے ٹرانسپورٹ کا بندوبست معتمر خود کرے گا۔</textarea></div>
                            </fieldset>

                            <div class="form-actions mt-4"><button type="submit" class="btn btn-primary btn-lg">Create Voucher & Invoice</button></div>
                        </form>
                    </div>
                </div>
            </main>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-datepicker/1.9.0/js/bootstrap-datepicker.min.js"></script>
    <script>
        const makkahHotels = <?= json_encode($makkah_hotels) ?>;
        const madinahHotels = <?= json_encode($madinah_hotels) ?>;
        
        function removeRow(btn) {
            const row = btn.closest('tr');
            const tbody = row.parentNode;
            
            if (tbody.id === 'hotel-tbody') {
                const parentId = row.id;
                const contactRow = document.querySelector(`tr[data-parent-row="${parentId}"]`);
                if (contactRow) {
                    contactRow.remove();
                }
            }
            row.remove();
        };

        function addMutamerRow() { document.getElementById('mutamers-tbody').insertAdjacentHTML('beforeend', `<tr><td><input type="text" name="mutamer_name[]" class="form-control"></td><td><input type="text" name="mutamer_passport_no[]" class="form-control"></td><td><select name="mutamer_pax_type[]" class="form-select"><option>Adult</option><option>Child</option><option>Infant</option></select></td><td><select name="mutamer_bed_required[]" class="form-select"><option>Yes</option><option>No</option></select></td><td><input type="text" name="mutamer_group_no[]" class="form-control"></td><td><input type="text" name="mutamer_visa_no[]" class="form-control"></td><td><input type="text" name="mutamer_pnr_no[]" class="form-control"></td><td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td></tr>`); }
        
        function addHotelRow() {
            const hotelTbody = document.getElementById('hotel-tbody');
            const roomTypeDefault = "<?= e($quick_booking_data['room_type'] ?? 'Quad') ?>";
            
            const roomTypeDropdown = `
                <select name="hotel_room_type[]" class="form-select">
                    <option value="Sharing" ${roomTypeDefault === 'Sharing' ? 'selected' : ''}>Sharing</option>
                    <option value="Quint" ${roomTypeDefault === 'Quint' ? 'selected' : ''}>Quint</option>
                    <option value="Quad" ${roomTypeDefault === 'Quad' ? 'selected' : ''}>Quad</option>
                    <option value="Triple" ${roomTypeDefault === 'Triple' ? 'selected' : ''}>Triple</option>
                    <option value="Double" ${roomTypeDefault === 'Double' ? 'selected' : ''}>Double</option>
                </select>
            `;
            const uniqueId = 'row-' + Date.now() + Math.random().toString(36).substr(2, 9);

            const newHotelRowHTML = `<tr id="${uniqueId}">
                <td><select name="hotel_city[]" class="form-select" onchange="updateHotelDropdown(this)"><option selected>Makkah</option><option>Madinah</option></select></td>
                <td><select name="hotel_name[]" class="form-select hotel-select2"></select></td>
                <td><input type="text" name="hotel_check_in[]" class="form-control" placeholder="dd/mm/yyyy" autocomplete="off"></td>
                <td><input type="number" name="hotel_nights[]" class="form-control" value="1"></td>
                <td><input type="text" name="hotel_check_out[]" class="form-control" placeholder="dd/mm/yyyy" autocomplete="off"></td>
                <td>${roomTypeDropdown}</td>
                <td><input type="text" name="hotel_meal_plan[]" class="form-control" value="R/O"></td>
                <td><input type="text" name="hotel_confirmation_no[]" class="form-control"></td>
                <td><button type="button" class="btn btn-sm btn-danger" onclick="removeRow(this)">X</button></td>
            </tr>`;
            
            hotelTbody.insertAdjacentHTML('beforeend', newHotelRowHTML);
            const newRow = hotelTbody.lastElementChild;
            initializeSelect2(newRow.querySelector('.hotel-select2'));
            updateHotelDropdown(newRow.querySelector('select[name="hotel_city[]"]'));
            
            const checkInInput = newRow.querySelector('input[name="hotel_check_in[]"]');
            const nightsInput = newRow.querySelector('input[name="hotel_nights[]"]');
            const checkOutInput = newRow.querySelector('input[name="hotel_check_out[]"]');

            $(checkInInput).datepicker({ format: 'dd/mm/yyyy', autoclose: true, todayHighlight: true }).on('changeDate', function() { calculateCheckoutDate(this); });
            $(checkOutInput).datepicker({ format: 'dd/mm/yyyy', autoclose: true, todayHighlight: true }).on('changeDate', function() { calculateNights(this); });
            $(nightsInput).on('input', function() { calculateCheckoutDate(this); });

            const makkahContactsTbody = document.getElementById('makkah-contacts-tbody');
            const newContactRowHTML = `<tr data-parent-row="${uniqueId}">
                <td class="hotel-name-ref">Hotel Not Selected</td>
                <td><input type="text" name="hotel_person_name[]" class="form-control"></td>
                <td><input type="text" name="hotel_call_number[]" class="form-control"></td>
                <td><input type="text" name="hotel_whatsapp_number[]" class="form-control"></td>
            </tr>`;
            makkahContactsTbody.insertAdjacentHTML('beforeend', newContactRowHTML);
        }
        
        function initializeSelect2(element) { $(element).select2({ placeholder: 'Search and select hotel', allowClear: true, width: '100%' }).on('change', function() { updateContactRef(this); }); }
        
        function updateHotelDropdown(citySelect) {
            const accomRow = citySelect.closest('tr'), hotelSelect = accomRow.querySelector('select[name="hotel_name[]"]');
            if ($(hotelSelect).data('select2')) { $(hotelSelect).select2('destroy'); }
            
            const hotels = citySelect.value.toLowerCase() === 'makkah' ? makkahHotels : madinahHotels;
            hotelSelect.innerHTML = '<option value=""></option>';
            hotels.forEach(h => { hotelSelect.innerHTML += `<option value="${h}">${h}</option>`; });
            initializeSelect2(hotelSelect); 
            updateContactRef(hotelSelect);

            const parentId = accomRow.id;
            const contactRow = document.querySelector(`tr[data-parent-row="${parentId}"]`);
            if (contactRow) {
                const targetTbodyId = citySelect.value.toLowerCase() === 'makkah' ? 'makkah-contacts-tbody' : 'madinah-contacts-tbody';
                document.getElementById(targetTbodyId).appendChild(contactRow);
            }
        }

        function formatDate(date) {
            let d = new Date(date), day = '' + d.getDate(), month = '' + (d.getMonth() + 1), year = d.getFullYear();
            if (day.length < 2) day = '0' + day;
            if (month.length < 2) month = '0' + month;
            return [day, month, year].join('/');
        }

        function parseDate(str) {
            if (!/^\d{1,2}\/\d{1,2}\/\d{4}$/.test(str)) return null;
            var parts = str.split('/');
            return new Date(parts[2], parts[1] - 1, parts[0]);
        }
        
        function calculateCheckoutDate(inputElement) {
            const row = inputElement.closest('tr');
            const checkInInput = row.querySelector('input[name="hotel_check_in[]"]');
            const nightsInput = row.querySelector('input[name="hotel_nights[]"]');
            const checkOutInput = row.querySelector('input[name="hotel_check_out[]"]');
            const checkInDate = parseDate(checkInInput.value);
            const nights = parseInt(nightsInput.value, 10);
            if (checkInDate && !isNaN(nights) && nights >= 0) {
                let date = new Date(checkInDate);
                date.setDate(date.getDate() + nights);
                const newCheckoutDate = formatDate(date);
                checkOutInput.value = newCheckoutDate;
                $(checkOutInput).datepicker('update', newCheckoutDate);
            } else {
                checkOutInput.value = '';
                $(checkOutInput).datepicker('update', '');
            }
        }
        
        function calculateNights(inputElement) {
            const row = inputElement.closest('tr');
            const checkInInput = row.querySelector('input[name="hotel_check_in[]"]');
            const nightsInput = row.querySelector('input[name="hotel_nights[]"]');
            const checkOutInput = row.querySelector('input[name="hotel_check_out[]"]');
            const checkInDate = parseDate(checkInInput.value);
            const checkOutDate = parseDate(checkOutInput.value);
            if (checkInDate && checkOutDate) {
                const diff = checkOutDate.getTime() - checkInDate.getTime();
                nightsInput.value = (diff >= 0) ? Math.ceil(diff / (1000 * 60 * 60 * 24)) : 0;
            }
        }
        
        function updateContactRef(hotelSelect) {
            const accomRow = hotelSelect.closest('tr');
            const parentId = accomRow.id;
            const contactsRow = document.querySelector(`tr[data-parent-row="${parentId}"]`);
            if (contactsRow) { 
                contactsRow.querySelector('.hotel-name-ref').textContent = hotelSelect.options[hotelSelect.selectedIndex]?.text || 'Hotel Not Selected'; 
            }
        }

        document.addEventListener('DOMContentLoaded', () => {
            $('#user_id, #vendor_id').select2({ width: '100%' });
            
            $('#user_id').on('change', function() {
                const selectedOption = $(this).find('option:selected');
                const userName = selectedOption.data('name');
                const familyHeadInput = $('#family_head_name');
                if (familyHeadInput.val() === '' || familyHeadInput.data('auto-filled')) {
                    familyHeadInput.val(userName || '');
                    familyHeadInput.data('auto-filled', !!userName);
                }
            });

            $('#family_head_name').on('input', function() { $(this).data('auto-filled', false); });

            if (document.querySelectorAll('#mutamers-tbody tr').length === 0) addMutamerRow();
            if (document.querySelectorAll('#hotel-tbody tr').length === 0) addHotelRow();
        });
    </script>
</body>
</html>