<?php
session_start();
include_once '../db-config.php';

// A helper function for safely echoing output
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- SECURITY CHECK: Ensure user is a logged-in admin ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- 1. SETUP FILTERS (SYNCED WITH ADMIN-LEDGER.PHP) ---
$filter_user_id_raw = $_GET['user_id'] ?? '0';
$filter_vendor_id_raw = $_GET['vendor_id'] ?? '0';
$filter_start_date = $_GET['start_date'] ?? '';
$filter_end_date = $_GET['end_date'] ?? '';

// DEFINE THE COST TYPES ARRAY HERE SO IT'S AVAILABLE TO BOTH SECTIONS
$cost_types = ['pilgrim', 'transport', 'ticket', 'hotel_specific', 'service_specific', 'hotel_main', 'service_main'];

// --- 2. FETCH SELECTED USER/VENDOR DETAILS FOR PRINT HEADER ---
$selected_user_details = null;
if (is_numeric($filter_user_id_raw) && $filter_user_id_raw > 0) {
    $stmt_user = $conn->prepare("SELECT id, name, company_name, mobile_number, logo_path FROM users WHERE id = ?");
    $stmt_user->bind_param("i", $filter_user_id_raw);
    $stmt_user->execute();
    $result_user = $stmt_user->get_result();
    if ($result_user) $selected_user_details = $result_user->fetch_assoc();
    $stmt_user->close();
}

$selected_vendor_details = null;
if (is_numeric($filter_vendor_id_raw) && $filter_vendor_id_raw > 0) {
    $stmt_vendor = $conn->prepare("SELECT id, name FROM vendors WHERE id = ?");
    $stmt_vendor->bind_param("i", $filter_vendor_id_raw);
    $stmt_vendor->execute();
    $result_vendor = $stmt_vendor->get_result();
    if ($result_vendor) $selected_vendor_details = $result_vendor->fetch_assoc();
    $stmt_vendor->close();
}

// --- 3. CALCULATE THE OPENING BALANCE (SYNCED WITH ADMIN-LEDGER.PHP) ---
$opening_balance = 0;
if (!empty($filter_start_date)) {
    $opening_sql_parts = [];
    $opening_params = [];
    $opening_types = '';

    $op_where_receivable_pkg = ["i.issue_date < ?"];
    $op_params_receivable_pkg = [$filter_start_date];
    $op_types_receivable_pkg = 's';
    $op_where_receivable_tkt = ["ti.issue_date < ?"];
    $op_params_receivable_tkt = [$filter_start_date];
    $op_types_receivable_tkt = 's';
    $op_where_cost = ["i.issue_date < ?"];
    $op_params_cost = [$filter_start_date];
    $op_types_cost = 's';
    $op_where_cost_tkt = ["ti.issue_date < ?"];
    $op_params_cost_tkt = [$filter_start_date];
    $op_types_cost_tkt = 's';
    $op_where_pay = ["p.payment_date < ?"];
    $op_params_pay = [$filter_start_date];
    $op_types_pay = 's';

    if (is_numeric($filter_user_id_raw) && $filter_user_id_raw > 0) {
        $op_where_receivable_pkg[] = "i.user_id = ?";
        $op_params_receivable_pkg[] = (int)$filter_user_id_raw;
        $op_types_receivable_pkg .= 'i';
        $op_where_receivable_tkt[] = "ti.user_id = ?";
        $op_params_receivable_tkt[] = (int)$filter_user_id_raw;
        $op_types_receivable_tkt .= 'i';
        $op_where_cost[] = "i.user_id = ?";
        $op_params_cost[] = (int)$filter_user_id_raw;
        $op_types_cost .= 'i';
        $op_where_cost_tkt[] = "ti.user_id = ?";
        $op_params_cost_tkt[] = (int)$filter_user_id_raw;
        $op_types_cost_tkt .= 'i';
        $op_where_pay[] = "p.user_id = ?";
        $op_params_pay[] = (int)$filter_user_id_raw;
        $op_types_pay .= 'i';
    } elseif ($filter_user_id_raw === 'none') {
        $op_where_receivable_pkg[] = "i.user_id IS NULL";
        $op_where_receivable_tkt[] = "ti.user_id IS NULL";
        $op_where_cost[] = "i.user_id IS NULL";
        $op_where_cost_tkt[] = "ti.user_id IS NULL";
        $op_where_pay[] = "p.user_id IS NULL";
    } elseif ($filter_user_id_raw === 'all_vendors') {
        $op_where_receivable_pkg[] = "1=0";
        $op_where_receivable_tkt[] = "1=0";
    }

    if (is_numeric($filter_vendor_id_raw) && $filter_vendor_id_raw > 0) {
        $op_where_cost_tkt[] = "ti.vendor_id = ?";
        $op_params_cost_tkt[] = (int)$filter_vendor_id_raw;
        $op_types_cost_tkt .= 'i';
        $op_where_pay[] = "p.vendor_id = ?";
        $op_params_pay[] = (int)$filter_vendor_id_raw;
        $op_types_pay .= 'i';
    } elseif ($filter_vendor_id_raw === 'none') {
        $op_where_cost_tkt[] = "ti.vendor_id IS NULL";
        $op_where_pay[] = "p.vendor_id IS NULL";
    } elseif ($filter_vendor_id_raw === 'all_users') {
        $op_where_cost_tkt[] = "1=0";
        $op_where_pay[] = "p.vendor_id IS NULL";
    }

    $opening_sql_parts[] = "(SELECT i.grand_total_pkr as debit, 0 as credit FROM invoices i WHERE " . implode(' AND ', $op_where_receivable_pkg) . ")";
    $opening_params = array_merge($opening_params, $op_params_receivable_pkg);
    $opening_types .= $op_types_receivable_pkg;
    $opening_sql_parts[] = "(SELECT ti.grand_total_pkr as debit, 0 as credit FROM ticket_invoices ti WHERE " . implode(' AND ', $op_where_receivable_tkt) . ")";
    $opening_params = array_merge($opening_params, $op_params_receivable_tkt);
    $opening_types .= $op_types_receivable_tkt;
    $opening_sql_parts[] = "(SELECT p.debit_amount as debit, p.credit_amount as credit FROM payments p WHERE " . implode(' AND ', $op_where_pay) . ")";
    $opening_params = array_merge($opening_params, $op_params_pay);
    $opening_types .= $op_types_pay;

    $op_cost_queries = [];
    $base_where_cost_opening = $op_where_cost;
    $base_params_cost_opening = $op_params_cost;
    $base_types_cost_opening = $op_types_cost;
    $vendor_params_opening = [];
    $vendor_types_opening = '';
    if (is_numeric($filter_vendor_id_raw) && $filter_vendor_id_raw > 0) {
        $vendor_params_opening[] = (int)$filter_vendor_id_raw;
        $vendor_types_opening .= 'i';
    }

    foreach ($cost_types as $type) {
        $where = $base_where_cost_opening;
        $params = $base_params_cost_opening;
        $types = $base_types_cost_opening;
        switch ($type) {
            case 'pilgrim':
                $where[] = "i.pilgrims_vendor_id IS NOT NULL AND ip.visa_price_sar_cost > 0";
                if (!empty($vendor_params_opening)) {
                    $where[] = "i.pilgrims_vendor_id = ?";
                    $params[] = $vendor_params_opening[0];
                    $types .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(ip.visa_price_sar_cost * i.exchange_rate) AS credit FROM invoice_pilgrims ip JOIN invoices i ON ip.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
                break;
            case 'transport':
                $where[] = "i.transport_vendor_id IS NOT NULL AND it.total_amount_cost > 0";
                if (!empty($vendor_params_opening)) {
                    $where[] = "i.transport_vendor_id = ?";
                    $params[] = $vendor_params_opening[0];
                    $types .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(it.total_amount_cost) AS credit FROM invoice_transports it JOIN invoices i ON it.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
                break;
            case 'ticket':
                $where[] = "i.tickets_vendor_id IS NOT NULL AND iat.total_amount_cost > 0";
                if (!empty($vendor_params_opening)) {
                    $where[] = "i.tickets_vendor_id = ?";
                    $params[] = $vendor_params_opening[0];
                    $types .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(iat.total_amount_cost) AS credit FROM invoice_airline_tickets iat JOIN invoices i ON iat.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
                break;
            case 'hotel_specific':
                $where[] = "ih.vendor_id IS NOT NULL AND ih.total_sar_cost > 0";
                if (!empty($vendor_params_opening)) {
                    $where[] = "ih.vendor_id = ?";
                    $params[] = $vendor_params_opening[0];
                    $types .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(ih.total_sar_cost * i.exchange_rate) AS credit FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
                break;
            case 'service_specific':
                $where[] = "ios.vendor_id IS NOT NULL AND ios.total_amount_cost > 0";
                if (!empty($vendor_params_opening)) {
                    $where[] = "ios.vendor_id = ?";
                    $params[] = $vendor_params_opening[0];
                    $types .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(ios.total_amount_cost * i.exchange_rate) AS credit FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
                break;
            case 'hotel_main':
                $where_main = array_merge($base_where_cost_opening, ["i.vendor_id IS NOT NULL", "ih.vendor_id IS NULL", "ih.total_sar_cost > 0"]);
                $params_main = $base_params_cost_opening;
                $types_main = $base_types_cost_opening;
                if (!empty($vendor_params_opening)) {
                    $where_main[] = "i.vendor_id = ?";
                    $params_main[] = $vendor_params_opening[0];
                    $types_main .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where_main[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where_main[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(ih.total_sar_cost * i.exchange_rate) AS credit FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE " . implode(' AND ', $where_main) . " GROUP BY i.id)", 'params' => $params_main, 'types' => $types_main];
                break;
            case 'service_main':
                $where_main = array_merge($base_where_cost_opening, ["i.vendor_id IS NOT NULL", "ios.vendor_id IS NULL", "ios.total_amount_cost > 0"]);
                $params_main = $base_params_cost_opening;
                $types_main = $base_types_cost_opening;
                if (!empty($vendor_params_opening)) {
                    $where_main[] = "i.vendor_id = ?";
                    $params_main[] = $vendor_params_opening[0];
                    $types_main .= $vendor_types_opening;
                }
                if ($filter_vendor_id_raw === 'none') $where_main[] = "1=0";
                if ($filter_vendor_id_raw === 'all_users') $where_main[] = "1=0"; // FIX: Exclude vendor data
                $op_cost_queries[] = ['sql' => "(SELECT 0 AS debit, SUM(ios.total_amount_cost * i.exchange_rate) AS credit FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE " . implode(' AND ', $where_main) . " GROUP BY i.id)", 'params' => $params_main, 'types' => $types_main];
                break;
        }
    }
    foreach ($op_cost_queries as $query_info) {
        $opening_sql_parts[] = $query_info['sql'];
        $opening_params = array_merge($opening_params, $query_info['params']);
        $opening_types .= $query_info['types'];
    }

    $op_where_cost_tkt[] = "ti.grand_total_pkr_cost > 0";
    $opening_sql_parts[] = "(SELECT 0 as debit, ti.grand_total_pkr_cost as credit FROM ticket_invoices ti WHERE " . implode(' AND ', $op_where_cost_tkt) . ")";
    $opening_params = array_merge($opening_params, $op_params_cost_tkt);
    $opening_types .= $op_types_cost_tkt;

    $final_sql_opening = "SELECT SUM(debit - credit) as opening_balance FROM (" . implode(" UNION ALL ", $opening_sql_parts) . ") AS opening_parts";
    $stmt_opening = $conn->prepare($final_sql_opening);
    if ($stmt_opening) {
        if (!empty($opening_params)) {
            $stmt_opening->bind_param($opening_types, ...$opening_params);
        }
        $stmt_opening->execute();
        $opening_balance = (float)($stmt_opening->get_result()->fetch_assoc()['opening_balance'] ?? 0);
        $stmt_opening->close();
    }
}

// --- 4. BUILD WHERE CLAUSES & PARAMS for main query ---
$where_receivable_pkg = [];
$params_receivable_pkg = [];
$types_receivable_pkg = '';
$where_receivable_tkt = [];
$params_receivable_tkt = [];
$types_receivable_tkt = '';
$where_cost = [];
$params_cost = [];
$types_cost = '';
$where_cost_tkt = [];
$params_cost_tkt = [];
$types_cost_tkt = '';
$where_pay = [];
$params_pay = [];
$types_pay = '';

if (!empty($filter_start_date)) {
    $where_receivable_pkg[] = "i.issue_date >= ?";
    $params_receivable_pkg[] = $filter_start_date;
    $types_receivable_pkg .= 's';
    $where_receivable_tkt[] = "ti.issue_date >= ?";
    $params_receivable_tkt[] = $filter_start_date;
    $types_receivable_tkt .= 's';
    $where_cost[] = "i.issue_date >= ?";
    $params_cost[] = $filter_start_date;
    $types_cost .= 's';
    $where_cost_tkt[] = "ti.issue_date >= ?";
    $params_cost_tkt[] = $filter_start_date;
    $types_cost_tkt .= 's';
    $where_pay[] = "p.payment_date >= ?";
    $params_pay[] = $filter_start_date;
    $types_pay .= 's';
}
if (!empty($filter_end_date)) {
    $where_receivable_pkg[] = "i.issue_date <= ?";
    $params_receivable_pkg[] = $filter_end_date;
    $types_receivable_pkg .= 's';
    $where_receivable_tkt[] = "ti.issue_date <= ?";
    $params_receivable_tkt[] = $filter_end_date;
    $types_receivable_tkt .= 's';
    $where_cost[] = "i.issue_date <= ?";
    $params_cost[] = $filter_end_date;
    $types_cost .= 's';
    $where_cost_tkt[] = "ti.issue_date <= ?";
    $params_cost_tkt[] = $filter_end_date;
    $types_cost_tkt .= 's';
    $where_pay[] = "p.payment_date <= ?";
    $params_pay[] = $filter_end_date;
    $types_pay .= 's';
}
if (is_numeric($filter_user_id_raw) && $filter_user_id_raw > 0) {
    $where_receivable_pkg[] = "i.user_id = ?";
    $params_receivable_pkg[] = (int)$filter_user_id_raw;
    $types_receivable_pkg .= 'i';
    $where_receivable_tkt[] = "ti.user_id = ?";
    $params_receivable_tkt[] = (int)$filter_user_id_raw;
    $types_receivable_tkt .= 'i';
    $where_cost[] = "i.user_id = ?";
    $params_cost[] = (int)$filter_user_id_raw;
    $types_cost .= 'i';
    $where_cost_tkt[] = "ti.user_id = ?";
    $params_cost_tkt[] = (int)$filter_user_id_raw;
    $types_cost_tkt .= 'i';
    $where_pay[] = "p.user_id = ?";
    $params_pay[] = (int)$filter_user_id_raw;
    $types_pay .= 'i';
} elseif ($filter_user_id_raw === 'none') {
    $where_receivable_pkg[] = "i.user_id IS NULL";
    $where_receivable_tkt[] = "ti.user_id IS NULL";
    $where_cost[] = "i.user_id IS NULL";
    $where_cost_tkt[] = "ti.user_id IS NULL";
    $where_pay[] = "p.user_id IS NULL";
} elseif ($filter_user_id_raw === 'all_vendors') {
    $where_receivable_pkg[] = "1=0";
    $where_receivable_tkt[] = "1=0";
}
if (is_numeric($filter_vendor_id_raw) && $filter_vendor_id_raw > 0) {
    $where_cost_tkt[] = "ti.vendor_id = ?";
    $params_cost_tkt[] = (int)$filter_vendor_id_raw;
    $types_cost_tkt .= 'i';
    $where_pay[] = "p.vendor_id = ?";
    $params_pay[] = (int)$filter_vendor_id_raw;
    $types_pay .= 'i';
} elseif ($filter_vendor_id_raw === 'none') {
    $where_cost_tkt[] = "ti.vendor_id IS NULL";
    $where_pay[] = "p.vendor_id IS NULL";
} elseif ($filter_vendor_id_raw === 'all_users') {
    $where_cost_tkt[] = "1=0";
    $where_pay[] = "p.vendor_id IS NULL";
}

$where_receivable_pkg_str = !empty($where_receivable_pkg) ? 'WHERE ' . implode(' AND ', $where_receivable_pkg) : '';
$where_receivable_tkt_str = !empty($where_receivable_tkt) ? 'WHERE ' . implode(' AND ', $where_receivable_tkt) : '';
$where_cost_tkt_str = !empty($where_cost_tkt) ? 'WHERE ' . implode(' AND ', $where_cost_tkt) : '';
$where_pay_str = !empty($where_pay) ? 'WHERE ' . implode(' AND ', $where_pay) : '';

// --- 5. FETCH TRANSACTIONS ---
$sql_parts = [];
$params_period = [];
$types_period = '';
$collation_fix = "COLLATE utf8mb4_unicode_ci";

$sql_parts[] = "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Booking' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT(i.guest_name, ' x ', (SELECT COUNT(*) FROM invoice_pilgrims ip WHERE ip.invoice_id = i.id), ' Pax') AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, i.grand_total_pkr AS debit, 0 AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoices i $where_receivable_pkg_str)";
$params_period = array_merge($params_period, $params_receivable_pkg);
$types_period .= $types_receivable_pkg;

$cost_queries_period = [];
$base_where_cost_period = $where_cost;
$base_params_cost_period = $params_cost;
$base_types_cost_period = $types_cost;
$vendor_params_period = [];
$vendor_types_period = '';
if (is_numeric($filter_vendor_id_raw) && $filter_vendor_id_raw > 0) {
    $vendor_params_period[] = (int)$filter_vendor_id_raw;
    $vendor_types_period .= 'i';
}

foreach ($cost_types as $type) {
    $where = $base_where_cost_period;
    $params = $base_params_cost_period;
    $types = $base_types_cost_period;
    switch ($type) {
        case 'pilgrim':
            $where[] = "i.pilgrims_vendor_id IS NOT NULL AND ip.visa_price_sar_cost > 0";
            if (!empty($vendor_params_period)) {
                $where[] = "i.pilgrims_vendor_id = ?";
                $params[] = $vendor_params_period[0];
                $types .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Visa Cost x ', COUNT(ip.id)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(ip.visa_price_sar_cost * i.exchange_rate) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_pilgrims ip JOIN invoices i ON ip.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
            break;
        case 'transport':
            $where[] = "i.transport_vendor_id IS NOT NULL AND it.total_amount_cost > 0";
            if (!empty($vendor_params_period)) {
                $where[] = "i.transport_vendor_id = ?";
                $params[] = $vendor_params_period[0];
                $types .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Transport Cost x ', SUM(it.qty)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(it.total_amount_cost) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_transports it JOIN invoices i ON it.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
            break;
        case 'ticket':
            $where[] = "i.tickets_vendor_id IS NOT NULL AND iat.total_amount_cost > 0";
            if (!empty($vendor_params_period)) {
                $where[] = "i.tickets_vendor_id = ?";
                $params[] = $vendor_params_period[0];
                $types .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Ticket Cost (Pkg) x ', SUM(iat.adult_qty+iat.child_qty+iat.infant_qty)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(iat.total_amount_cost) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_airline_tickets iat JOIN invoices i ON iat.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
            break;
        case 'hotel_specific':
            $where[] = "ih.vendor_id IS NOT NULL AND ih.total_sar_cost > 0";
            if (!empty($vendor_params_period)) {
                $where[] = "ih.vendor_id = ?";
                $params[] = $vendor_params_period[0];
                $types .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Hotel Cost x ', SUM(ih.rooms)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(ih.total_sar_cost * i.exchange_rate) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
            break;
        case 'service_specific':
            $where[] = "ios.vendor_id IS NOT NULL AND ios.total_amount_cost > 0";
            if (!empty($vendor_params_period)) {
                $where[] = "ios.vendor_id = ?";
                $params[] = $vendor_params_period[0];
                $types .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Service Cost x ', COUNT(ios.id)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(ios.total_amount_cost * i.exchange_rate) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE " . implode(' AND ', $where) . " GROUP BY i.id)", 'params' => $params, 'types' => $types];
            break;
        case 'hotel_main':
            $where_main = array_merge($base_where_cost_period, ["i.vendor_id IS NOT NULL", "ih.vendor_id IS NULL", "ih.total_sar_cost > 0"]);
            $params_main = $base_params_cost_period;
            $types_main = $base_types_cost_period;
            if (!empty($vendor_params_period)) {
                $where_main[] = "i.vendor_id = ?";
                $params_main[] = $vendor_params_period[0];
                $types_main .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where_main[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where_main[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Hotel Cost (Main) x ', SUM(ih.rooms)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(ih.total_sar_cost * i.exchange_rate) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id WHERE " . implode(' AND ', $where_main) . " GROUP BY i.id)", 'params' => $params_main, 'types' => $types_main];
            break;
        case 'service_main':
            $where_main = array_merge($base_where_cost_period, ["i.vendor_id IS NOT NULL", "ios.vendor_id IS NULL", "ios.total_amount_cost > 0"]);
            $params_main = $base_params_cost_period;
            $types_main = $base_types_cost_period;
            if (!empty($vendor_params_period)) {
                $where_main[] = "i.vendor_id = ?";
                $params_main[] = $vendor_params_period[0];
                $types_main .= $vendor_types_period;
            }
            if ($filter_vendor_id_raw === 'none') $where_main[] = "1=0";
            if ($filter_vendor_id_raw === 'all_users') $where_main[] = "1=0"; // FIX: Exclude vendor data
            $cost_queries_period[] = ['sql' => "(SELECT i.id AS original_id, i.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(i.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Service Cost (Main) x ', COUNT(ios.id)) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, SUM(ios.total_amount_cost * i.exchange_rate) AS credit, i.id AS link_id, CAST('package' AS CHAR(50)) $collation_fix AS link_type FROM invoice_other_services ios JOIN invoices i ON ios.invoice_id = i.id WHERE " . implode(' AND ', $where_main) . " GROUP BY i.id)", 'params' => $params_main, 'types' => $types_main];
            break;
    }
}
foreach ($cost_queries_period as $query_info) {
    $sql_parts[] = $query_info['sql'];
    $params_period = array_merge($params_period, $query_info['params']);
    $types_period .= $query_info['types'];
}

$sql_parts[] = "(SELECT ti.id AS original_id, ti.issue_date AS transaction_date, CAST('Ticket' AS CHAR(50)) $collation_fix AS transaction_type, CAST(ti.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT(ti.guest_name, ' x ', (SELECT COUNT(*) FROM ticket_invoice_passengers tip WHERE tip.ticket_invoice_id = ti.id), ' Pax') AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, ti.grand_total_pkr AS debit, 0 AS credit, ti.id AS link_id, CAST('ticket' AS CHAR(50)) $collation_fix AS link_type FROM ticket_invoices ti $where_receivable_tkt_str)";
$params_period = array_merge($params_period, $params_receivable_tkt);
$types_period .= $types_receivable_tkt;

$where_cost_tkt_str .= (empty($where_cost_tkt) ? 'WHERE ' : ' AND ') . "ti.grand_total_pkr_cost > 0";
$sql_parts[] = "(SELECT ti.id AS original_id, ti.issue_date AS transaction_date, CAST('Vendor Cost' AS CHAR(50)) $collation_fix AS transaction_type, CAST(ti.invoice_number AS CHAR(255)) $collation_fix AS trans_num, CAST(CONCAT('Cost for Ticket Inv: ', ti.guest_name) AS CHAR(255)) $collation_fix AS particulars, CAST('' AS CHAR(255)) $collation_fix AS invoice_reference, 0 AS debit, ti.grand_total_pkr_cost AS credit, ti.id AS link_id, CAST('ticket' AS CHAR(50)) $collation_fix AS link_type FROM ticket_invoices ti $where_cost_tkt_str)";
$params_period = array_merge($params_period, $params_cost_tkt);
$types_period .= $types_cost_tkt;

$sql_parts[] = "(SELECT p.id AS original_id, p.payment_date AS transaction_date, 
    CAST(CASE WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BP' WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN 'CP' WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN 'BR' WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN 'CR' WHEN p.debit_amount > 0 THEN 'Expense / Paid' ELSE 'Payment Received' END AS CHAR(50)) $collation_fix AS transaction_type, 
    CAST(CASE WHEN p.debit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN CONCAT('BP-', p.id) WHEN p.debit_amount > 0 AND p.payment_method = 'Cash' THEN CONCAT('CP-', p.id) WHEN p.credit_amount > 0 AND p.payment_method IN ('Bank Transfer', 'Card') THEN CONCAT('BR-', p.id) WHEN p.credit_amount > 0 AND p.payment_method = 'Cash' THEN CONCAT('CR-', p.id) ELSE CONCAT('PAY-', p.id) END AS CHAR(255)) $collation_fix AS trans_num, 
    CAST(p.notes AS CHAR(255)) $collation_fix AS particulars, CAST(p.invoice_reference AS CHAR(255)) $collation_fix AS invoice_reference, p.debit_amount AS debit, p.credit_amount AS credit, p.invoice_id AS link_id, CAST(p.invoice_type AS CHAR(50)) $collation_fix AS link_type FROM payments p $where_pay_str)";
$params_period = array_merge($params_period, $params_pay);
$types_period .= $types_pay;

$final_sql = implode(" UNION ALL ", $sql_parts) . " ORDER BY transaction_date ASC, original_id ASC, credit ASC, debit ASC";
$transactions_raw = [];
$stmt_period = $conn->prepare($final_sql);
if ($stmt_period) {
    if (!empty($params_period)) {
        $stmt_period->bind_param($types_period, ...$params_period);
    }
    $stmt_period->execute();
    $result = $stmt_period->get_result();
    if ($result) {
        $transactions_raw = $result->fetch_all(MYSQLI_ASSOC);
    }
    $stmt_period->close();
}

// --- 6. PROCESS DATA FOR DISPLAY ---
$transactions = [];
$running_balance = $opening_balance;
$total_receivables_period = 0;
$total_payments_received_period = 0;
$total_expenses_paid_period = 0;
$total_costs_incurred_period = 0;
$total_debit_period = 0;
$total_credit_period = 0;

foreach ($transactions_raw as $transaction) {
    // Now using associative keys because we added aliases to the SQL
    $debit = (float)$transaction['debit'];
    $credit = (float)$transaction['credit'];

    if (strpos($transaction['transaction_type'], 'Booking') !== false || strpos($transaction['transaction_type'], 'Ticket') !== false && strpos($transaction['transaction_type'], 'Cost') === false) {
        $total_receivables_period += $debit;
    } elseif (strpos($transaction['transaction_type'], 'Cost') !== false) {
        $total_costs_incurred_period += $credit;
    } elseif ($debit > 0) {
        $total_expenses_paid_period += $debit;
    } else {
        $total_payments_received_period += $credit;
    }

    $total_debit_period += $debit;
    $total_credit_period += $credit;
    $running_balance += $debit - $credit;
    $transaction['balance'] = $running_balance;
    $transactions[] = $transaction;
}
$closing_balance = $running_balance;
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Ledger Statement</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <style>
        :root {
            --theme-color: #f0f0f0;
            --border-color: #ccc;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background-color: #e9e9e9;
            margin: 0;
            padding: 20px;
            font-size: 10pt;
            color: #333;
        }

        .actions-bar {
            max-width: 1100px;
            margin: 0 auto 15px auto;
            display: flex;
            justify-content: flex-end;
            gap: 10px;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            color: white;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }

        .btn-print {
            background-color: #0d2d4c;
        }

        .print-wrapper {
            max-width: 1100px;
            margin: 0 auto;
            padding: 30px;
            border: 1px solid #ccc;
            background-color: #fff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        td,
        th {
            padding: 4px;
            vertical-align: top;
        }

        .header-table td {
            border: none;
            padding: 0;
        }

        .agent-logo {
            width: 33%;
            height: 80px;
            text-align: left;
        }

        .agent-logo img {
            max-height: 80px;
            max-width: 180px;
        }

        .company-logo-container {
            width: 34%;
            text-align: center;
        }

        .company-logo-container img {
            max-height: 50px;
        }

        .company-details {
            font-size: 9pt;
            line-height: 1.4;
            padding-top: 5px;
        }

        .statement-meta {
            width: 33%;
        }

        .statement-meta table {
            border: 1px solid var(--border-color);
        }

        .statement-meta td {
            padding: 5px 8px;
            font-size: 9pt;
        }

        .statement-meta td:first-child {
            font-weight: bold;
            background-color: var(--theme-color);
            width: 100px;
        }

        .customer-details {
            border: 1px solid var(--border-color);
            margin-top: 20px;
            padding: 15px;
            background: #fafafa;
            border-radius: 5px;
        }

        .customer-details h3 {
            margin: 0 0 10px 0;
            font-size: 12pt;
            border-bottom: 1px solid #eee;
            padding-bottom: 5px;
        }

        .customer-details table td {
            padding: 3px 0;
        }

        .customer-details table td:first-child {
            font-weight: bold;
            width: 120px;
        }

        .summary-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
            gap: 1rem;
            margin: 20px 0;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e0e0e0;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .summary-item .value {
            font-size: 1.5em;
            font-weight: 600;
        }

        .summary-item .receivable {
            color: #007bff;
        }

        .summary-item .cost {
            color: #fd7e14;
        }

        .summary-item .payment {
            color: #28a745;
        }

        .summary-item .expense {
            color: #dc3545;
        }

        .summary-item .balance {
            color: #17a2b8;
        }

        .summary-item .profit {
            color: #6f42c1;
        }

        .ledger-table th {
            background-color: var(--theme-color);
            border: 1px solid var(--border-color);
            padding: 8px;
            text-align: left;
            font-weight: 600;
        }

        .ledger-table td {
            border: 1px solid var(--border-color);
            padding: 7px;
            vertical-align: middle;
        }

        .ledger-table td.number {
            text-align: right;
            font-family: 'SFMono-Regular', Consolas, 'Liberation Mono', Menlo, Courier, monospace;
        }

        .ledger-table td.closing-balance {
            text-align: right;
            font-family: 'SFMono-Regular', Consolas, 'Liberation Mono', Menlo, Courier, monospace;
        }

        .ledger-table .particulars {
            white-space: normal;
            word-break: break-word;
        }

        .ledger-table .balance-row,
        .ledger-table .totals-row {
            font-weight: bold;
            background-color: #f9f9f9;
        }

        .footer {
            text-align: center;
            margin-top: 30px;
            font-size: 9pt;
            color: #777;
            border-top: 1px solid #eee;
            padding-top: 15px;
        }

        @media print {
            body {
                background-color: #fff;
                margin: 0;
                padding: 0;
                font-size: 9pt;
            }

            .actions-bar {
                display: none;
            }

            .print-wrapper {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
                max-width: 100%;
            }

            * {
                color-adjust: exact !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
        }
    </style>
</head>

<body>
    <div class="actions-bar">
        <a href="javascript:window.print()" class="btn btn-print">Print Statement</a>
    </div>
    <div class="print-wrapper" id="invoice-to-print">
        <header>
            <table class="header-table">
                <tr>
                    <td class="agent-logo"><?php if (!empty($selected_user_details['logo_path'])): ?><img src="../uploads/logos/<?= e($selected_user_details['logo_path']) ?>" alt="Agent Logo"><?php endif; ?></td>
                    <td class="company-logo-container"><img src="../images/logo.png" alt="Company Logo">
                        <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan<br>Mob: 0092 305 23 94 810, 0092 305 23 94 810 UAN</div>
                    </td>
                    <td class="statement-meta">
                        <table>
                            <tr>
                                <td colspan="2" style="text-align:center; font-weight:bold; font-size: 14pt; background: #333; color: #fff;">Ledger Statement</td>
                            </tr>
                            <tr>
                                <td>Statement Date:</td>
                                <td><?= date('d M, Y') ?></td>
                            </tr>
                            <tr>
                                <td>Period:</td>
                                <td><?= !empty($filter_start_date) || !empty($filter_end_date) ? e(date('d M Y', strtotime($filter_start_date))) . ' to ' . e(date('d M Y', strtotime($filter_end_date))) : 'All Time' ?></td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </table>
        </header>

        <?php if ($selected_user_details || $selected_vendor_details): ?>
            <section class="customer-details">
                <h3>Statement For:</h3>
                <table>
                    <?php if ($selected_user_details): ?>
                        <tr>
                            <td>Name:</td>
                            <td><?= e($selected_user_details['name']) ?></td>
                        </tr>
                        <?php if (!empty($selected_user_details['company_name'])): ?><tr>
                                <td>Company:</td>
                                <td><?= e($selected_user_details['company_name']) ?></td>
                            </tr><?php endif; ?>
                        <?php if (!empty($selected_user_details['mobile_number'])): ?><tr>
                                <td>Mobile:</td>
                                <td><?= e($selected_user_details['mobile_number']) ?></td>
                            </tr><?php endif; ?>
                    <?php elseif ($selected_vendor_details): ?>
                        <tr>
                            <td>Vendor:</td>
                            <td><?= e($selected_vendor_details['name']) ?></td>
                        </tr>
                    <?php endif; ?>
                </table>
            </section>
        <?php else: ?>
            <section class="customer-details">
                <h3>General Ledger Statement</h3>
                <table>
                    <tr>
                        <td>Filters Applied:</td>
                        <td>
                            <?php
                            $filters_applied = [];
                            if ($filter_user_id_raw === 'all_vendors') $filters_applied[] = 'User: All Vendors';
                            elseif ($filter_user_id_raw === 'none') $filters_applied[] = 'User: Direct Customers';
                            elseif ($filter_user_id_raw === '0') $filters_applied[] = 'User: All';
                            if ($filter_vendor_id_raw === 'all_users') $filters_applied[] = 'Vendor: All Users';
                            elseif ($filter_vendor_id_raw === 'none') $filters_applied[] = 'Vendor: No Vendor';
                            elseif ($filter_vendor_id_raw === '0') $filters_applied[] = 'Vendor: All';
                            echo empty($filters_applied) ? 'None' : implode(', ', $filters_applied);
                            ?>
                        </td>
                    </tr>
                </table>
            </section>
        <?php endif; ?>

        <section class="summary-container">
            <div class="summary-item">
                <div class="label">Opening Balance</div>
                <div class="value balance"><?= number_format($opening_balance, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Company Receivables</div>
                <div class="value receivable"><?= number_format($total_receivables_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Payments Received</div>
                <div class="value payment"><?= number_format($total_payments_received_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Closing Balance</div>
                <div class="value balance"><?= number_format($closing_balance, 2) ?></div>
            </div>

            <div class="summary-item">
                <div class="label">Company Payable</div>
                <div class="value cost"><?= number_format($total_costs_incurred_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Payments Made</div>
                <div class="value expense"><?= number_format($total_expenses_paid_period, 2) ?></div>
            </div>

            <div class="summary-item">
                <div class="label">Net Profit</div>
                <div class="value profit"><?= number_format($total_receivables_period - $total_costs_incurred_period, 2) ?></div>
            </div>
        </section>

        <main>
            <table class="ledger-table">
                <thead>
                    <tr>
                        <th style="width: 8%; text-align:center;">Date</th>
                        <th style="width: 10%; text-align:center;">Type</th>
                        <th style="width: 8%; text-align:center;">Trans.#</th>
                        <th style="text-align:center;">Particulars</th>
                        <th style="width: 6%; text-align:center;">Inv/Ref</th>
                        <th class="number" style="width: 10%; text-align:center;">Debit</th>
                        <th class="number" style="width: 10%; text-align:center;">Credit</th>
                        <th class="number" style="width: 10%; text-align:center;">Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($filter_start_date)): ?>
                        <tr class="balance-row">
                            <td colspan="7"><strong>Opening Balance</strong></td>
                            <td class="number"><strong><?= number_format($opening_balance, 2) ?></strong></td>
                        </tr>
                    <?php endif; ?>
                    <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center; padding: 20px;">No transactions found in the selected period.</td>
                        </tr>
                        <?php else:
                        foreach ($transactions as $transaction):
                        ?>
                            <tr>
                                <td style="text-align:center;"><?= date('d M, y', strtotime(e($transaction['transaction_date']))) ?></td>
                                <td><?= e($transaction['transaction_type']) ?></td>
                                <td><?= e($transaction['trans_num']) ?></td>
                                <td class="particulars"><?= e($transaction['particulars']) ?></td>
                                <td style="text-align:center;"><?= e($transaction['invoice_reference']) ?></td>
                                <td class="number"><?= (float)$transaction['debit'] > 0 ? number_format((float)$transaction['debit'], 2) : '' ?></td>
                                <td class="number"><?= (float)$transaction['credit'] > 0 ? number_format((float)$transaction['credit'], 2) : '' ?></td>
                                <td class="number closing-balance"><?= number_format((float)$transaction['balance'], 2) ?></td>
                            </tr>
                    <?php endforeach;
                    endif; ?>
                    <tr class="totals-row">
                        <td colspan="5" style="text-align:right;"><strong>Period Totals</strong></td>
                        <td class="number"><strong><?= number_format($total_debit_period, 2) ?></strong></td>
                        <td class="number"><strong><?= number_format($total_credit_period, 2) ?></strong></td>
                        <td></td>
                    </tr>
                    <tr class="balance-row">
                        <td colspan="7" style="text-align:right;"><strong>Closing Balance</strong></td>
                        <td class="number"><strong><?= number_format($closing_balance, 2) ?></strong></td>
                    </tr>
                </tbody>
            </table>
        </main>
        <footer class="footer">This is a computer-generated statement and does not require a signature.</footer>
    </div>


    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>