<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
include_once '../db-config.php';

//======================================================================
//  SECURITY GATEWAY
//======================================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || ($_SESSION['user_type'] !== 'admin' && $_SESSION['user_type'] !== 'agent')) {
    header("location: ../login.php");
    exit;
}

// Helper function for secure output
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

$edit_mode = false;
$edit_type = null;
$edit_data = null;

if (isset($_GET['edit_id']) && isset($_GET['type'])) {
    $edit_id = (int)$_GET['edit_id'];
    $edit_type = $_GET['type'];
    $table_map = [
        'hotel' => 'custom_package_hotel_rates',
        'transport' => 'custom_package_transport_rates',
        'ziyarat' => 'custom_package_ziyarat_rates',
        'visa' => 'custom_package_visa_rates'
    ];
    if (array_key_exists($edit_type, $table_map)) {
        $stmt = $conn->prepare("SELECT * FROM {$table_map[$edit_type]} WHERE id = ?");
        $stmt->bind_param("i", $edit_id);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $edit_data = $result->fetch_assoc();
            $edit_mode = true;
        }
        $stmt->close();
    }
}

//======================================================================
//  HANDLE FORM SUBMISSIONS
//======================================================================
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {
    $action = $_POST['action'];
    $id = $_POST['id'] ?? null;

    try {
        // --- RATE LIST SETTINGS ---
        if ($action === 'update_rate_list_settings') {
            $settings_to_update = [
                'rate_list_main_title', 'rate_list_subtitle', 'rate_list_vouchers_from', 
                'rate_list_check_till', 'rate_list_instructions', 'rate_list_header_bg', 
                'rate_list_header_text', 'rate_list_table_header_bg', 'rate_list_table_header_text', 
                'rate_list_accent_color', 'rate_list_font', 'rate_list_header_fs',
                'rate_list_table_head_fs', 'rate_list_table_body_fs'
            ];
            $stmt = $conn->prepare("INSERT INTO custom_package_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
            foreach ($settings_to_update as $key) {
                if (isset($_POST[$key])) {
                    $stmt->bind_param("ss", $key, $_POST[$key]);
                    $stmt->execute();
                }
            }
            $stmt->close();

            // Handle Logo Upload
            if (isset($_FILES['rate_list_logo']) && $_FILES['rate_list_logo']['error'] == 0) {
                $target_dir = "../uploads/logos/";
                if (!file_exists($target_dir)) mkdir($target_dir, 0777, true);
                $file_name = "rate_list_logo_" . time() . basename($_FILES["rate_list_logo"]["name"]);
                $target_file = $target_dir . $file_name;
                if (move_uploaded_file($_FILES["rate_list_logo"]["tmp_name"], $target_file)) {
                    $logo_path = "uploads/logos/" . $file_name;
                    $key = 'rate_list_logo';
                    $stmt = $conn->prepare("INSERT INTO custom_package_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
                    $stmt->bind_param("ss", $key, $logo_path);
                    $stmt->execute();
                    $stmt->close();
                } else { throw new Exception("Error uploading logo."); }
            }
            
            // Handle Background Image Upload
            if (isset($_FILES['rate_list_bg_image']) && $_FILES['rate_list_bg_image']['error'] == 0) {
                $target_dir = "../uploads/backgrounds/";
                if (!file_exists($target_dir)) mkdir($target_dir, 0777, true);
                $file_name = "rate_list_bg_" . time() . basename($_FILES["rate_list_bg_image"]["name"]);
                $target_file = $target_dir . $file_name;
                if (move_uploaded_file($_FILES["rate_list_bg_image"]["tmp_name"], $target_file)) {
                    $bg_path = "uploads/backgrounds/" . $file_name;
                    $key = 'rate_list_bg_image';
                    $stmt = $conn->prepare("INSERT INTO custom_package_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");
                    $stmt->bind_param("ss", $key, $bg_path);
                    $stmt->execute();
                    $stmt->close();
                } else { throw new Exception("Error uploading background image."); }
            }

            $_SESSION['success_message'] = "Rate list settings updated successfully.";
            header("Location: manage-custom-package-rates.php#rate-list-customization");
            exit();
        }

        // --- BATCH DELETE ACTIONS ---
        if (strpos($action, 'delete_selected_') === 0) {
            $type_to_delete = str_replace('delete_selected_', '', $action);
            $table_map = [
                'hotels' => 'custom_package_hotel_rates',
                'transports' => 'custom_package_transport_rates',
                'ziyarats' => 'custom_package_ziyarat_rates',
                'visas' => 'custom_package_visa_rates'
            ];
            $ids_to_delete = $_POST['selected_ids'] ?? [];
            if (!empty($ids_to_delete) && isset($table_map[$type_to_delete])) {
                $table_name = $table_map[$type_to_delete];
                $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
                $types = str_repeat('i', count($ids_to_delete));
                $stmt = $conn->prepare("DELETE FROM {$table_name} WHERE id IN ({$placeholders})");
                $stmt->bind_param($types, ...$ids_to_delete);
                $stmt->execute();
                $_SESSION['success_message'] = "Selected items deleted successfully.";
            } else {
                $_SESSION['error_message'] = "No items selected for deletion.";
            }
            header("Location: manage-custom-package-rates.php");
            exit();
        }

        // --- OTHER ACTIONS ---
        switch ($action) {
            case 'update_settings':
                $stmt = $conn->prepare("INSERT INTO custom_package_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
                $stmt->bind_param("sss", $_POST['setting_key'], $_POST['sar_to_pkr_rate'], $_POST['sar_to_pkr_rate']);
                break;
                
            case 'add_visa': case 'update_visa':
                $stmt = $conn->prepare($action === 'add_visa' ? "INSERT INTO custom_package_visa_rates (visa_type, duration_days, rate_per_pax_1, rate_per_pax_2, rate_per_pax_3, rate_per_pax_4, rate_per_pax_5_plus) VALUES (?, ?, ?, ?, ?, ?, ?)" : "UPDATE custom_package_visa_rates SET visa_type=?, duration_days=?, rate_per_pax_1=?, rate_per_pax_2=?, rate_per_pax_3=?, rate_per_pax_4=?, rate_per_pax_5_plus=? WHERE id=?");
                $pax1 = !empty($_POST['rate_per_pax_1']) ? (float)$_POST['rate_per_pax_1'] : null; $pax2 = !empty($_POST['rate_per_pax_2']) ? (float)$_POST['rate_per_pax_2'] : null; $pax3 = !empty($_POST['rate_per_pax_3']) ? (float)$_POST['rate_per_pax_3'] : null; $pax4 = !empty($_POST['rate_per_pax_4']) ? (float)$_POST['rate_per_pax_4'] : null; $pax5plus = !empty($_POST['rate_per_pax_5_plus']) ? (float)$_POST['rate_per_pax_5_plus'] : null;
                if ($action === 'add_visa') { $stmt->bind_param("siddddd", $_POST['visa_type'], $_POST['duration_days'], $pax1, $pax2, $pax3, $pax4, $pax5plus); } else { $stmt->bind_param("sidddddi", $_POST['visa_type'], $_POST['duration_days'], $pax1, $pax2, $pax3, $pax4, $pax5plus, $id); }
                break;

            case 'add_hotel': case 'update_hotel':
                $stmt = $conn->prepare($action === 'add_hotel' ? "INSERT INTO custom_package_hotel_rates (hotel_name, city, star_rating, distance_info, notes, rate_sharing, rate_quad, rate_trpl, rate_dubl, rate_room_basis, currency, rate_list_type) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)" : "UPDATE custom_package_hotel_rates SET hotel_name=?, city=?, star_rating=?, distance_info=?, notes=?, rate_sharing=?, rate_quad=?, rate_trpl=?, rate_dubl=?, rate_room_basis=?, currency=?, rate_list_type=? WHERE id=?");
                $rate_sharing = !empty($_POST['rate_sharing']) ? (float)$_POST['rate_sharing'] : null; $rate_quad = !empty($_POST['rate_quad']) ? (float)$_POST['rate_quad'] : null; $rate_trpl = !empty($_POST['rate_trpl']) ? (float)$_POST['rate_trpl'] : null; $rate_dubl = !empty($_POST['rate_dubl']) ? (float)$_POST['rate_dubl'] : null; $rate_room = !empty($_POST['rate_room_basis']) ? (float)$_POST['rate_room_basis'] : null;
                if ($action === 'add_hotel') { $stmt->bind_param("sssssdddddss", $_POST['hotel_name'], $_POST['city'], $_POST['star_rating'], $_POST['distance_info'], $_POST['notes'], $rate_sharing, $rate_quad, $rate_trpl, $rate_dubl, $rate_room, $_POST['currency'], $_POST['rate_list_type']); } else { $stmt->bind_param("sssssdddddssi", $_POST['hotel_name'], $_POST['city'], $_POST['star_rating'], $_POST['distance_info'], $_POST['notes'], $rate_sharing, $rate_quad, $rate_trpl, $rate_dubl, $rate_room, $_POST['currency'], $_POST['rate_list_type'], $id); }
                break;

            case 'add_transport': case 'update_transport':
                $is_round_trip = isset($_POST['is_round_trip']) ? 1 : 0;
                $stmt = $conn->prepare($action === 'add_transport' ? "INSERT INTO custom_package_transport_rates (sector_name, vehicle_type, is_round_trip, trip_details, rate_sar) VALUES (?, ?, ?, ?, ?)" : "UPDATE custom_package_transport_rates SET sector_name = ?, vehicle_type = ?, is_round_trip = ?, trip_details = ?, rate_sar = ? WHERE id = ?");
                $rate_sar = (float) $_POST['rate_sar'];
                if ($action === 'add_transport') { $stmt->bind_param("ssisd", $_POST['sector_name'], $_POST['vehicle_type'], $is_round_trip, $_POST['trip_details'], $rate_sar); } else { $stmt->bind_param("ssisdi", $_POST['sector_name'], $_POST['vehicle_type'], $is_round_trip, $_POST['trip_details'], $rate_sar, $id); }
                break;
            case 'update_transport_order':
                $order = json_decode($_POST['order']); $stmt = $conn->prepare("UPDATE custom_package_transport_rates SET display_order = ? WHERE id = ?");
                foreach ($order as $index => $itemId) { $stmt->bind_param("ii", $index, $itemId); $stmt->execute(); }
                echo json_encode(['status' => 'success']); exit();

            case 'add_ziyarat': case 'update_ziyarat':
                $stmt = $conn->prepare($action === 'add_ziyarat' ? "INSERT INTO custom_package_ziyarat_rates (city, vehicle_type, rate_sar) VALUES (?, ?, ?)" : "UPDATE custom_package_ziyarat_rates SET city = ?, vehicle_type = ?, rate_sar = ? WHERE id = ?");
                $rate_sar = (float) $_POST['rate_sar'];
                if ($action === 'add_ziyarat') { $stmt->bind_param("ssd", $_POST['city'], $_POST['vehicle_type'], $rate_sar); } else { $stmt->bind_param("ssdi", $_POST['city'], $_POST['vehicle_type'], $rate_sar, $id); }
                break;
            
            default: throw new Exception("Invalid action specified.");
        }
        
        if (isset($stmt) && $stmt->execute()) {
            $_SESSION['success_message'] = "Operation completed successfully.";
        } else {
            if(isset($stmt)) throw new Exception("Database operation failed: " . $stmt->error);
        }
        if(isset($stmt)) $stmt->close();

    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error: " . $e->getMessage();
    }
    header("Location: manage-custom-package-rates.php"); exit();
}

//======================================================================
//  FETCH DATA FOR DISPLAY
//======================================================================
$settings_result = $conn->query("SELECT * FROM custom_package_settings");
$settings = [];
while($row = $settings_result->fetch_assoc()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}
$visa_rates_result = $conn->query("SELECT * FROM custom_package_visa_rates ORDER BY duration_days");
$transport_rates_result = $conn->query("SELECT * FROM custom_package_transport_rates ORDER BY display_order ASC, sector_name, vehicle_type");
$ziyarat_rates_result = $conn->query("SELECT * FROM custom_package_ziyarat_rates ORDER BY city, vehicle_type");
$all_hotels_result = $conn->query("SELECT * FROM custom_package_hotel_rates ORDER BY rate_list_type, city, hotel_name");

$rates1_makkah_hotels = $rates1_madinah_hotels = [];
$rates2_makkah_hotels = $rates2_madinah_hotels = [];

if ($all_hotels_result) {
    while ($hotel = $all_hotels_result->fetch_assoc()) {
        if ($hotel['rate_list_type'] === 'Rates 1') {
            if ($hotel['city'] === 'Makkah') $rates1_makkah_hotels[] = $hotel;
            elseif ($hotel['city'] === 'Madinah') $rates1_madinah_hotels[] = $hotel;
        } elseif ($hotel['rate_list_type'] === 'Rates 2') {
            if ($hotel['city'] === 'Makkah') $rates2_makkah_hotels[] = $hotel;
            elseif ($hotel['city'] === 'Madinah') $rates2_madinah_hotels[] = $hotel;
        }
    }
} else { 
    $_SESSION['error_message'] = "Could not fetch hotel rates. Error: " . $conn->error; 
}

$success_message = $_SESSION['success_message'] ?? null; unset($_SESSION['success_message']);
$error_message = $_SESSION['error_message'] ?? null; unset($_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Custom Package Rates</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <style>
        .rate-management-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(500px, 1fr)); gap: 2rem; }
        .rate-card { background-color: #fff; border: 1px solid var(--border-color); border-radius: 8px; padding: 1.5rem; }
        .rate-card h2, .content-card h2 { margin-top: 0; border-bottom: 2px solid var(--primary-color); padding-bottom: 0.5rem; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 10px; }
        .rate-form .form-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 1rem; }
        .form-actions { margin-top: 1.5rem; display: flex; align-items: center; gap: 1rem; }
        .table-responsive { width: 100%; overflow-x: auto; margin-top: 1.5rem; }
        .data-table { width: 100%; min-width: 480px; } .data-table .actions-cell { white-space: nowrap; }
        .data-table tbody tr { cursor: grab; }
        .checkbox-inline { display: flex; align-items: center; gap: 0.5rem; padding-top: 1.75rem; }
        .color-picker-wrapper { display: flex; align-items: center; gap: 10px; }
        .color-picker-wrapper input[type="color"] { width: 40px; height: 40px; padding: 2px; border: 1px solid #ccc; border-radius: 4px; cursor: pointer; }
        .color-picker-wrapper input[type="text"] { flex-grow: 1; }
        
        #rate-list-preview-a4 {
            border: 2px dashed #ccc; padding: 20px; background-color: #f0f0f0; margin-top: 20px;
            width: 100%; max-width: 842px;
            margin-left: auto; margin-right: auto;
        }
        #rate-list-content { background: white; width: 100%; aspect-ratio: 1 / 1.414; box-shadow: 0 10px 25px rgba(0,0,0,0.1); border: 1px solid #ddd; }
        .check-all { width: 20px; }
        .table-actions { margin-top: 1rem; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Welcome, <?php echo e(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
            </header>
            <main class="content-body">
                <div class="page-header"><h1 class="page-title">Manage Custom Package Rates</h1></div>
                <?php if ($success_message): ?><div class="notice success"><?= e($success_message) ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error"><?= e($error_message) ?></div><?php endif; ?>
                
                <div class="content-card" id="rate-list-customization" style="margin-bottom: 2rem;">
                    <h2><i class="fas fa-palette"></i>Rate List Design Studio</h2>
                    <form action="manage-custom-package-rates.php" method="POST" class="rate-form" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="update_rate_list_settings">
                        
                        <div class="form-grid" style="grid-template-columns: 1fr 1fr;">
                            <div class="form-group">
                                <label>Company Logo</label>
                                <input type="file" name="rate_list_logo" accept="image/*">
                                <?php if (!empty($settings['rate_list_logo'])): ?><small>Current: <img src="../<?= e($settings['rate_list_logo']) ?>" alt="logo" style="height: 20px; vertical-align: middle;"></small><?php endif; ?>
                            </div>
                            <div class="form-group">
                                <label>Background Image (Optional)</label>
                                <input type="file" name="rate_list_bg_image" accept="image/*">
                                <?php if (!empty($settings['rate_list_bg_image'])): ?><small>Current: <a href="../<?= e($settings['rate_list_bg_image']) ?>" target="_blank">View Image</a></small><?php endif; ?>
                            </div>
                        </div>

                        <div class="form-grid" style="grid-template-columns: 1fr 1fr; margin-top: 1rem;">
                            <div class="form-group"><label>Main Title</label><input type="text" name="rate_list_main_title" value="<?= e($settings['rate_list_main_title'] ?? 'UMRAH 1447-H') ?>"></div>
                            <div class="form-group"><label>Subtitle</label><input type="text" name="rate_list_subtitle" value="<?= e($settings['rate_list_subtitle'] ?? 'AVIATION - HAJJ - UMRAH - TOURS') ?>"></div>
                        </div>
                        <div class="form-grid" style="grid-template-columns: 1fr 1fr; margin-top: 1rem;">
                            <div class="form-group"><label>Vouchers From</label><input type="text" name="rate_list_vouchers_from" value="<?= e($settings['rate_list_vouchers_from'] ?? '15 SEP, 2025') ?>"></div>
                            <div class="form-group"><label>Check Till</label><input type="text" name="rate_list_check_till" value="<?= e($settings['rate_list_check_till'] ?? '15 NOV, 2025') ?>"></div>
                        </div>

                        <p style="margin: 1.5rem 0 0.5rem; font-weight: 500;">Color Scheme:</p>
                        <div class="form-grid" style="grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));">
                            <div class="form-group"><label>Header Background</label><div class="color-picker-wrapper"><input type="color" name="rate_list_header_bg" value="<?= e($settings['rate_list_header_bg'] ?? '#B31B1B') ?>"><input type="text" value="<?= e($settings['rate_list_header_bg'] ?? '#B31B1B') ?>" onchange="this.previousElementSibling.value = this.value"></div></div>
                            <div class="form-group"><label>Header Text</label><div class="color-picker-wrapper"><input type="color" name="rate_list_header_text" value="<?= e($settings['rate_list_header_text'] ?? '#ffffff') ?>"><input type="text" value="<?= e($settings['rate_list_header_text'] ?? '#ffffff') ?>" onchange="this.previousElementSibling.value = this.value"></div></div>
                            <div class="form-group"><label>Table Header BG</label><div class="color-picker-wrapper"><input type="color" name="rate_list_table_header_bg" value="<?= e($settings['rate_list_table_header_bg'] ?? '#343A40') ?>"><input type="text" value="<?= e($settings['rate_list_table_header_bg'] ?? '#343A40') ?>" onchange="this.previousElementSibling.value = this.value"></div></div>
                            <div class="form-group"><label>Table Header Text</label><div class="color-picker-wrapper"><input type="color" name="rate_list_table_header_text" value="<?= e($settings['rate_list_table_header_text'] ?? '#ffffff') ?>"><input type="text" value="<?= e($settings['rate_list_table_header_text'] ?? '#ffffff') ?>" onchange="this.previousElementSibling.value = this.value"></div></div>
                            <div class="form-group"><label>Accent Color</label><div class="color-picker-wrapper"><input type="color" name="rate_list_accent_color" value="<?= e($settings['rate_list_accent_color'] ?? '#0056b3') ?>"><input type="text" value="<?= e($settings['rate_list_accent_color'] ?? '#0056b3') ?>" onchange="this.previousElementSibling.value = this.value"></div></div>
                        </div>

                        <p style="margin: 1.5rem 0 0.5rem; font-weight: 500;">Typography (use 'pt' for print consistency):</p>
                        <div class="form-grid" style="grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));">
                             <div class="form-group"><label for="rate_list_font">Font Family</label><select name="rate_list_font" id="rate_list_font"><option value="Poppins" <?= ($settings['rate_list_font'] ?? 'Poppins') == 'Poppins' ? 'selected' : '' ?>>Poppins</option><option value="Lato" <?= ($settings['rate_list_font'] ?? '') == 'Lato' ? 'selected' : '' ?>>Lato</option><option value="Montserrat" <?= ($settings['rate_list_font'] ?? '') == 'Montserrat' ? 'selected' : '' ?>>Montserrat</option><option value="Open Sans" <?= ($settings['rate_list_font'] ?? '') == 'Open Sans' ? 'selected' : '' ?>>Open Sans</option><option value="Roboto" <?= ($settings['rate_list_font'] ?? '') == 'Roboto' ? 'selected' : '' ?>>Roboto</option></select></div>
                             <div class="form-group"><label>Header Font Size (pt)</label><input type="number" step="0.5" name="rate_list_header_fs" value="<?= e($settings['rate_list_header_fs'] ?? '32') ?>"></div>
                             <div class="form-group"><label>Table Head Font Size (pt)</label><input type="number" step="0.5" name="rate_list_table_head_fs" value="<?= e($settings['rate_list_table_head_fs'] ?? '10') ?>"></div>
                             <div class="form-group"><label>Table Body Font Size (pt)</label><input type="number" step="0.5" name="rate_list_table_body_fs" value="<?= e($settings['rate_list_table_body_fs'] ?? '9') ?>"></div>
                        </div>

                        <div class="form-group" style="margin-top: 1.5rem;"><label>Footer Instructions (one per line)</label><textarea name="rate_list_instructions" rows="5"><?= e($settings['rate_list_instructions'] ?? "Rates are subject to availability & can be changed at any time without prior notice.\nAny Tax/Fee imposed by KSA will be charged additionally.") ?></textarea></div>
                        <div class="form-actions"><button type="submit" class="btn btn-primary">Save Design Settings</button></div>
                    </form>
                </div>

                <div class="content-card" id="rate-list-generator" style="margin-bottom: 2rem;">
                    <h2><i class="fas fa-file-image"></i> Hotel Rate List Generator</h2>
                    <p>After saving your design, generate a preview below and download the final JPG.</p>
                     <div class="form-grid" style="grid-template-columns: 1fr 1fr 1fr; align-items: end;">
                        <div class="form-group">
                            <label for="rate-list-select">Select Rate List</label>
                            <select id="rate-list-select">
                                <option value="rates1">Rates 1</option>
                                <option value="rates2">Rates 2</option>
                            </select>
                        </div>
                        <div class="form-actions" style="margin-top:0;">
                            <button id="generate-rate-list-btn" type="button" class="btn btn-primary">Generate & Preview</button>
                            <button id="download-rate-list-btn" type="button" class="btn btn-secondary" style="display:none;">Download as JPG</button>
                        </div>
                    </div>
                    <div id="rate-list-preview-a4" style="display:none;"><div id="rate-list-content"></div></div>
                </div>

                <div class="content-card" id="global-settings" style="margin-bottom: 2rem;">
                    <h2><i class="fas fa-cogs"></i> Global Settings</h2>
                    <form action="manage-custom-package-rates.php#global-settings" method="POST" class="rate-form">
                        <input type="hidden" name="action" value="update_settings"><input type="hidden" name="setting_key" value="sar_to_pkr_rate">
                        <div class="form-grid" style="grid-template-columns: 1fr 3fr;"><div class="form-group"><label>SAR to PKR Exchange Rate</label><input type="number" name="sar_to_pkr_rate" step="0.01" value="<?= e($settings['sar_to_pkr_rate'] ?? '75.00') ?>" required></div></div>
                        <div class="form-actions"><button type="submit" class="btn btn-primary">Save Exchange Rate</button></div>
                    </form>
                </div>

                <div class="rate-management-grid">
                    <div class="rate-card" id="visa-rates">
                        <h2><i class="fas fa-passport"></i> Visa Rates</h2>
                        <form action="manage-custom-package-rates.php#visa-rates" method="POST" class="rate-form">
                            <input type="hidden" name="action" value="<?= ($edit_mode && $edit_type == 'visa') ? 'update_visa' : 'add_visa' ?>">
                            <?php if ($edit_mode && $edit_type == 'visa'): ?><input type="hidden" name="id" value="<?= e($edit_data['id']) ?>"><?php endif; ?>
                            <div class="form-grid" style="grid-template-columns: 1fr 1fr; margin-bottom:1rem;"><div class="form-group"><label>Visa Type Name</label><input type="text" name="visa_type" placeholder="e.g., 21 Days Visa" value="<?= e($edit_data['visa_type'] ?? '') ?>" required></div><div class="form-group"><label>Duration (days)</label><input type="number" name="duration_days" placeholder="e.g., 21" value="<?= e($edit_data['duration_days'] ?? '') ?>" required></div></div>
                            <p style="margin-bottom: 0.5rem; font-weight: 500;">Rates Per Person (SAR):</p>
                            <div class="form-grid" style="grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));"><div class="form-group"><label>1 Person</label><input type="number" name="rate_per_pax_1" step="0.01" value="<?= e($edit_data['rate_per_pax_1'] ?? '') ?>"></div><div class="form-group"><label>2 Persons</label><input type="number" name="rate_per_pax_2" step="0.01" value="<?= e($edit_data['rate_per_pax_2'] ?? '') ?>"></div><div class="form-group"><label>3 Persons</label><input type="number" name="rate_per_pax_3" step="0.01" value="<?= e($edit_data['rate_per_pax_3'] ?? '') ?>"></div><div class="form-group"><label>4 Persons</label><input type="number" name="rate_per_pax_4" step="0.01" value="<?= e($edit_data['rate_per_pax_4'] ?? '') ?>"></div><div class="form-group"><label>5+ Persons</label><input type="number" name="rate_per_pax_5_plus" step="0.01" value="<?= e($edit_data['rate_per_pax_5_plus'] ?? '') ?>"></div></div>
                            <div class="form-actions"><button type="submit" class="btn btn-primary"><?= ($edit_mode && $edit_type == 'visa') ? 'Update Visa Type' : 'Add Visa Type' ?></button><?php if ($edit_mode && $edit_type == 'visa'): ?><a href="manage-custom-package-rates.php#visa-rates" class="btn btn-secondary">Cancel Edit</a><?php endif; ?></div>
                        </form>
                        
                        <form action="manage-custom-package-rates.php#visa-rates" method="POST" onsubmit="return confirm('Are you sure you want to delete the selected items?');">
                             <input type="hidden" name="action" value="delete_selected_visas">
                             <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="visa-checkbox"></th><th>Type</th><th>Duration</th><th>1 Pax</th><th>2 Pax</th><th>3 Pax</th><th>4 Pax</th><th>5+ Pax</th><th>Actions</th></tr></thead><tbody><?php if ($visa_rates_result && $visa_rates_result->num_rows > 0): ?><?php while($row = $visa_rates_result->fetch_assoc()): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="visa-checkbox"></td><td><strong><?= e($row['visa_type']) ?></strong></td><td><?= e($row['duration_days']) ?> days</td><td><?= e($row['rate_per_pax_1']) ?></td><td><?= e($row['rate_per_pax_2']) ?></td><td><?= e($row['rate_per_pax_3']) ?></td><td><?= e($row['rate_per_pax_4']) ?></td><td><?= e($row['rate_per_pax_5_plus']) ?></td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=visa#visa-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endwhile; ?><?php else: ?><tr><td colspan="9" style="text-align:center;">No visa rates found. Add one using the form above.</td></tr><?php endif; ?></tbody></table></div>
                            <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                    </div>

                    <div class="rate-card" id="transport-rates">
                        <h2><i class="fas fa-bus"></i> Transport Rates</h2>
                        <form action="manage-custom-package-rates.php#transport-rates" method="POST" class="rate-form">
                            <input type="hidden" name="action" value="<?= ($edit_mode && $edit_type == 'transport') ? 'update_transport' : 'add_transport' ?>"><?php if ($edit_mode && $edit_type == 'transport'): ?><input type="hidden" name="id" value="<?= e($edit_data['id']) ?>"><?php endif; ?>
                            <div class="form-grid" style="grid-template-columns: 1fr 1fr;"><div class="form-group"><label>Sector / Route Name</label><select name="sector_name" required><option value="Round Trip A" <?= (($edit_data['sector_name'] ?? '') == 'Round Trip A') ? 'selected' : '' ?>>Round Trip A</option><option value="Round Trip B" <?= (($edit_data['sector_name'] ?? '') == 'Round Trip B') ? 'selected' : '' ?>>Round Trip B</option><option value="Round Trip C" <?= (($edit_data['sector_name'] ?? '') == 'Round Trip C') ? 'selected' : '' ?>>Round Trip C</option></select></div><div class="form-group"><label>Vehicle Type</label><input type="text" name="vehicle_type" placeholder="e.g., Sedan, Bus" value="<?= e($edit_data['vehicle_type'] ?? '') ?>" required></div></div>
                            <div class="form-group" style="margin-top: 1rem;"><label>Trip Details (e.g., JED -> MAK -> MED -> MED APT)</label><input type="text" name="trip_details" value="<?= e($edit_data['trip_details'] ?? '') ?>"></div>
                            <div class="form-group" style="margin-top: 1rem;"><label>Rate (SAR)</label><input type="number" name="rate_sar" step="0.01" value="<?= e($edit_data['rate_sar'] ?? '') ?>" required></div>
                            <input type="hidden" name="is_round_trip" value="1">
                            <div class="form-actions"><button type="submit" class="btn btn-primary"><?= ($edit_mode && $edit_type == 'transport') ? 'Update' : 'Add Transport Rate' ?></button><?php if ($edit_mode && $edit_type == 'transport'): ?><a href="manage-custom-package-rates.php#transport-rates" class="btn btn-secondary">Cancel</a><?php endif; ?></div>
                        </form>
                        
                        <form action="manage-custom-package-rates.php#transport-rates" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_transports">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="transport-checkbox"></th><th>Route Name</th><th>Vehicle</th><th>Rate</th><th>Actions</th></tr></thead><tbody id="transport-sortable"><?php if ($transport_rates_result && $transport_rates_result->num_rows > 0): ?><?php while($row = $transport_rates_result->fetch_assoc()): ?><tr data-id="<?= $row['id'] ?>"><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="transport-checkbox"></td><td><strong><?= e($row['sector_name']) ?></strong><br><small><?= e($row['trip_details']) ?></small></td><td><?= e($row['vehicle_type']) ?></td><td><?= e(number_format($row['rate_sar'], 2)) ?> SAR</td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=transport#transport-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endwhile; ?><?php endif; ?></tbody></table></div>
                            <div class="table-actions form-actions"><button type="submit" class="btn btn-danger">Delete Selected</button><button id="save-transport-order" type="button" class="btn btn-secondary">Save Order</button><span id="order-status" style="color: green;"></span></div>
                        </form>
                    </div>
                
                    <div class="rate-card" id="ziyarat-rates">
                        <h2><i class="fas fa-kaaba"></i> Ziyarat Rates</h2>
                        <form action="manage-custom-package-rates.php#ziyarat-rates" method="POST" class="rate-form">
                            <input type="hidden" name="action" value="<?= ($edit_mode && $edit_type == 'ziyarat') ? 'update_ziyarat' : 'add_ziyarat' ?>"><?php if ($edit_mode && $edit_type == 'ziyarat'): ?><input type="hidden" name="id" value="<?= e($edit_data['id']) ?>"><?php endif; ?>
                            <div class="form-grid"><div class="form-group"><label>City / Tour</label><input type="text" name="city" placeholder="e.g., Makkah Ziyarat" value="<?= e($edit_data['city'] ?? '') ?>" required></div><div class="form-group"><label>Vehicle Type</label><input type="text" name="vehicle_type" placeholder="e.g., Sedan, Bus" value="<?= e($edit_data['vehicle_type'] ?? '') ?>" required></div><div class="form-group"><label>Rate (SAR)</label><input type="number" name="rate_sar" step="0.01" value="<?= e($edit_data['rate_sar'] ?? '') ?>" required><small>For 'Bus' type, enter rate per person.</small></div></div>
                            <div class="form-actions"><button type="submit" class="btn btn-primary"><?= ($edit_mode && $edit_type == 'ziyarat') ? 'Update' : 'Add Ziyarat Rate' ?></button><?php if ($edit_mode && $edit_type == 'ziyarat'): ?><a href="manage-custom-package-rates.php#ziyarat-rates" class="btn btn-secondary">Cancel</a><?php endif; ?></div>
                        </form>
                        
                        <form action="manage-custom-package-rates.php#ziyarat-rates" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_ziyarats">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="ziyarat-checkbox"></th><th>City / Tour</th><th>Vehicle</th><th>Rate</th><th>Actions</th></tr></thead><tbody><?php if ($ziyarat_rates_result && $ziyarat_rates_result->num_rows > 0): ?><?php while($row = $ziyarat_rates_result->fetch_assoc()): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="ziyarat-checkbox"></td><td><?= e($row['city']) ?></td><td><?= e($row['vehicle_type']) ?></td><td><?= e(number_format($row['rate_sar'], 2)) ?> SAR</td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=ziyarat#ziyarat-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endwhile; ?><?php endif; ?></tbody></table></div>
                            <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                    </div>
                </div>

                <div class="content-card" id="hotel-rates" style="margin-top:2rem;">
                    <h2><i class="fas fa-hotel"></i> Hotel Rates</h2>
                    <form action="manage-custom-package-rates.php#hotel-rates" method="POST" class="rate-form">
                        <input type="hidden" name="action" value="<?= ($edit_mode && $edit_type == 'hotel') ? 'update_hotel' : 'add_hotel' ?>"><?php if ($edit_mode && $edit_type == 'hotel'): ?><input type="hidden" name="id" value="<?= e($edit_data['id']) ?>"><?php endif; ?>
                        <div class="form-grid" style="grid-template-columns: 2fr repeat(4, 1fr); align-items: end;">
                            <div class="form-group"><label>Hotel Name</label><input type="text" name="hotel_name" value="<?= e($edit_data['hotel_name'] ?? '') ?>" required></div>
                            <div class="form-group"><label>City</label><select name="city" required><option value="Makkah" <?= (($edit_data['city'] ?? '') == 'Makkah') ? 'selected' : '' ?>>Makkah</option><option value="Madinah" <?= (($edit_data['city'] ?? '') == 'Madinah') ? 'selected' : '' ?>>Madinah</option></select></div>
                            <div class="form-group"><label>Star Rating</label><input type="text" name="star_rating" value="<?= e($edit_data['star_rating'] ?? '') ?>"></div>
                            <div class="form-group"><label>Distance</label><input type="text" name="distance_info" value="<?= e($edit_data['distance_info'] ?? '') ?>"></div>
                            <div class="form-group"><label>Rate List Type</label><select name="rate_list_type" required><option value="Rates 1" <?= (($edit_data['rate_list_type'] ?? '') == 'Rates 1') ? 'selected' : '' ?>>Rates 1</option><option value="Rates 2" <?= (($edit_data['rate_list_type'] ?? '') == 'Rates 2') ? 'selected' : '' ?>>Rates 2</option></select></div>
                        </div>
                        <div class="form-grid" style="grid-template-columns: 2fr 1fr; margin-top: 1rem; align-items: end;">
                            <div class="form-group"><label>Notes</label><input type="text" name="notes" value="<?= e($edit_data['notes'] ?? '') ?>"></div>
                            <div class="form-group"><label>Currency</label><select name="currency" required><option value="SAR" selected>SAR</option></select></div>
                        </div>
                        <p style="margin: 1rem 0 0.5rem; font-weight: 500;">Rates per person per night:</p>
                        <div class="form-grid" style="grid-template-columns: repeat(5, 1fr);"><div class="form-group"><label>Sharing Rate/Night</label><input type="number" name="rate_sharing" step="0.01" value="<?= e($edit_data['rate_sharing'] ?? '') ?>"></div><div class="form-group"><label>Quad Rate/Night</label><input type="number" name="rate_quad" step="0.01" value="<?= e($edit_data['rate_quad'] ?? '') ?>"></div><div class="form-group"><label>Triple Rate/Night</label><input type="number" name="rate_trpl" step="0.01" value="<?= e($edit_data['rate_trpl'] ?? '') ?>"></div><div class="form-group"><label>Double Rate/Night</label><input type="number" name="rate_dubl" step="0.01" value="<?= e($edit_data['rate_dubl'] ?? '') ?>"></div><div class="form-group"><label>Room Basis Rate</label><input type="number" name="rate_room_basis" step="0.01" value="<?= e($edit_data['rate_room_basis'] ?? '') ?>"></div></div>
                        <div class="form-actions"><button type="submit" class="btn btn-primary"><?= ($edit_mode && $edit_type == 'hotel') ? 'Update Hotel Rate' : 'Add Hotel Rate' ?></button><?php if ($edit_mode && $edit_type == 'hotel'): ?><a href="manage-custom-package-rates.php#hotel-rates" class="btn btn-secondary">Cancel</a><?php endif; ?></div>
                    </form>

                    <div id="rates1-hotels" style="margin-top: 2rem;">
                        <h3><i class="fas fa-list-ol"></i> Rate List 1</h3>
                        <h4 style="margin-top:1rem;">Makkah Hotels</h4>
                        <form action="manage-custom-package-rates.php#rates1-hotels" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_hotels">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="rates1-makkah-hotel-checkbox"></th><th>Hotel</th><th>Dist.</th><th>Sharing</th><th>Quad</th><th>Triple</th><th>Double</th><th>Room</th><th>Actions</th></tr></thead><tbody><?php foreach($rates1_makkah_hotels as $row): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="rates1-makkah-hotel-checkbox"></td><td><strong><?= e($row['hotel_name']) ?></strong><br><small><?= e($row['notes']) ?></small></td><td><?= e($row['distance_info']) ?></td><td><?= $row['rate_sharing'] ? e(number_format($row['rate_sharing'], 2)) : '-' ?></td><td><?= $row['rate_quad'] ? e(number_format($row['rate_quad'], 2)) : '-' ?></td><td><?= $row['rate_trpl'] ? e(number_format($row['rate_trpl'], 2)) : '-' ?></td><td><?= $row['rate_dubl'] ? e(number_format($row['rate_dubl'], 2)) : '-' ?></td><td><?= $row['rate_room_basis'] ? e(number_format($row['rate_room_basis'], 2)) : '-' ?></td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=hotel#hotel-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endforeach; ?></tbody></table></div>
                            <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                        <h4 style="margin-top:2rem;">Madinah Hotels</h4>
                        <form action="manage-custom-package-rates.php#rates1-hotels" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_hotels">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="rates1-madinah-hotel-checkbox"></th><th>Hotel</th><th>Dist.</th><th>Sharing</th><th>Quad</th><th>Triple</th><th>Double</th><th>Room</th><th>Actions</th></tr></thead><tbody><?php foreach($rates1_madinah_hotels as $row): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="rates1-madinah-hotel-checkbox"></td><td><strong><?= e($row['hotel_name']) ?></strong><br><small><?= e($row['notes']) ?></small></td><td><?= e($row['distance_info']) ?></td><td><?= $row['rate_sharing'] ? e(number_format($row['rate_sharing'], 2)) : '-' ?></td><td><?= $row['rate_quad'] ? e(number_format($row['rate_quad'], 2)) : '-' ?></td><td><?= $row['rate_trpl'] ? e(number_format($row['rate_trpl'], 2)) : '-' ?></td><td><?= $row['rate_dubl'] ? e(number_format($row['rate_dubl'], 2)) : '-' ?></td><td><?= $row['rate_room_basis'] ? e(number_format($row['rate_room_basis'], 2)) : '-' ?></td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=hotel#hotel-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endforeach; ?></tbody></table></div>
                             <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                    </div>

                    <div id="rates2-hotels" style="margin-top: 2rem;">
                        <h3><i class="fas fa-list-ol"></i> Rate List 2</h3>
                        <h4 style="margin-top:1rem;">Makkah Hotels</h4>
                        <form action="manage-custom-package-rates.php#rates2-hotels" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_hotels">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="rates2-makkah-hotel-checkbox"></th><th>Hotel</th><th>Dist.</th><th>Sharing</th><th>Quad</th><th>Triple</th><th>Double</th><th>Room</th><th>Actions</th></tr></thead><tbody><?php foreach($rates2_makkah_hotels as $row): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="rates2-makkah-hotel-checkbox"></td><td><strong><?= e($row['hotel_name']) ?></strong><br><small><?= e($row['notes']) ?></small></td><td><?= e($row['distance_info']) ?></td><td><?= $row['rate_sharing'] ? e(number_format($row['rate_sharing'], 2)) : '-' ?></td><td><?= $row['rate_quad'] ? e(number_format($row['rate_quad'], 2)) : '-' ?></td><td><?= $row['rate_trpl'] ? e(number_format($row['rate_trpl'], 2)) : '-' ?></td><td><?= $row['rate_dubl'] ? e(number_format($row['rate_dubl'], 2)) : '-' ?></td><td><?= $row['rate_room_basis'] ? e(number_format($row['rate_room_basis'], 2)) : '-' ?></td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=hotel#hotel-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endforeach; ?></tbody></table></div>
                            <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                         <h4 style="margin-top:2rem;">Madinah Hotels</h4>
                        <form action="manage-custom-package-rates.php#rates2-hotels" method="POST" onsubmit="return confirm('Are you sure you want to delete selected items?');">
                            <input type="hidden" name="action" value="delete_selected_hotels">
                            <div class="table-responsive"><table class="data-table"><thead><tr><th class="check-all"><input type="checkbox" class="check-all-box" data-target="rates2-madinah-hotel-checkbox"></th><th>Hotel</th><th>Dist.</th><th>Sharing</th><th>Quad</th><th>Triple</th><th>Double</th><th>Room</th><th>Actions</th></tr></thead><tbody><?php foreach($rates2_madinah_hotels as $row): ?><tr><td><input type="checkbox" name="selected_ids[]" value="<?= $row['id'] ?>" class="rates2-madinah-hotel-checkbox"></td><td><strong><?= e($row['hotel_name']) ?></strong><br><small><?= e($row['notes']) ?></small></td><td><?= e($row['distance_info']) ?></td><td><?= $row['rate_sharing'] ? e(number_format($row['rate_sharing'], 2)) : '-' ?></td><td><?= $row['rate_quad'] ? e(number_format($row['rate_quad'], 2)) : '-' ?></td><td><?= $row['rate_trpl'] ? e(number_format($row['rate_trpl'], 2)) : '-' ?></td><td><?= $row['rate_dubl'] ? e(number_format($row['rate_dubl'], 2)) : '-' ?></td><td><?= $row['rate_room_basis'] ? e(number_format($row['rate_room_basis'], 2)) : '-' ?></td><td class="actions-cell"><a href="?edit_id=<?= $row['id'] ?>&type=hotel#hotel-rates" class="btn btn-sm btn-secondary"><i class="fas fa-edit"></i></a></td></tr><?php endforeach; ?></tbody></table></div>
                             <div class="table-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                        </form>
                    </div>

                </div>
            </main>
        </div>
    </div>
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/Sortable/1.15.0/Sortable.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const menuToggle = document.getElementById('menu-toggle');
    if (menuToggle) {
        menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
    }
    
    document.querySelectorAll('.check-all-box').forEach(headerCheckbox => {
        headerCheckbox.addEventListener('change', function(e) {
            const targetClass = e.target.dataset.target;
            document.querySelectorAll(`.${targetClass}`).forEach(checkbox => { checkbox.checked = e.target.checked; });
        });
    });

    const sortableList = document.getElementById('transport-sortable');
    if (sortableList) new Sortable(sortableList, { animation: 150 });

    $('#save-transport-order').on('click', function() {
        const order = Array.from(sortableList.children).map(row => $(row).data('id'));
        const statusSpan = $('#order-status').text('Saving...').css('color', 'orange');
        $.ajax({
            url: 'manage-custom-package-rates.php', type: 'POST',
            data: { action: 'update_transport_order', order: JSON.stringify(order) },
            success: () => { statusSpan.text('Order saved!').css('color', 'green'); setTimeout(() => statusSpan.text(''), 3000); },
            error: () => statusSpan.text('Error saving.').css('color', 'red')
        });
    });

    function loadGoogleFont(fontName) {
        const fontId = `google-font-${fontName.replace(' ', '-')}`;
        if (document.getElementById(fontId)) return;
        const link = document.createElement('link');
        link.id = fontId; link.rel = 'stylesheet';
        link.href = `https://fonts.googleapis.com/css2?family=${fontName.replace(' ', '+')}:wght@400;500;600;700;800&display=swap`;
        document.head.appendChild(link);
    }

    $('#generate-rate-list-btn').on('click', function() {
        const settings = {
            logo: "<?= !empty($settings['rate_list_logo']) ? '../' . e($settings['rate_list_logo']) : '' ?>",
            bgImage: "<?= !empty($settings['rate_list_bg_image']) ? '../' . e($settings['rate_list_bg_image']) : '' ?>",
            mainTitle: $('[name="rate_list_main_title"]').val(),
            subtitle: $('[name="rate_list_subtitle"]').val(),
            vouchersFrom: $('[name="rate_list_vouchers_from"]').val(),
            checkTill: $('[name="rate_list_check_till"]').val(),
            instructions: $('[name="rate_list_instructions"]').val().split('\n').filter(line => line.trim() !== ''),
            headerBg: $('[name="rate_list_header_bg"]').val(),
            headerText: $('[name="rate_list_header_text"]').val(),
            tableHeaderBg: $('[name="rate_list_table_header_bg"]').val(),
            tableHeaderText: $('[name="rate_list_table_header_text"]').val(),
            accentColor: $('[name="rate_list_accent_color"]').val(),
            font: $('[name="rate_list_font"]').val(),
            headerFs: $('[name="rate_list_header_fs"]').val(),
            tableHeadFs: $('[name="rate_list_table_head_fs"]').val(),
            tableBodyFs: $('[name="rate_list_table_body_fs"]').val()
        };
        
        loadGoogleFont(settings.font);

        const rates1_makkah = <?php echo json_encode($rates1_makkah_hotels); ?>;
        const rates1_madinah = <?php echo json_encode($rates1_madinah_hotels); ?>;
        const rates2_makkah = <?php echo json_encode($rates2_makkah_hotels); ?>;
        const rates2_madinah = <?php echo json_encode($rates2_madinah_hotels); ?>;
        
        const selectedList = $('#rate-list-select').val();
        let makkahData, madinahData;

        if (selectedList === 'rates1') {
            makkahData = rates1_makkah;
            madinahData = rates1_madinah;
        } else {
            makkahData = rates2_makkah;
            madinahData = rates2_madinah;
        }

        const backgroundStyle = settings.bgImage ? `background-image: url('${settings.bgImage}'); background-size: cover; background-position: center;` : 'background-color: #fff;';
        
        const dynamicStyles = `
        <style>
            #rate-list-content *, #rate-list-content *::before, #rate-list-content *::after {
                box-sizing: border-box; margin: 0; padding: 0;
            }
            #rate-list-content {
                font-family: '${settings.font}', sans-serif; display: flex; flex-direction: column; height: 100%; width: 100%; color: #333; ${backgroundStyle} padding: 30px;
            }
            .rl-header {
                background-color: ${settings.headerBg}; color: ${settings.headerText}; padding: 20px; text-align: center; flex-shrink: 0;
            }
            .rl-header img { max-width: 180px; height: auto; max-height: 70px; margin-bottom: 15px; }
            .rl-header h1 { font-size: ${settings.headerFs}pt; font-weight: 800; line-height: 1.1; text-transform: uppercase; letter-spacing: 1px; }
            .rl-header p { margin-top: 5px; font-size: ${parseFloat(settings.headerFs) * 0.35}pt; opacity: 0.9; font-weight: 500; letter-spacing: 0.5px; }
            .rl-info-bar { display: flex; justify-content: space-between; padding: 10px 0; margin-bottom: 15px; border-bottom: 2px solid #eee; flex-shrink: 0; }
            .rl-info-box { text-align: center; background-color: #f8f9fa; padding: 8px 12px; border-radius: 6px; border: 1px solid #dee2e6; }
            .rl-info-box span { font-size: ${parseFloat(settings.tableBodyFs) * 0.85}pt; color: #6c757d; font-weight: 600; text-transform: uppercase; letter-spacing: 0.05em; }
            .rl-info-box h4 { margin-top: 4px; font-size: ${parseFloat(settings.tableBodyFs) * 1.2}pt; font-weight: 700; color: ${settings.accentColor}; }
            .rl-main-content { flex-grow: 1; overflow: hidden; }
            .rl-table { width: 100%; border-collapse: separate; border-spacing: 0; margin-bottom: 25px; }
            .rl-table th, .rl-table td { padding: 9px 12px; text-align: center; vertical-align: middle; border-bottom: 1px solid #dee2e6; }
            .rl-table th { background-color: ${settings.tableHeaderBg}; color: ${settings.tableHeaderText}; font-weight: 600; font-size: ${settings.tableHeadFs}pt; text-transform: uppercase; border: none; }
            .rl-table td { font-size: ${settings.tableBodyFs}pt; }
            .rl-table tbody tr:nth-child(even) { background-color: #f8f9fa; }
            .rl-table .city-header td { background-color: #e9ecef; color: #212529; font-weight: 700; text-align: left; font-size: ${parseFloat(settings.tableHeadFs) * 1.3}pt; padding: 12px; border-radius: 6px 6px 0 0; border-bottom: 3px solid ${settings.accentColor}; }
            .rl-table td:first-child { text-align: left; font-weight: 600; }
            .rl-table .notes { font-size: ${parseFloat(settings.tableBodyFs) * 0.85}pt; color: #6c757d; display: block; margin-top: 2px; font-weight: 400; }
            .rl-footer { margin-top: auto; padding-top: 15px; border-top: 2px solid #eee; flex-shrink: 0; display: flex; justify-content: space-between; align-items: flex-end; }
            .rl-footer .instructions-box h3 { font-size: ${parseFloat(settings.tableHeadFs) * 1.1}pt; color: ${settings.accentColor}; margin-bottom: 8px; text-transform: uppercase; }
            .rl-footer .instructions-box ul { list-style: none; padding: 0; margin: 0; }
            .rl-footer .instructions-box li { font-size: ${parseFloat(settings.tableBodyFs) * 0.85}pt; color: #495057; margin-bottom: 5px; padding-left: 18px; position: relative; line-height: 1.5; }
            .rl-footer .instructions-box li::before { content: '✔'; position: absolute; left: 0; top: 0.25em; color: ${settings.accentColor}; font-weight: bold; }
            .rl-footer .publish-date { font-size: ${parseFloat(settings.tableBodyFs) * 0.8}pt; color: #6c757d; text-align: right; }
        </style>`;
        
        const rateListHTML = `${dynamicStyles}<div class="rl-header">${settings.logo ? `<img src="${settings.logo}" alt="Company Logo">` : ''}<h1>${e(settings.mainTitle)}</h1><p>${e(settings.subtitle)}</p></div><div class="rl-info-bar"><div class="rl-info-box"><span>VOUCHERS FROM</span><h4>${e(settings.vouchersFrom)}</h4></div><div class="rl-info-box"><span>CHECK TILL</span><h4>${e(settings.checkTill)}</h4></div></div><div class="rl-main-content">${generateHotelTable('MAKKAH HOTELS', makkahData)}${generateHotelTable('MADINAH HOTELS', madinahData)}</div><div class="rl-footer"><div class="instructions-box"><h3>INSTRUCTIONS</h3><ul>${settings.instructions.map(line => `<li>${e(line)}</li>`).join('')}</ul></div><p class="publish-date">Published: ${new Date().toLocaleDateString('en-GB', { day: 'numeric', month: 'short', year: 'numeric' })}</p></div>`;
        
        $('#rate-list-content').html(rateListHTML);
        $('#rate-list-preview-a4, #download-rate-list-btn').show();
    });

    $('#download-rate-list-btn').on('click', function() {
        const content = document.getElementById('rate-list-content');
        const button = $(this);
        button.text('Generating...').prop('disabled', true);
        
        html2canvas(content, { scale: 3, useCORS: true }).then(canvas => {
            const link = document.createElement('a');
            const selectedList = $('#rate-list-select').val();
            const listNumber = (selectedList === 'rates1') ? '1' : '2';
            link.download = `umrah-hotel-rate-list-${listNumber}.jpg`;
            link.href = canvas.toDataURL('image/jpeg', 0.95);
            link.click();
            button.text('Download as JPG').prop('disabled', false);
        });
    });

    function generateHotelTable(title, hotels) {
        if (!hotels || hotels.length === 0) return '';
        const rows = hotels.map(hotel => `<tr><td>${e(hotel.hotel_name)}<span class="notes">${e(hotel.notes)}</span></td><td>${e(hotel.star_rating)}</td><td>${e(hotel.distance_info)}</td><td>${hotel.rate_sharing ? parseFloat(hotel.rate_sharing).toFixed(0) : 'N/A'}</td><td>${hotel.rate_quad ? parseFloat(hotel.rate_quad).toFixed(0) : 'N/A'}</td><td>${hotel.rate_trpl ? parseFloat(hotel.rate_trpl).toFixed(0) : 'N/A'}</td><td>${hotel.rate_dubl ? parseFloat(hotel.rate_dubl).toFixed(0) : 'N/A'}</td></tr>`).join('');
        return `<table class="rl-table"><thead><tr class="city-header"><td colspan="7">${title}</td></tr><tr><th>HOTELS</th><th>STAR</th><th>DISTANCE</th><th>SHARING</th><th>QUAD</th><th>TRIPLE</th><th>DOUBLE</th></tr></thead><tbody>${rows}</tbody></table>`;
    }

    function e(str) {
        if (str === null || str === undefined) return '';
        return String(str).replace(/[&<>"']/g, s => ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' }[s]));
    }
});
</script>
</body>
</html>