<?php
session_start();
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

// =======================================================
//  SETUP & HELPER FUNCTIONS
// =======================================================
define('HOTEL_IMAGE_UPLOAD_DIR', '../images/hotels/');
define('HOTEL_IMAGE_DB_PATH', 'images/hotels/');

if (!is_dir(HOTEL_IMAGE_UPLOAD_DIR)) {
    if (!mkdir(HOTEL_IMAGE_UPLOAD_DIR, 0777, true)) {
        die("Failed to create the hotel image upload directory. Please check server permissions for the '/images/' folder.");
    }
}


//======================================================================
//  HANDLE FORM SUBMISSIONS (CREATE, UPDATE, DELETE, BULK DELETE)
//======================================================================
$edit_mode = false;
$hotel_to_edit = null;
if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {

    $image_url_to_save = $_POST['image_url'] ?? '';
    if (isset($_FILES['hotel_image_upload']) && $_FILES['hotel_image_upload']['error'] === UPLOAD_ERR_OK) {
        $file_tmp_name = $_FILES["hotel_image_upload"]["tmp_name"];
        $file_name = time() . '_' . basename(preg_replace("/[^a-zA-Z0-9.\-\_]/", "", $_FILES["hotel_image_upload"]["name"]));
        $target_file = HOTEL_IMAGE_UPLOAD_DIR . $file_name;
        
        $check = getimagesize($file_tmp_name);
        if($check !== false) {
            if (move_uploaded_file($file_tmp_name, $target_file)) {
                $image_url_to_save = HOTEL_IMAGE_DB_PATH . $file_name;
                $stmt_gallery = $conn->prepare("INSERT INTO image_gallery (file_path) VALUES (?)");
                $stmt_gallery->bind_param("s", $image_url_to_save);
                $stmt_gallery->execute();
                $stmt_gallery->close();
            } else {
                $_SESSION['error_message'] = "Sorry, there was an error moving the uploaded file. Check folder permissions for '" . HOTEL_IMAGE_UPLOAD_DIR . "'.";
                header("Location: manage-hotels.php"); exit();
            }
        } else {
            $_SESSION['error_message'] = "File is not a valid image. Please upload a valid image file.";
            header("Location: manage-hotels.php"); exit();
        }
    }

    if ($_POST['action'] === 'add') {
        try {
            $sql = "INSERT INTO hotels (hotel_name, location, rating, price_per_night, price_per_night_agent, agent_incentive, exchange_rate_pkr, logo_background_style, image_url, is_active, display_order, last_updated) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())";
            $stmt = $conn->prepare($sql);
            
            $price_agent = !empty($_POST['price_per_night_agent']) ? (float)$_POST['price_per_night_agent'] : null;
            $incentive = !empty($_POST['agent_incentive']) ? (float)$_POST['agent_incentive'] : 0.00;
            $exchange_rate = !empty($_POST['exchange_rate_pkr']) ? (float)$_POST['exchange_rate_pkr'] : null;
            $price_customer = (float)$_POST['price_per_night'];
            $rating = (int)$_POST['rating'];
            $is_active = (int)$_POST['is_active'];
            $display_order = (int)$_POST['display_order'];
            $logo_bg_style = $_POST['logo_background_style'] ?? 'white_bg';


            $stmt->bind_param("ssiddsdssii", $_POST['hotel_name'], $_POST['location'], $rating, $price_customer, $price_agent, $incentive, $exchange_rate, $logo_bg_style, $image_url_to_save, $is_active, $display_order);
            $stmt->execute();
            $_SESSION['success_message'] = "Hotel '{$_POST['hotel_name']}' was added successfully!";
        } catch (Exception $e) {
            $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
        }
        header("Location: manage-hotels.php");
        exit();
    }

    if ($_POST['action'] === 'update' && isset($_POST['id'])) {
        try {
            $sql = "UPDATE hotels SET hotel_name = ?, location = ?, rating = ?, price_per_night = ?, price_per_night_agent = ?, agent_incentive = ?, exchange_rate_pkr = ?, logo_background_style = ?, image_url = ?, is_active = ?, display_order = ?, last_updated = NOW() WHERE id = ?";
            $stmt = $conn->prepare($sql);

            $price_agent = !empty($_POST['price_per_night_agent']) ? (float)$_POST['price_per_night_agent'] : null;
            $incentive = !empty($_POST['agent_incentive']) ? (float)$_POST['agent_incentive'] : 0.00;
            $exchange_rate = !empty($_POST['exchange_rate_pkr']) ? (float)$_POST['exchange_rate_pkr'] : null;
            $price_customer = (float)$_POST['price_per_night'];
            $rating = (int)$_POST['rating'];
            $is_active = (int)$_POST['is_active'];
            $display_order = (int)$_POST['display_order'];
            $hotel_id = (int)$_POST['id'];
            $logo_bg_style = $_POST['logo_background_style'] ?? 'white_bg';

            $stmt->bind_param("ssiddsdssiii", $_POST['hotel_name'], $_POST['location'], $rating, $price_customer, $price_agent, $incentive, $exchange_rate, $logo_bg_style, $image_url_to_save, $is_active, $display_order, $hotel_id);
            $stmt->execute();
            $_SESSION['success_message'] = "Hotel '{$_POST['hotel_name']}' was updated successfully!";
        } catch (Exception $e) {
            $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
        }
        header("Location: manage-hotels.php");
        exit();
    }

    if ($_POST['action'] === 'delete' && isset($_POST['id'])) {
        $stmt = $conn->prepare("DELETE FROM hotels WHERE id = ?");
        $stmt->bind_param("i", $_POST['id']);
        $stmt->execute();
        $_SESSION['success_message'] = "The hotel was deleted successfully!";
        header("Location: manage-hotels.php");
        exit();
    }
    
    if ($_POST['action'] === 'bulk_delete') {
        if (!empty($_POST['hotel_ids']) && is_array($_POST['hotel_ids'])) {
            $ids_to_delete = array_map('intval', $_POST['hotel_ids']);
            if (!empty($ids_to_delete)) {
                $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
                $types = str_repeat('i', count($ids_to_delete));
                $sql = "DELETE FROM hotels WHERE id IN ($placeholders)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param($types, ...$ids_to_delete);
                if ($stmt->execute()) {
                    $_SESSION['success_message'] = $stmt->affected_rows . " hotels were deleted successfully!";
                } else {
                    $_SESSION['error_message'] = "Error deleting hotels: " . $conn->error;
                }
            } else { $_SESSION['error_message'] = "No valid hotel IDs selected."; }
        } else { $_SESSION['error_message'] = "No hotels were selected."; }
        header("Location: manage-hotels.php");
        exit();
    }
}

// Handle GET request for edit mode
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $stmt = $conn->prepare("SELECT * FROM hotels WHERE id = ?");
    $stmt->bind_param("i", $_GET['id']);
    $stmt->execute();
    $hotel_to_edit = $stmt->get_result()->fetch_assoc();
}

// Fetch data for display
$hotels_result = $conn->query("SELECT * FROM hotels ORDER BY display_order ASC, last_updated DESC");
$gallery_images_result = $conn->query("SELECT id, file_path FROM image_gallery ORDER BY id DESC");
$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Hotel Listings</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        :root { --form-grid-gap: 1.25rem; }
        .package-form fieldset { border: 1px solid var(--border-color); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem; background-color: #fdfdfd; }
        .package-form legend { font-weight: 600; padding: 0 10px; font-size: 1.1rem; color: var(--primary-color); }
        .form-grid-4col { display: grid; grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)); gap: var(--form-grid-gap); }
        .image-upload-group { display: flex; flex-direction: column; gap: 0.75rem; }
        .image-upload-group .upload-row { display: flex; align-items: center; gap: 1rem; }
        .image-upload-group input[type="file"] { flex-grow: 1; }
        .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.7); backdrop-filter: blur(5px); }
        .modal-content { background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 1200px; border-radius: 8px; }
        .modal-header { display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #ddd; padding-bottom: 10px; margin-bottom: 20px; }
        .modal-title { font-size: 1.5rem; color: var(--text-dark); }
        .close-button { color: #aaa; font-size: 28px; font-weight: bold; cursor: pointer; }
        .gallery-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 15px; max-height: 60vh; overflow-y: auto; padding: 5px; }
        .gallery-item img { width: 100%; height: 120px; object-fit: cover; border-radius: 4px; cursor: pointer; border: 2px solid transparent; transition: border-color 0.2s; }
        .gallery-item img:hover, .gallery-item img.selected { border-color: var(--primary-color); box-shadow: 0 0 10px rgba(0,123,255,0.5); }
        .table-responsive { width: 100%; }
        .data-table { width: 100%; border-collapse: collapse; }
        .data-table thead { display: none; }
        .data-table tr { display: block; margin-bottom: 1rem; border: 1px solid var(--border-color); border-radius: 8px; overflow: hidden; background: #fff; }
        .data-table td { display: flex; justify-content: space-between; align-items: center; text-align: right; font-size: 0.9em; border-bottom: 1px solid #eee; padding: 0.75rem 1rem; }
        .data-table td:before { content: attr(data-label); font-weight: 600; text-align: left; padding-right: 1rem; }
        .data-table td:last-child { border-bottom: 0; }
        @media (min-width: 1100px) {
            .data-table thead { display: table-header-group; }
            .data-table tr { display: table-row; margin-bottom: 0; border: 0; border-bottom: 1px solid var(--border-color); border-radius: 0; }
            .data-table tr:last-child { border-bottom: 0; }
            .data-table td { display: table-cell; text-align: left; border-bottom: 0; }
            .data-table td:before { display: none; }
            .data-table .actions-cell { text-align: center; }
        }
        .bulk-actions { margin-top: 1rem; }
    </style>
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Manage Website Hotels</span></div>
            </header>
            <main class="content-body" id="top">
                <div class="content-card">
                    <h2 class="form-title" style="margin-bottom: 2rem;"><?= $edit_mode ? 'Edit Hotel Listing' : 'Add New Hotel Listing' ?></h2>
                    <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
                    <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>

                    <form action="manage-hotels.php" method="POST" class="package-form" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                        <?php if ($edit_mode): ?><input type="hidden" name="id" value="<?= htmlspecialchars($hotel_to_edit['id']) ?>"><?php endif; ?>

                        <fieldset><legend>Basic Information</legend>
                            <div class="form-grid">
                                <div class="form-group"><label for="hotel_name">Hotel Name *</label><input type="text" id="hotel_name" name="hotel_name" value="<?= htmlspecialchars($hotel_to_edit['hotel_name'] ?? '') ?>" required></div>
                                <div class="form-group"><label for="location">Location *</label><input type="text" id="location" name="location" placeholder="e.g. Makkah, Madinah" value="<?= htmlspecialchars($hotel_to_edit['location'] ?? '') ?>" required></div>
                            </div>
                            <div class="form-group"><label for="image_url">Hotel Image</label>
                               <div class="image-upload-group">
                                    <input type="text" id="image_url" name="image_url" placeholder="Paste URL or upload/choose an image" value="<?= htmlspecialchars($hotel_to_edit['image_url'] ?? '') ?>" required>
                                    <div class="upload-row">
                                        <input type="file" id="hotel_image_upload" name="hotel_image_upload" accept="image/*">
                                        <button type="button" class="btn btn-secondary" id="openGalleryBtn">Choose from Gallery</button>
                                    </div>
                               </div>
                            </div>
                        </fieldset>

                        <fieldset><legend>Pricing & Rates</legend>
                             <div class="form-grid-4col">
                                <div class="form-group"><label for="price_per_night">Customer Price (SAR) *</label><input type="number" step="0.01" id="price_per_night" name="price_per_night" placeholder="e.g. 250.00" value="<?= htmlspecialchars($hotel_to_edit['price_per_night'] ?? '') ?>" required></div>
                                <div class="form-group"><label for="price_per_night_agent">Agent Price (SAR)</label><input type="number" step="0.01" id="price_per_night_agent" name="price_per_night_agent" placeholder="e.g. 240.00" value="<?= htmlspecialchars($hotel_to_edit['price_per_night_agent'] ?? '') ?>"></div>
                                <div class="form-group"><label for="agent_incentive">Agent Incentive (SAR)</label><input type="number" step="0.01" id="agent_incentive" name="agent_incentive" placeholder="e.g. 10.00" value="<?= htmlspecialchars($hotel_to_edit['agent_incentive'] ?? '0.00') ?>"></div>
                                <div class="form-group"><label for="exchange_rate_pkr">PKR Exchange Rate</label><input type="number" step="0.01" id="exchange_rate_pkr" name="exchange_rate_pkr" placeholder="e.g. 75.50 (optional)" value="<?= htmlspecialchars($hotel_to_edit['exchange_rate_pkr'] ?? '') ?>"></div>
                            </div>
                        </fieldset>

                        <fieldset><legend>Settings</legend>
                            <div class="form-grid">
                                <div class="form-group"><label for="rating">Star Rating *</label>
                                    <select id="rating" name="rating" required>
                                        <option value="5" <?= (($hotel_to_edit['rating'] ?? 5) == 5) ? 'selected' : '' ?>>5 Stars</option><option value="4" <?= (($hotel_to_edit['rating'] ?? 0) == 4) ? 'selected' : '' ?>>4 Stars</option><option value="3" <?= (($hotel_to_edit['rating'] ?? 0) == 3) ? 'selected' : '' ?>>3 Stars</option><option value="2" <?= (($hotel_to_edit['rating'] ?? 0) == 2) ? 'selected' : '' ?>>2 Stars</option><option value="1" <?= (($hotel_to_edit['rating'] ?? 0) == 1) ? 'selected' : '' ?>>1 Star</option>
                                    </select>
                                </div>
                                <div class="form-group"><label for="is_active">Status</label>
                                    <select id="is_active" name="is_active">
                                        <option value="1" <?= (($hotel_to_edit['is_active'] ?? 1) == 1) ? 'selected' : '' ?>>Active (Show on Website)</option><option value="0" <?= (($hotel_to_edit['is_active'] ?? 1) == 0) ? 'selected' : '' ?>>Inactive (Hide)</option>
                                    </select>
                                </div>
                                <div class="form-group"><label for="logo_background_style">Logo Background Style</label>
                                    <select id="logo_background_style" name="logo_background_style">
                                        <option value="white_bg" <?= (($hotel_to_edit['logo_background_style'] ?? 'white_bg') == 'white_bg') ? 'selected' : '' ?>>White Background (Default)</option>
                                        <option value="transparent" <?= (($hotel_to_edit['logo_background_style'] ?? '') == 'transparent') ? 'selected' : '' ?>>Transparent</option>
                                    </select>
                                </div>
                                <div class="form-group"><label for="display_order">Display Order</label><input type="number" id="display_order" name="display_order" placeholder="0 is highest" value="<?= htmlspecialchars($hotel_to_edit['display_order'] ?? '0') ?>"></div>
                            </div>
                        </fieldset>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Hotel' : 'Add Hotel' ?></button>
                            <?php if ($edit_mode): ?><a href="manage-hotels.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                        </div>
                    </form>
                </div>

                <div class="content-card">
                    <h2 class="form-title">Existing Hotel Listings</h2>
                    <form action="manage-hotels.php" method="POST" id="bulk-delete-form" onsubmit="return confirm('Are you sure you want to delete all selected hotels? This action cannot be undone.');">
                        <input type="hidden" name="action" value="bulk_delete">
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th><input type="checkbox" id="select-all-checkbox"></th>
                                        <th>Order</th>
                                        <th>Hotel Name</th>
                                        <th>Location</th>
                                        <th>Price (SAR)</th>
                                        <th>Exch. Rate</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if ($hotels_result && $hotels_result->num_rows > 0): while ($hotel = $hotels_result->fetch_assoc()): ?>
                                        <tr class="<?= ($edit_mode && isset($hotel_to_edit['id']) && $hotel['id'] === $hotel_to_edit['id']) ? 'editing' : '' ?>">
                                            <td data-label="Select"><input type="checkbox" name="hotel_ids[]" value="<?= $hotel['id'] ?>" class="row-checkbox"></td>
                                            <td data-label="Order"><?= htmlspecialchars($hotel['display_order']) ?></td>
                                            <td data-label="Name"><strong><?= htmlspecialchars($hotel['hotel_name']) ?></strong></td>
                                            <td data-label="Location"><?= htmlspecialchars($hotel['location']) ?></td>
                                            <td data-label="Price">SAR <?= htmlspecialchars(number_format($hotel['price_per_night'], 2)) ?></td>
                                            <td data-label="Exch. Rate"><?= htmlspecialchars($hotel['exchange_rate_pkr'] ?? 'Default') ?></td>
                                            <td data-label="Status"><span class="status-badge <?= $hotel['is_active'] ? 'status-completed' : 'status-rejected' ?>"><?= $hotel['is_active'] ? 'Active' : 'Inactive' ?></span></td>
                                            <td class="actions-cell" data-label="Actions">
                                                <a href="?action=edit&id=<?= $hotel['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                                <button type="submit" form="delete-form-<?= $hotel['id'] ?>" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                            </td>
                                        </tr>
                                    <?php endwhile; else: ?>
                                        <tr><td colspan="8" class="empty-state">No hotels found. Use the form above to add one.</td></tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="bulk-actions"><button type="submit" class="btn btn-danger">Delete Selected</button></div>
                    </form>

                    <?php if ($hotels_result && $hotels_result->data_seek(0) !== false): while ($hotel = $hotels_result->fetch_assoc()): ?>
                    <form action="manage-hotels.php" method="POST" id="delete-form-<?= $hotel['id'] ?>" onsubmit="return confirm('Are you sure you want to delete this hotel?');" style="display:none;">
                        <input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="<?= $hotel['id'] ?>">
                    </form>
                    <?php endwhile; endif; ?>
                </div>
            </main>
        </div>
    </div>
    
    <div id="galleryModal" class="modal">
        <div class="modal-content">
            <div class="modal-header"><h3 class="modal-title">Select an Image</h3><span class="close-button">&times;</span></div>
            <div class="gallery-grid">
                <?php if ($gallery_images_result && $gallery_images_result->num_rows > 0): while($image = $gallery_images_result->fetch_assoc()): ?>
                    <div class="gallery-item"><img src="../<?= htmlspecialchars($image['file_path']) ?>" alt="Gallery Image" data-path="<?= htmlspecialchars($image['file_path']) ?>"></div>
                <?php endwhile; else: ?><p>No images in gallery. Upload one to get started.</p><?php endif; ?>
            </div>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        document.getElementById('menu-toggle').addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
        const modal = document.getElementById("galleryModal"), btn = document.getElementById("openGalleryBtn"), span = document.getElementsByClassName("close-button")[0], imageUrlInput = document.getElementById("image_url"), galleryImages = document.querySelectorAll(".gallery-item img");
        if(btn) btn.onclick = () => modal.style.display = "block";
        if(span) span.onclick = () => modal.style.display = "none";
        window.onclick = (event) => { if (event.target == modal) modal.style.display = "none"; }
        galleryImages.forEach(item => { item.addEventListener('click', function() { galleryImages.forEach(img => img.classList.remove('selected')); this.classList.add('selected'); imageUrlInput.value = this.dataset.path; modal.style.display = "none"; }); });
        const selectAllCheckbox = document.getElementById('select-all-checkbox'), rowCheckboxes = document.querySelectorAll('.row-checkbox');
        if(selectAllCheckbox) { selectAllCheckbox.addEventListener('change', function() { rowCheckboxes.forEach(checkbox => { checkbox.checked = this.checked; }); }); }
    });
    </script>
</body>
</html>