<?php
session_start();
include_once '../db-config.php';

function e($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

// --- HANDLE DELETE REQUESTS ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- Single Delete Action ---
    if (isset($_POST['delete_quick_booking'])) {
        $booking_id_to_delete = (int)($_POST['booking_id'] ?? 0);
        if ($booking_id_to_delete > 0) {
            $conn->begin_transaction();
            try {
                // Cascading delete is handled by database constraints, but explicit deletion is safer
                $stmt_del_pass = $conn->prepare("DELETE FROM quick_booking_passengers WHERE booking_id = ?");
                $stmt_del_pass->bind_param("i", $booking_id_to_delete);
                $stmt_del_pass->execute();

                $stmt_del_book = $conn->prepare("DELETE FROM quick_bookings WHERE id = ?");
                $stmt_del_book->bind_param("i", $booking_id_to_delete);
                $stmt_del_book->execute();
                
                $conn->commit();
                $_SESSION['success_message'] = "Quick Booking #" . $booking_id_to_delete . " deleted successfully.";
            } catch (Exception $e) {
                $conn->rollback();
                $_SESSION['error_message'] = "Error deleting Quick Booking: " . $e->getMessage();
            }
        }
    }
    // --- Multi-Select Delete Action ---
    elseif (isset($_POST['delete_selected'])) {
        if (!empty($_POST['selected_ids']) && is_array($_POST['selected_ids'])) {
            $ids_to_delete = array_map('intval', $_POST['selected_ids']);
            if (!empty($ids_to_delete)) {
                $placeholders = implode(',', array_fill(0, count($ids_to_delete), '?'));
                $types = str_repeat('i', count($ids_to_delete));
                
                $conn->begin_transaction();
                try {
                    // Delete passengers first
                    $sql_del_pass = "DELETE FROM quick_booking_passengers WHERE booking_id IN ($placeholders)";
                    $stmt_del_pass = $conn->prepare($sql_del_pass);
                    $stmt_del_pass->bind_param($types, ...$ids_to_delete);
                    $stmt_del_pass->execute();

                    // Then delete main bookings
                    $sql_del_book = "DELETE FROM quick_bookings WHERE id IN ($placeholders)";
                    $stmt_del_book = $conn->prepare($sql_del_book);
                    $stmt_del_book->bind_param($types, ...$ids_to_delete);
                    $stmt_del_book->execute();

                    $conn->commit();
                    $_SESSION['success_message'] = count($ids_to_delete) . " booking(s) deleted successfully.";
                } catch (Exception $e) {
                    $conn->rollback();
                    $_SESSION['error_message'] = "Error during bulk deletion: " . $e->getMessage();
                }
            }
        } else {
            $_SESSION['error_message'] = "No bookings were selected for deletion.";
        }
    }
    header("Location: manage-quick-bookings.php");
    exit;
}

// --- SEARCH & PAGINATION SETUP ---
$search_term = $_GET['search'] ?? '';
$items_per_page = 20;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

// --- DYNAMICALLY BUILD SQL QUERY ---
$base_sql = "FROM quick_bookings qb 
             LEFT JOIN users u ON qb.user_id = u.id
             LEFT JOIN vouchers v ON qb.id = v.manual_no
             LEFT JOIN ticket_invoices ti ON qb.id = ti.quick_booking_id";
$where_clause = "";
$search_params = [];
$search_types = "";
if (!empty($search_term)) {
    $where_clause = " WHERE (qb.customer_name LIKE ? OR u.name LIKE ? OR qb.package_type LIKE ? OR qb.room_type LIKE ? OR qb.id LIKE ?)";
    $like_term = "%" . $search_term . "%";
    $search_params = [$like_term, $like_term, $like_term, $like_term, $like_term];
    $search_types = "sssss";
}

// Get total count for pagination
$count_sql = "SELECT COUNT(DISTINCT qb.id) as total " . $base_sql . $where_clause;
$stmt_count = $conn->prepare($count_sql);
if (!empty($search_params)) { $stmt_count->bind_param($search_types, ...$search_params); }
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_items / $items_per_page);

// Get paginated data
$data_sql = "
    SELECT qb.*, u.name as linked_user_name, u.user_type, u.company_name, v.id as voucher_id, v.invoice_id, ti.id as ticket_invoice_id,
           (SELECT COUNT(id) FROM quick_booking_passengers qbp WHERE qbp.booking_id = qb.id) as pax_count,
           (SELECT full_name FROM quick_booking_passengers qbp_head WHERE qbp_head.booking_id = qb.id AND qbp_head.is_family_head = 1 LIMIT 1) as family_head_name
    " . $base_sql . $where_clause . " GROUP BY qb.id ORDER BY qb.created_at DESC LIMIT ?, ?";
$data_params = array_merge($search_params, [$offset, $items_per_page]);
$data_types = $search_types . "ii";
$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($data_types, ...$data_params);
$stmt_data->execute();
$bookings_result = $stmt_data->get_result();

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Quick Bookings</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>
<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Manage Quick Bookings</span></div>
            </header>
            <main class="content-body">
                <div class="page-header" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 20px;">
                    <h1 class="page-title">Quick Bookings</h1>
                    <div class="header-actions" style="display: flex; gap: 1rem;">
                        <button type="submit" form="bulk-delete-form" name="delete_selected" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete all selected bookings?');"><i class="fas fa-trash"></i> Delete Selected</button>
                        <a href="create-quick-booking.php" class="btn btn-primary"><i class="fas fa-plus"></i> Create New Booking</a>
                    </div>
                </div>

                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>
                
                <div class="content-card">
                    <div class="card-body">
                        <div class="search-bar" style="max-width: 500px; margin-bottom: 20px;">
                            <form action="manage-quick-bookings.php" method="GET" style="display: flex;">
                                <input type="text" name="search" class="form-control" placeholder="Search by name, package, ID..." value="<?= e($search_term) ?>" style="border-top-right-radius:0; border-bottom-right-radius:0;">
                                <button type="submit" class="btn btn-primary" style="border-top-left-radius:0; border-bottom-left-radius:0;"><i class="fas fa-search"></i></button>
                            </form>
                        </div>

                        <form action="manage-quick-bookings.php" method="POST" id="bulk-delete-form">
                            <div class="table-responsive">
                                <table class="data-table">
                                    <thead>
                                        <tr>
                                            <th style="width: 40px;"><input type="checkbox" id="select-all-checkbox"></th>
                                            <th>ID</th><th>Date</th><th>Family Head</th><th>Pax</th><th>Package</th><th>Status</th><th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($bookings_result && $bookings_result->num_rows > 0): while ($booking = $bookings_result->fetch_assoc()): ?>
                                            <tr>
                                                <td><input type="checkbox" name="selected_ids[]" class="row-checkbox" value="<?= $booking['id'] ?>"></td>
                                                <td><strong>#<?= e($booking['id']); ?></strong></td>
                                                <td><?= date('d M, Y', strtotime($booking['created_at'])); ?></td>
                                                <td>
                                                    <?= e($booking['family_head_name'] ?: $booking['customer_name']); ?><br>
                                                    <small style="color:var(--text-muted)"><?= e($booking['linked_user_name'] ?: 'Direct'); ?></small>
                                                </td>
                                                <td><?= e($booking['pax_count']); ?></td>
                                                <td><?= e($booking['package_type']); ?><br><small><?= e($booking['room_type']) ?></small></td>
                                                <td><span class="status-badge status-<?= strtolower(e($booking['status'])) ?>"><?= e($booking['status']); ?></span></td>
                                                <td class="actions-cell">
                                                    <div style="display: flex; gap: 0.5rem;">
                                                        <a href="edit-quick-booking.php?id=<?= $booking['id']; ?>" class="btn btn-sm btn-primary" title="Edit Booking"><i class="fas fa-edit"></i></a>
                                                        <?php if (empty($booking['voucher_id'])): ?>
                                                            <a href="create-voucher.php?quick_booking_id=<?= $booking['id']; ?>" class="btn btn-sm btn-success" title="Create Voucher"><i class="fas fa-ticket-alt"></i></a>
                                                        <?php endif; ?>
                                                        <form action="manage-quick-bookings.php" method="POST" onsubmit="return confirm('Delete this quick booking?');" style="display:inline;">
                                                            <input type="hidden" name="booking_id" value="<?= $booking['id'] ?>">
                                                            <button type="submit" name="delete_quick_booking" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                                        </form>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endwhile; else: ?>
                                            <tr><td colspan="8" class="text-center p-4">No quick bookings found.</td></tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                        </form>

                        <?php if ($total_pages > 1): ?>
                            <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                                <?php if ($current_page > 1): ?><a href="?search=<?= e($search_term) ?>&page=<?= $current_page - 1 ?>" class="btn btn-secondary btn-sm">« Previous</a><?php endif; ?>
                                <span style="padding: 0 10px;">Page <?= $current_page ?> of <?= $total_pages ?></span>
                                <?php if ($current_page < $total_pages): ?><a href="?search=<?= e($search_term) ?>&page=<?= $current_page + 1 ?>" class="btn btn-secondary btn-sm">Next »</a><?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const menuToggle = document.getElementById('menu-toggle');
        if (menuToggle) {
            menuToggle.addEventListener('click', () => document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open'));
        }

        const selectAllCheckbox = document.getElementById('select-all-checkbox');
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                document.querySelectorAll('.row-checkbox').forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
            });
        }
    });
</script>
</body>
</html>