<?php
/**
 * manage-visa-services.php
 * - Self-contained CRUD system for managing visa service listings.
 * - Handles automatic page creation/deletion based on a slug.
 * - Expanded with detailed fields for individual visa pages.
 * - UPDATE: Changed pricing to SAR with a PKR exchange rate. Agent Incentive removed.
 * - UPDATE: Made the entire interface responsive for mobile and tablet use.
 * - UPDATE: Added image uploader and gallery selection.
 * - UPDATE: Added city selection and management.
 * - FIX: Corrected image upload path to use the root /images/ folder.
 * - MAJOR UPDATE: Implemented Many-to-Many relationship for Service Cities using Select2.
 * - MAJOR UPDATE: Added a 'Destination City' field.
 * - MAJOR UPDATE: Overhauled City Management with full CRUD and multi-delete.
 * - FIX (2024-05-21): Corrected a typo in the bind_param type string for the UPDATE query.
 */

// Handles security, sessions, and the database connection ($conn)
include_once '../db-config.php';

// =======================================================
// SECURITY GATEWAY
// =======================================================
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: login.php");
    exit;
}

// =======================================================
//  1. SETUP & HELPER FUNCTIONS
// =======================================================

// --- Page Generation Helpers ---
define('PAGE_TEMPLATE_PATH', '../visa-service-template.php');
define('GENERATED_PAGES_DIR', '../');

// --- Image Upload Helpers (UPDATED PATHS) ---
define('VISA_IMAGE_UPLOAD_DIR', '../images/'); 
define('VISA_IMAGE_DB_PATH', 'images/');      

if (!is_dir(VISA_IMAGE_UPLOAD_DIR)) {
    mkdir(VISA_IMAGE_UPLOAD_DIR, 0777, true);
}


function sanitize_slug($slug) {
    if ($slug === null) return '';
    $slug = trim($slug);
    $slug = strtolower($slug);
    $slug = preg_replace('/[\s_]+/', '-', $slug);
    $slug = preg_replace('/[^a-z0-9\-]/', '', $slug);
    return trim($slug, '-');
}

function create_visa_page($slug) {
    if (empty($slug)) return ['success' => true]; 
    $sanitized_slug = sanitize_slug($slug);
    if (empty($sanitized_slug)) return ['success' => true];
    $new_filepath = GENERATED_PAGES_DIR . $sanitized_slug . '.php';
    if (!file_exists(PAGE_TEMPLATE_PATH)) return ['success' => false, 'message' => "Template file 'visa-service-template.php' not found."];
    if (file_exists($new_filepath)) return ['success' => false, 'message' => "A page with this slug already exists."];
    $template_content = file_get_contents(PAGE_TEMPLATE_PATH);
    if ($template_content === false) return ['success' => false, 'message' => "Could not read template file."];
    if (file_put_contents($new_filepath, $template_content) === false) return ['success' => false, 'message' => "Could not create page. Check server permissions."];
    return ['success' => true, 'message' => "Page '{$sanitized_slug}.php' created."];
}

function delete_visa_page($slug) {
    if (empty($slug)) return ['success' => true];
    $sanitized_slug = sanitize_slug($slug);
    if (empty($sanitized_slug)) return ['success' => true];
    $filepath = GENERATED_PAGES_DIR . $sanitized_slug . '.php';
    if (file_exists($filepath)) {
        if (!is_writable($filepath)) return ['success' => false, 'message' => "Could not delete page '{$filepath}'. Check permissions."];
        if (!unlink($filepath)) return ['success' => false, 'message' => "Failed to delete page."];
    }
    return ['success' => true];
}


//======================================================================
//  2. HANDLE FORM SUBMISSIONS (CREATE, UPDATE, DELETE, CITIES)
//======================================================================

$edit_mode = false;
$visa_to_edit = null;
$assigned_city_ids = [];

if (isset($_SESSION['success_message'])) { $success_message = $_SESSION['success_message']; unset($_SESSION['success_message']); }
if (isset($_SESSION['error_message'])) { $error_message = $_SESSION['error_message']; unset($_SESSION['error_message']); }

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['action'])) {
    
    // --- MANAGE CITIES ---
    if ($_POST['action'] === 'add_city') {
        $city_name = trim($_POST['city_name']);
        if (!empty($city_name)) {
            $stmt = $conn->prepare("INSERT INTO cities (name) VALUES (?)");
            $stmt->bind_param("s", $city_name);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "City '{$city_name}' added successfully.";
            } else {
                $_SESSION['error_message'] = "Failed to add city. It might already exist.";
            }
        }
        header("Location: manage-visa-services.php#cities-section");
        exit();
    }
    
    if ($_POST['action'] === 'update_city') {
        $city_name = trim($_POST['city_name']);
        $city_id = (int)$_POST['id'];
        if (!empty($city_name) && $city_id > 0) {
            $stmt = $conn->prepare("UPDATE cities SET name = ? WHERE id = ?");
            $stmt->bind_param("si", $city_name, $city_id);
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "City updated successfully.";
            } else {
                $_SESSION['error_message'] = "Failed to update city.";
            }
        }
        header("Location: manage-visa-services.php#cities-section");
        exit();
    }

    if ($_POST['action'] === 'delete_city' && isset($_POST['id'])) {
        $stmt = $conn->prepare("DELETE FROM cities WHERE id = ?");
        $stmt->bind_param("i", $_POST['id']);
        if ($stmt->execute()) {
            $_SESSION['success_message'] = "City deleted successfully.";
        } else {
            $_SESSION['error_message'] = "Failed to delete city.";
        }
        header("Location: manage-visa-services.php#cities-section");
        exit();
    }
    
    if ($_POST['action'] === 'delete_multiple_cities' && !empty($_POST['city_ids'])) {
        $city_ids = array_map('intval', $_POST['city_ids']);
        $placeholders = implode(',', array_fill(0, count($city_ids), '?'));
        $types = str_repeat('i', count($city_ids));
        
        $stmt = $conn->prepare("DELETE FROM cities WHERE id IN ($placeholders)");
        $stmt->bind_param($types, ...$city_ids);
        
        if ($stmt->execute()) {
            $_SESSION['success_message'] = count($city_ids) . " cities deleted successfully.";
        } else {
            $_SESSION['error_message'] = "Failed to delete cities.";
        }
        header("Location: manage-visa-services.php#cities-section");
        exit();
    }


    // --- MANAGE VISAS ---
    $price = !empty($_POST['price']) ? (float)$_POST['price'] : null;
    $price_agent = !empty($_POST['price_agent']) ? (float)$_POST['price_agent'] : null;
    $exchange_rate = !empty($_POST['exchange_rate']) ? (float)$_POST['exchange_rate'] : 75.00;
    
    $image_url_to_save = $_POST['image_url']; 
    if (isset($_FILES['visa_image_upload']) && $_FILES['visa_image_upload']['error'] == 0) {
        $file_name = time() . '_' . basename($_FILES["visa_image_upload"]["name"]);
        $target_file = VISA_IMAGE_UPLOAD_DIR . $file_name;
        $imageFileType = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));
        $check = getimagesize($_FILES["visa_image_upload"]["tmp_name"]);
        if($check !== false) {
            if (move_uploaded_file($_FILES["visa_image_upload"]["tmp_name"], $target_file)) {
                $image_url_to_save = VISA_IMAGE_DB_PATH . $file_name;
                $stmt_gallery = $conn->prepare("INSERT INTO image_gallery (file_path) VALUES (?)");
                $stmt_gallery->bind_param("s", $image_url_to_save);
                $stmt_gallery->execute();
            } else {
                $_SESSION['error_message'] = "Sorry, there was an error uploading your file. Check folder permissions for '/images/'.";
                header("Location: manage-visa-services.php");
                exit();
            }
        } else {
            $_SESSION['error_message'] = "File is not a valid image.";
            header("Location: manage-visa-services.php");
            exit();
        }
    }


    // --- ADD A NEW VISA ---
    if ($_POST['action'] === 'add') {
        try {
            $conn->begin_transaction();
            $page_link_slug = sanitize_slug($_POST['page_link']);
            
            $sql = "INSERT INTO visas (visa_name, page_link, visa_type, destination_city, image_url, image_alt, overview, documents_required, how_to_apply, fees_and_charges, important_notes, processing_time, entry_type, price, price_agent, exchange_rate, price_note, is_active) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = $conn->prepare($sql);
            if ($stmt === false) { throw new Exception("Database prepare failed: " . $conn->error); }
            
            $stmt->bind_param("sssssssssssssdddsi", 
                $_POST['visa_name'], $page_link_slug, $_POST['visa_type'], $_POST['destination_city'], $image_url_to_save, $_POST['image_alt'], 
                $_POST['overview'], $_POST['documents_required'], $_POST['how_to_apply'], $_POST['fees_and_charges'], 
                $_POST['important_notes'], $_POST['processing_time'], $_POST['entry_type'], $price, $price_agent, 
                $exchange_rate, $_POST['price_note'], $_POST['is_active']
            );
            
            if (!$stmt->execute()) { throw new Exception("Database execution failed: " . $stmt->error); }
            $new_visa_id = $conn->insert_id;

            // Handle Service Cities
            if (!empty($_POST['service_cities']) && is_array($_POST['service_cities'])) {
                $stmt_city = $conn->prepare("INSERT INTO visa_cities (visa_id, city_id) VALUES (?, ?)");
                foreach ($_POST['service_cities'] as $city_id) {
                    $stmt_city->bind_param("ii", $new_visa_id, $city_id);
                    $stmt_city->execute();
                }
            }
            
            $conn->commit();
            $_SESSION['success_message'] = "Visa '{$_POST['visa_name']}' was added successfully. ";

            $page_result = create_visa_page($page_link_slug);
            if ($page_result['success']) {
                if (isset($page_result['message'])) $_SESSION['success_message'] .= $page_result['message'];
            } else {
                $_SESSION['error_message'] = "DB record created, but page creation failed: " . $page_result['message'];
            }

        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
        }
        header("Location: manage-visa-services.php");
        exit();
    }

    // --- UPDATE AN EXISTING VISA ---
    if ($_POST['action'] === 'update' && isset($_POST['id'])) {
        try {
            $conn->begin_transaction();
            $visa_id = (int)$_POST['id'];

            $stmt_old = $conn->prepare("SELECT page_link FROM visas WHERE id = ?");
            $stmt_old->bind_param("i", $visa_id);
            $stmt_old->execute();
            $old_visa_data = $stmt_old->get_result()->fetch_assoc();
            $old_slug = $old_visa_data['page_link'] ?? '';
            $new_slug = sanitize_slug($_POST['page_link']);
            $page_op_message = '';

            if ($new_slug !== $old_slug) {
                delete_visa_page($old_slug);
                $create_result = create_visa_page($new_slug);
                if ($create_result['success']) {
                    if (isset($create_result['message'])) $page_op_message .= $create_result['message'];
                } else {
                    $page_op_message .= "Error (New Page): " . $create_result['message'];
                }
            }

            $sql = "UPDATE visas SET 
                     visa_name = ?, page_link = ?, visa_type = ?, destination_city = ?, image_url = ?, image_alt = ?, 
                     overview = ?, documents_required = ?, how_to_apply = ?, fees_and_charges = ?, important_notes = ?,
                     processing_time = ?, entry_type = ?, price = ?, price_agent = ?, exchange_rate = ?, price_note = ?, is_active = ?
                     WHERE id = ?";
            
            $stmt = $conn->prepare($sql);
            if ($stmt === false) { throw new Exception("Database prepare failed: " . $conn->error); }
            
            // <<< THIS IS THE CORRECTED LINE >>>
            // The type string now has 19 characters to match the 19 variables.
            $stmt->bind_param("sssssssssssssdddsii",
                $_POST['visa_name'], $new_slug, $_POST['visa_type'], $_POST['destination_city'], $image_url_to_save, $_POST['image_alt'], 
                $_POST['overview'], $_POST['documents_required'], $_POST['how_to_apply'], $_POST['fees_and_charges'], 
                $_POST['important_notes'], $_POST['processing_time'], $_POST['entry_type'], $price, $price_agent, 
                $exchange_rate, $_POST['price_note'], $_POST['is_active'], $visa_id
            );

            if (!$stmt->execute()) { throw new Exception("Database execution failed: " . $stmt->error); }
            
            // Handle Service Cities
            $stmt_delete_cities = $conn->prepare("DELETE FROM visa_cities WHERE visa_id = ?");
            $stmt_delete_cities->bind_param("i", $visa_id);
            $stmt_delete_cities->execute();
            
            if (!empty($_POST['service_cities']) && is_array($_POST['service_cities'])) {
                $stmt_city = $conn->prepare("INSERT INTO visa_cities (visa_id, city_id) VALUES (?, ?)");
                foreach ($_POST['service_cities'] as $city_id) {
                    $stmt_city->bind_param("ii", $visa_id, $city_id);
                    $stmt_city->execute();
                }
            }
            
            $conn->commit();
            $_SESSION['success_message'] = "Visa '{$_POST['visa_name']}' was updated successfully. " . $page_op_message;

        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
        }
        header("Location: manage-visa-services.php");
        exit();
    }
    
    // --- DELETE A VISA ---
    if ($_POST['action'] === 'delete' && isset($_POST['id'])) {
        $stmt_select = $conn->prepare("SELECT page_link FROM visas WHERE id = ?");
        $stmt_select->bind_param("i", $_POST['id']);
        $stmt_select->execute();
        $result = $stmt_select->get_result();
        if ($row = $result->fetch_assoc()) {
            delete_visa_page($row['page_link']);
        }
        $stmt_select->close();

        // Deletion from `visa_cities` will happen automatically due to ON DELETE CASCADE
        $stmt_delete = $conn->prepare("DELETE FROM visas WHERE id = ?");
        $stmt_delete->bind_param("i", $_POST['id']);
        $stmt_delete->execute();
        $_SESSION['success_message'] = "The visa service and its associated page were deleted successfully!";
        header("Location: manage-visa-services.php");
        exit();
    }
}

// =======================================================
//  3. FETCH DATA FOR PAGE DISPLAY
// =======================================================

// --- PREPARE FOR EDITING ---
if (isset($_GET['action']) && $_GET['action'] === 'edit' && isset($_GET['id'])) {
    $edit_mode = true;
    $edit_id = (int)$_GET['id'];
    $stmt = $conn->prepare("SELECT * FROM visas WHERE id = ?");
    $stmt->bind_param("i", $edit_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $visa_to_edit = $result->fetch_assoc();
    
    // Fetch assigned cities for this visa
    $stmt_cities = $conn->prepare("SELECT city_id FROM visa_cities WHERE visa_id = ?");
    $stmt_cities->bind_param("i", $edit_id);
    $stmt_cities->execute();
    $cities_result = $stmt_cities->get_result();
    while ($row = $cities_result->fetch_assoc()) {
        $assigned_city_ids[] = $row['city_id'];
    }
}

// Fetch all existing visas to display in the table
$visas_query = "SELECT v.*, GROUP_CONCAT(c.name SEPARATOR ', ') as service_cities_list
                FROM visas v
                LEFT JOIN visa_cities vc ON v.id = vc.visa_id
                LEFT JOIN cities c ON vc.city_id = c.id
                GROUP BY v.id
                ORDER BY v.id DESC";
$visas_result = $conn->query($visas_query);

// Fetch all cities for the dropdowns and management table
$cities_result = $conn->query("SELECT * FROM cities ORDER BY name ASC");
$cities_for_dropdown = [];
if ($cities_result && $cities_result->num_rows > 0) {
    while($row = $cities_result->fetch_assoc()) {
        $cities_for_dropdown[] = $row;
    }
    $cities_result->data_seek(0); 
}


// Fetch all gallery images
$gallery_images_result = $conn->query("SELECT id, file_path FROM image_gallery ORDER BY id DESC");

$current_page = basename($_SERVER['PHP_SELF']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Visa Services</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <!-- Select2 CSS -->
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        .package-form fieldset { border: 1px solid var(--border-color); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem; background-color: #fdfdfd; }
        .package-form legend { font-weight: 600; padding: 0 10px; font-size: 1.1rem; color: var(--primary-color); }
        .form-hint { font-size: 0.8em; color: #777; display: block; margin-top: 4px; }
        .details-grid-3col { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; }
        .image-upload-group { display: flex; align-items: center; gap: 1rem; margin-top: 0.5rem;}
        .image-upload-group input[type="file"] { flex-grow: 1; }

        /* Select2 Styles */
        .select2-container .select2-selection--multiple { border: 1px solid #ccc; border-radius: 5px; padding: 4px; }
        .select2-container--default .select2-selection--multiple .select2-selection__choice { background-color: var(--primary-color); color: white; border: none; }
        .select2-container--default .select2-selection--multiple .select2-selection__choice__remove { color: white; }

        /* Gallery Modal Styles */
        .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.6); }
        .modal-content { background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 1200px; border-radius: 8px; }
        .modal-header { display: flex; justify-content: space-between; align-items: center; border-bottom: 1px solid #ddd; padding-bottom: 10px; margin-bottom: 20px; }
        .modal-title { font-size: 1.5rem; color: var(--text-dark); }
        .close-button { color: #aaa; font-size: 28px; font-weight: bold; cursor: pointer; }
        .close-button:hover, .close-button:focus { color: black; text-decoration: none; }
        .gallery-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 15px; max-height: 60vh; overflow-y: auto; }
        .gallery-item img { width: 100%; height: 120px; object-fit: cover; border-radius: 4px; cursor: pointer; border: 2px solid transparent; transition: border-color 0.2s; }
        .gallery-item img:hover { border-color: var(--primary-color); }

        /* Responsive Styles */
        @media (max-width: 992px) {
            .details-grid-3col, .form-grid { grid-template-columns: 1fr 1fr; }
        }
        @media (max-width: 768px) {
            .form-grid, .details-grid-3col { grid-template-columns: 1fr; }
            .content-body { padding: 1rem; }
            .package-form fieldset { padding: 1rem; }
            .modal-content { width: 95%; margin-top: 10%; }
        }

        .table-responsive { width: 100%; overflow-x: auto; -webkit-overflow-scrolling: touch; }
        .data-table { width: 100%; min-width: 1100px; }
    </style>
</head>
<body>
<div class="dashboard-wrapper">
    <?php include 'sidebar.php'; ?>
    <div class="main-content">
        <header class="main-header">
            <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
            <div class="user-info"><span>Welcome, <?php echo htmlspecialchars(ucfirst($_SESSION['user_name'] ?? 'Admin')); ?></span></div>
        </header>
        <main class="content-body" id="top">
            <div class="content-card">
                <h2 class="form-title" style="margin-bottom: 2rem;"><?= $edit_mode ? 'Edit Visa Service' : 'Add New Visa Service' ?></h2>
                <?php if (isset($success_message)): ?><div class="notice success"><?= htmlspecialchars($success_message) ?></div><?php endif; ?>
                <?php if (isset($error_message)): ?><div class="notice error"><?= htmlspecialchars($error_message) ?></div><?php endif; ?>

                <form action="manage-visa-services.php" method="POST" class="package-form" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="<?= $edit_mode ? 'update' : 'add' ?>">
                    <?php if($edit_mode): ?><input type="hidden" name="id" value="<?= $visa_to_edit['id'] ?>"><?php endif; ?>
                    
                    <fieldset>
                        <legend>Core Details</legend>
                        <div class="form-grid">
                            <div class="form-group"><label for="visa_name">Visa Name *</label><input type="text" id="visa_name" name="visa_name" placeholder="e.g. UAE / Dubai Visa" value="<?= htmlspecialchars($visa_to_edit['visa_name'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="visa_type">Visa Type *</label><input type="text" id="visa_type" name="visa_type" placeholder="e.g. 30-Day Tourist Visa" value="<?= htmlspecialchars($visa_to_edit['visa_type'] ?? '') ?>" required></div>
                            <div class="form-group"><label for="destination_city">Destination City</label><input type="text" id="destination_city" name="destination_city" placeholder="e.g. Jeddah" value="<?= htmlspecialchars($visa_to_edit['destination_city'] ?? '') ?>"></div>
                            <div class="form-group"><label for="processing_time">Processing Time</label><input type="text" id="processing_time" name="processing_time" placeholder="e.g. 5-7 Working Days" value="<?= htmlspecialchars($visa_to_edit['processing_time'] ?? '') ?>"></div>
                            <div class="form-group"><label for="entry_type">Entry Type</label><input type="text" id="entry_type" name="entry_type" placeholder="e.g. Single Entry" value="<?= htmlspecialchars($visa_to_edit['entry_type'] ?? '') ?>"></div>
                             <div class="form-group"><label for="service_cities">Service Cities (where available) *</label>
                                <select id="service_cities" name="service_cities[]" class="select2-multiple" multiple="multiple" required style="width: 100%;">
                                    <?php foreach ($cities_for_dropdown as $city): ?>
                                        <option value="<?= $city['id'] ?>" <?= in_array($city['id'], $assigned_city_ids) ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($city['name']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </fieldset>

                    <fieldset>
                        <legend>Pricing (SAR)</legend>
                        <div class="details-grid-3col">
                            <div class="form-group"><label for="price">Customer Price (SAR)</label><input type="number" step="0.01" id="price" name="price" placeholder="e.g. 550.00" value="<?= htmlspecialchars($visa_to_edit['price'] ?? '') ?>"></div>
                            <div class="form-group"><label for="price_agent">Agent Price (SAR)</label><input type="number" step="0.01" id="price_agent" name="price_agent" placeholder="e.g. 525.00" value="<?= htmlspecialchars($visa_to_edit['price_agent'] ?? '') ?>"></div>
                            <div class="form-group"><label for="exchange_rate">Exchange Rate (to PKR) *</label><input type="number" step="0.0001" id="exchange_rate" name="exchange_rate" placeholder="e.g. 75.50" value="<?= htmlspecialchars($visa_to_edit['exchange_rate'] ?? '75.00') ?>" required></div>
                        </div>
                        <div class="form-group" style="margin-top: 1rem;"><label for="price_note">Price Note</label><input type="text" id="price_note" name="price_note" placeholder="e.g. Per Person" value="<?= htmlspecialchars($visa_to_edit['price_note'] ?? '') ?>"></div>
                    </fieldset>

                    <fieldset>
                        <legend>Website Content</legend>
                        <div class="form-group"><label for="image_url">Image Link (URL) *</label><input type="text" id="image_url" name="image_url" placeholder="Paste URL here, or upload a new image below" value="<?= htmlspecialchars($visa_to_edit['image_url'] ?? '') ?>" required></div>
                        <div class="form-group">
                            <label for="visa_image_upload">Or Upload New Image (will be saved in /images/ folder)</label>
                            <div class="image-upload-group">
                                <input type="file" id="visa_image_upload" name="visa_image_upload" accept="image/png, image/jpeg, image/jpg, image/webp">
                                <button type="button" class="btn btn-secondary" id="openGalleryBtn">Choose from Gallery</button>
                            </div>
                        </div>
                        <div class="form-group"><label for="image_alt">Image Alt Text *</label><input type="text" id="image_alt" name="image_alt" placeholder="e.g. Dubai skyline with Burj Khalifa" value="<?= htmlspecialchars($visa_to_edit['image_alt'] ?? '') ?>" required></div>
                        <div class="form-group"><label for="page_link">Page Link (URL Slug)</label><input type="text" id="page_link" name="page_link" placeholder="e.g. 30-day-dubai-tourist-visa" value="<?= htmlspecialchars($visa_to_edit['page_link'] ?? '') ?>"><small class="form-hint">Creates a page like `your-slug.php`. Leave blank for no page.</small></div>
                        <div class="form-group"><label for="overview">Overview</label><textarea id="overview" name="overview" rows="4" placeholder="A brief description of the visa service."><?= htmlspecialchars($visa_to_edit['overview'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="documents_required">Documents Required</label><small class="form-hint">Enter each document on a new line.</small><textarea id="documents_required" name="documents_required" rows="6" placeholder="- Scanned Passport Copy&#10;- Passport-size Photo"><?= htmlspecialchars($visa_to_edit['documents_required'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="how_to_apply">How to Apply</label><small class="form-hint">Enter each step on a new line.</small><textarea id="how_to_apply" name="how_to_apply" rows="5" placeholder="1. Submit your documents.&#10;2. Make the payment.&#10;3. Receive your e-visa via email."><?= htmlspecialchars($visa_to_edit['how_to_apply'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="fees_and_charges">Fees & Charges</label><small class="form-hint">Provide a breakdown of costs.</small><textarea id="fees_and_charges" name="fees_and_charges" rows="4" placeholder="Visa Fee: $100&#10;Service Fee: $50&#10;Total: $150"><?= htmlspecialchars($visa_to_edit['fees_and_charges'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="important_notes">Important Notes</label><small class="form-hint">Add any disclaimers or important information.</small><textarea id="important_notes" name="important_notes" rows="4" placeholder="- Fees are non-refundable.&#10;- Approval is subject to immigration."><?= htmlspecialchars($visa_to_edit['important_notes'] ?? '') ?></textarea></div>
                        <div class="form-group"><label for="is_active">Show on Website?</label>
                            <select id="is_active" name="is_active">
                                <option value="1" <?= (($visa_to_edit['is_active'] ?? 1) == 1) ? 'selected' : '' ?>>Yes (Active)</option>
                                <option value="0" <?= (($visa_to_edit['is_active'] ?? 1) == 0) ? 'selected' : '' ?>>No (Inactive)</option>
                            </select>
                        </div>
                    </fieldset>

                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary"><?= $edit_mode ? 'Update Visa' : 'Add Visa' ?></button>
                        <?php if($edit_mode): ?><a href="manage-visa-services.php" class="btn btn-secondary">Cancel Edit</a><?php endif; ?>
                    </div>
                </form>
            </div>

            <div class="content-card">
                 <h2 class="form-title">Existing Visa Services</h2>
                 <div class="table-responsive">
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Visa Name</th>
                                <th>Service Cities</th>
                                <th>Destination</th>
                                <th>Page Link</th>
                                <th>Price (SAR)</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($visas_result && $visas_result->num_rows > 0): ?>
                                <?php while ($visa = $visas_result->fetch_assoc()): ?>
                                    <tr class="<?= ($edit_mode && isset($visa_to_edit['id']) && $visa['id'] === $visa_to_edit['id']) ? 'editing' : '' ?>">
                                        <td><strong><?= htmlspecialchars($visa['visa_name']) ?></strong><br><small><?= htmlspecialchars($visa['visa_type']) ?></small></td>
                                        <td><?= htmlspecialchars($visa['service_cities_list'] ?? 'N/A') ?></td>
                                        <td><?= htmlspecialchars($visa['destination_city'] ?? 'N/A') ?></td>
                                        <td>
                                            <?php if (!empty($visa['page_link'])): ?>
                                                <a href="../<?= htmlspecialchars($visa['page_link']) ?>.php" target="_blank"><?= htmlspecialchars($visa['page_link']) ?>.php <i class="fas fa-external-link-alt fa-xs"></i></a>
                                            <?php else: ?>
                                                <span style="color: #999;">No page</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>SAR <?= isset($visa['price']) ? htmlspecialchars(number_format($visa['price'], 2)) : 'N/A' ?></td>
                                        <td>
                                            <span class="status-badge <?= $visa['is_active'] ? 'status-completed' : 'status-rejected' ?>">
                                                <?= $visa['is_active'] ? 'Active' : 'Inactive' ?>
                                            </span>
                                        </td>
                                        <td class="actions-cell">
                                            <a href="?action=edit&id=<?= $visa['id'] ?>#top" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                            <form action="manage-visa-services.php" method="POST" onsubmit="return confirm('Are you sure you want to delete this visa service and its page? This cannot be undone.');" style="display:inline;">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="id" value="<?= $visa['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr><td colspan="7" class="empty-state">No visa services found. Use the form above to add one.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                 </div>
            </div>

            <div class="content-card" id="cities-section">
                <form id="cities-form" action="manage-visa-services.php" method="POST">
                    <input type="hidden" name="action" value="delete_multiple_cities">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                        <h2 class="form-title">Manage Service Cities</h2>
                        <button type="submit" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete the selected cities?');">Delete Selected</button>
                    </div>
                    <div class="table-responsive">
                        <table class="data-table" style="min-width: 0;">
                            <thead>
                                <tr>
                                    <th style="width: 40px;"><input type="checkbox" id="selectAllCities"></th>
                                    <th>City Name</th>
                                    <th style="width: 150px;">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($cities_result && $cities_result->num_rows > 0): ?>
                                    <?php while ($city = $cities_result->fetch_assoc()): ?>
                                    <tr id="city-row-<?= $city['id'] ?>">
                                        <td><input type="checkbox" name="city_ids[]" value="<?= $city['id'] ?>" class="city-checkbox"></td>
                                        <td class="city-name-cell"><?= htmlspecialchars($city['name']) ?></td>
                                        <td class="actions-cell">
                                            <button type="button" class="btn btn-sm btn-secondary edit-city-btn" data-id="<?= $city['id'] ?>" data-name="<?= htmlspecialchars($city['name']) ?>"><i class="fas fa-edit"></i> Edit</button>
                                            <form action="manage-visa-services.php" method="POST" onsubmit="return confirm('Are you sure you want to delete this city?');" style="display:inline;">
                                                <input type="hidden" name="action" value="delete_city">
                                                <input type="hidden" name="id" value="<?= $city['id'] ?>">
                                                <button type="submit" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr><td colspan="3" class="empty-state">No cities found.</td></tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
                <div class="form-grid" style="margin-top: 2rem;">
                    <div class="form-group">
                        <label for="city_name"><strong>Add New City</strong></label>
                        <form action="manage-visa-services.php" method="POST" style="display: flex; gap: 1rem;">
                            <input type="hidden" name="action" value="add_city">
                            <input type="text" id="city_name" name="city_name" placeholder="e.g. Lahore" required style="flex-grow: 1;">
                            <button type="submit" class="btn btn-primary">Add City</button>
                        </form>
                    </div>
                </div>
            </div>

        </main>
    </div>
</div>

<!-- Gallery Modal HTML -->
<div id="galleryModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 class="modal-title">Select an Image</h3>
            <span class="close-button">&times;</span>
        </div>
        <div class="gallery-grid">
            <?php if ($gallery_images_result && $gallery_images_result->num_rows > 0): ?>
                <?php while($image = $gallery_images_result->fetch_assoc()): ?>
                    <div class="gallery-item">
                        <img src="../<?= htmlspecialchars($image['file_path']) ?>" alt="Gallery Image">
                    </div>
                <?php endwhile; ?>
            <?php else: ?>
                <p>No images in the gallery. Upload one to get started.</p>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- jQuery (required for Select2) -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<!-- Select2 JS -->
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Select2
    $('.select2-multiple').select2({
        placeholder: "Select cities",
        allowClear: true
    });

    // Sidebar toggle
    const menuToggle = document.getElementById('menu-toggle');
    if(menuToggle) {
        menuToggle.addEventListener('click', function() {
            document.querySelector('.dashboard-wrapper').classList.toggle('sidebar-open');
        });
    }

    // Gallery Modal Logic
    const modal = document.getElementById("galleryModal");
    const btn = document.getElementById("openGalleryBtn");
    const span = document.getElementsByClassName("close-button")[0];
    const galleryItems = document.querySelectorAll(".gallery-item img");
    const imageUrlInput = document.getElementById("image_url");

    if (btn) btn.onclick = () => { modal.style.display = "block"; }
    if (span) span.onclick = () => { modal.style.display = "none"; }
    window.onclick = (event) => { if (event.target == modal) modal.style.display = "none"; }

    galleryItems.forEach(item => {
        item.addEventListener('click', function() {
            const rawSrc = this.getAttribute('src');
            const selectedImageUrl = rawSrc.replace('../', '');
            if (imageUrlInput) imageUrlInput.value = selectedImageUrl;
            modal.style.display = "none";
        });
    });

    // City Management Logic
    document.getElementById('selectAllCities')?.addEventListener('change', function(e) {
        document.querySelectorAll('.city-checkbox').forEach(checkbox => {
            checkbox.checked = e.target.checked;
        });
    });

    document.querySelectorAll('.edit-city-btn').forEach(button => {
        button.addEventListener('click', function() {
            const id = this.dataset.id;
            const name = this.dataset.name;
            const row = document.getElementById(`city-row-${id}`);
            const nameCell = row.querySelector('.city-name-cell');
            
            nameCell.innerHTML = `
                <form action="manage-visa-services.php" method="POST" style="display: flex; gap: 5px;">
                    <input type="hidden" name="action" value="update_city">
                    <input type="hidden" name="id" value="${id}">
                    <input type="text" name="city_name" value="${name}" required style="width: 100%; padding: 8px;">
                    <button type="submit" class="btn btn-sm btn-primary">Save</button>
                    <button type="button" class="btn btn-sm btn-secondary cancel-edit-city">Cancel</button>
                </form>
            `;
            
            row.querySelector('.cancel-edit-city').addEventListener('click', () => {
                nameCell.innerHTML = name;
            });
        });
    });
});
</script>
</body>
</html>