<?php
session_start();
include_once '../db-config.php';

// --- Helper Functions ---
function nf($number) {
    return number_format((float)$number, 0, '.', ',');
}
function nf_decimal($number) {
    $formatted = number_format((float)$number, 2, '.', ',');
    return preg_replace('/\.00$/', '', $formatted);
}
function e($string) {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- Fetch all invoice data ---
$invoice_id = (int)($_GET['id'] ?? 0);
if ($invoice_id <= 0) die("Invalid Invoice ID.");

// Fetch main invoice details
$stmt = $conn->prepare("SELECT * FROM ticket_invoices WHERE id = ?");
$stmt->bind_param("i", $invoice_id);
$stmt->execute();
$invoice = $stmt->get_result()->fetch_assoc();
if (!$invoice) die("Invoice not found.");

// Fetch related data
$passengers = $conn->query("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = $invoice_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
$flights = $conn->query("SELECT * FROM ticket_invoice_flights WHERE ticket_invoice_id = $invoice_id ORDER BY id ASC")->fetch_all(MYSQLI_ASSOC);
$payments = $conn->query("SELECT * FROM payments WHERE invoice_id = $invoice_id AND invoice_type = 'ticket'")->fetch_all(MYSQLI_ASSOC);

// Fetch agent's logo and name if available
$agent_name = null;
$logo_to_display = '../images/logo.png'; // Default company logo
if (!empty($invoice['user_id'])) {
    $stmt_agent = $conn->prepare("SELECT name, logo_path FROM users WHERE id = ?");
    $stmt_agent->bind_param("i", $invoice['user_id']);
    $stmt_agent->execute();
    $agent_result = $stmt_agent->get_result();
    if ($agent = $agent_result->fetch_assoc()) {
        $agent_name = $agent['name'];
        if (!empty($agent['logo_path'])) {
            $logo_to_display = '../uploads/logos/' . e($agent['logo_path']);
        }
    }
    $stmt_agent->close();
}

// Calculate payment totals
$amount_paid = array_sum(array_column($payments, 'credit_amount'));
$amount_due = (float)$invoice['grand_total_pkr'] - $amount_paid;

// Determine Watermark Color based on Status
$watermark_color = 'rgba(231, 76, 60, 0.1)'; // Default to Red
if ($invoice['status'] === 'Approved') {
    $watermark_color = 'rgba(46, 204, 113, 0.1)'; // Green for Approved
}

$conn->close();

$invoice_display_number = e($invoice['invoice_number'] ?: $invoice['id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ticket Invoice #<?= $invoice_display_number ?></title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <style>
        :root {
            --theme-color: #0056b3;
            --border-color: #000000;
        }
        body {
            font-family: Arial, sans-serif; background-color: #e9e9e9; margin: 0;
            padding: 10px; font-size: 9pt; color: #000;
        }
        .actions-bar {
            max-width: 900px; margin: 0 auto 15px auto; display: flex;
            justify-content: flex-end; gap: 10px;
        }
        .btn {
            padding: 8px 15px; border: none; border-radius: 4px; color: white;
            font-size: 14px; cursor: pointer;
        }
        .btn-print { background-color: #2980b9; }
        .invoice-wrapper {
            max-width: 900px; margin: 0 auto; padding: 20px; border: 1px solid #ccc;
            position: relative; overflow: hidden; background-color: white;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        .watermark {
            position: absolute; top: 50%; left: 50%;
            transform: translate(-50%, -50%) rotate(-45deg);
            font-size: 100px; font-weight: bold; z-index: 1; pointer-events: none;
            text-transform: uppercase; white-space: nowrap;
        }
        .invoice-content { position: relative; z-index: 2; }
        table { width: 100%; border-collapse: collapse; }
        td, th { padding: 4px; vertical-align: middle; }

        /* Header */
        .header-table { margin-bottom: 20px; }
        .header-table td { border: none; padding: 0; vertical-align: top; }
        .logo-container { width: 50%; text-align: left; }
        .logo-container img { max-height: 80px; max-width: 200px; }
        .meta-container { width: 50%; }
        .meta-table td {
            background-color: var(--theme-color); border: 1px solid var(--border-color);
            padding: 5px 8px; font-weight: bold; color: white;
        }
        .meta-table td:first-child { width: 100px; }
        .meta-table td:last-child { background-color: #fff; text-align: center; color: #000; }

        /* Sections */
        .section-title {
            background-color: black; color: white; font-weight: bold;
            text-align: center; border: 1px solid var(--border-color);
            padding: 5px; margin-top: 15px; font-size: 10pt;
        }
        .detail-table { border: 1px solid var(--border-color); margin-bottom: 10px; }
        .detail-table th {
            background-color: var(--theme-color); border: 1px solid var(--border-color);
            font-weight: bold; padding: 5px; color: white; white-space: nowrap;
        }
        .detail-table td { border: 1px solid var(--border-color); padding: 4px; text-align: center; }
        .detail-table .text-left { text-align: left; padding-left: 5px; }
        .table-responsive-container { overflow-x: auto; margin-bottom: 10px; }
        .table-responsive-container > .detail-table { min-width: 800px; }

        /* Fare Grid */
        .fare-breakdown-grid {
            display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px; margin-top: 5px; padding: 10px;
            border: 1px solid var(--border-color); background-color: rgba(0, 0, 0, 0.02);
        }
        .fare-box { border: 1px solid #b0b0b0; background-color: #fdfdfd; border-radius: 4px; overflow: hidden; }
        .fare-box-header {
            background-color: var(--theme-color); padding: 8px; font-weight: bold;
            text-align: center; font-size: 11pt; color: white;
        }
        .fare-box-content { padding: 10px; display: flex; justify-content: space-around; gap: 10px; }
        .fare-detail {
            text-align: center; flex: 1; padding: 8px 5px; background-color: #f0f0f0;
            border: 1px solid #ddd; border-radius: 3px;
        }
        .fare-detail .label { display: block; font-size: 8pt; color: #555; margin-bottom: 4px; text-transform: uppercase; }
        .fare-detail .value { display: block; font-weight: bold; font-size: 12pt; color: #000; }

        /* Footer */
        .footer-container { padding-top: 20px; overflow: hidden; clear: both; }
        .footer-notes { float: left; width: 53%; font-size: 8pt; line-height: 1.5; }
        .footer-notes h4 { margin-top: 0; margin-bottom: 5px; font-size: 9pt; }
        .summary-totals { float: right; width: 45%; }
        .summary-totals table td { border: 1px solid var(--border-color); padding: 6px 10px; font-size: 10pt; }
        .summary-totals table td:first-child { font-weight: bold; width: 65%; }
        .summary-totals table td:last-child { text-align: right; font-weight: bold; }
        .summary-totals .grand-total td { background-color: var(--theme-color); color: white; font-weight: bold; }
        .summary-totals .payment-received-row td { color: #27ae60; }
        .summary-totals .remaining-amount-row td { background-color: #c0392b; color: white; font-weight: bold; font-size: 11pt; }
        .final-warning {
            clear: both; text-align: center; font-weight: bold; font-size: 9pt;
            margin-top: 20px; padding-top: 10px; border-top: 1px dashed var(--border-color);
        }

        @media screen and (max-width: 768px) {
            body { font-size: 8pt; padding: 5px; }
            .invoice-wrapper { padding: 10px; }
            .watermark { font-size: 60px; }
            .header-table > tbody > tr > td { display: block; width: 100%; text-align: center; }
            .logo-container { margin-bottom: 15px; }
            .meta-container { margin: 0 auto; max-width: 350px; }
            .footer-notes, .summary-totals { float: none; width: 100%; margin-bottom: 20px; }
        }

        @media print {
            * { -webkit-print-color-adjust: exact !important; print-color-adjust: exact !important; }
            body, html { margin: 0 !important; padding: 0 !important; background: #fff !important; }
            .actions-bar, .no-print { display: none !important; }
            body * { visibility: hidden; }
            .invoice-wrapper, .invoice-wrapper * { visibility: visible; }
            .invoice-wrapper {
                position: absolute; left: 0; top: 0; margin: 0 !important; padding: 10px !important;
                width: 100% !important; max-width: 100% !important;
                border: none !important; box-shadow: none !important; box-sizing: border-box;
            }
            .table-responsive-container { overflow-x: visible; }
        }
    </style>
</head>
<body>
    <div class="actions-bar no-print">
        <button onclick="window.print()" class="btn btn-print">Print Invoice</button>
    </div>

    <div class="invoice-wrapper" id="invoice-to-print">
        <?php if (!empty($invoice['status'])): ?>
            <div class="watermark" style="color: <?= $watermark_color; ?>;"><?= strtoupper(e($invoice['status'])) ?></div>
        <?php endif; ?>

        <div class="invoice-content">
            <header>
                <table class="header-table">
                    <tr>
                        <td class="logo-container">
                            <img src="<?= $logo_to_display ?>" alt="Logo">
                        </td>
                        <td class="meta-container">
                            <table class="meta-table">
                                <tr>
                                    <td>Invoice No:</td>
                                    <td><?= $invoice_display_number ?></td>
                                </tr>
                                <tr>
                                    <td>Guest Name:</td>
                                    <td><?= e($invoice['guest_name']) ?></td>
                                </tr>
                                <?php if ($agent_name): ?>
                                    <tr>
                                        <td>Agent:</td>
                                        <td><?= e($agent_name) ?></td>
                                    </tr>
                                <?php endif; ?>
                                <tr>
                                    <td>Dated:</td>
                                    <td><?= date('d, M, Y', strtotime($invoice['issue_date'])) ?></td>
                                </tr>
                            </table>
                        </td>
                    </tr>
                </table>
            </header>

            <main>
                <?php if (!empty($passengers)): ?>
                    <div class="section-title">Passenger Information</div>
                    <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th>Sr.</th>
                                    <th>Full Name</th>
                                    <th>Type</th>
                                    <th>Passport No.</th>
                                    <th>PNR</th>
                                    <th>Ticket No.</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php $i = 1; foreach ($passengers as $p): ?>
                                    <tr>
                                        <td><?= $i++ ?></td>
                                        <td class="text-left"><?= e($p['full_name']) ?></td>
                                        <td><?= e($p['passenger_type']) ?></td>
                                        <td><?= e($p['passport_no']) ?></td>
                                        <td><?= e($p['pnr']) ?></td>
                                        <td><?= e($p['ticket_number']) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <?php if (!empty($flights)): ?>
                    <div class="section-title">Flight Itinerary</div>
                    <div class="table-responsive-container">
                        <table class="detail-table">
                            <thead>
                                <tr>
                                    <th>Airline</th>
                                    <th>Flight No.</th>
                                    <th>Sector</th>
                                    <th>Departure</th>
                                    <th>Arrival</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($flights as $f): ?>
                                    <tr>
                                        <td class="text-left"><?= e($f['airline']) ?></td>
                                        <td><?= e($f['flight_no']) ?></td>
                                        <td><?= e($f['sector']) ?></td>
                                        <td><?= $f['departure_datetime'] ? date('d, M, y H:i', strtotime($f['departure_datetime'])) : 'N/A' ?></td>
                                        <td><?= $f['arrival_datetime'] ? date('d, M, y H:i', strtotime($f['arrival_datetime'])) : 'N/A' ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>

                <div class="section-title">Fare Breakdown</div>
                <div class="fare-breakdown-grid">
                    <?php if ($invoice['adult_qty'] > 0): ?>
                        <div class="fare-box">
                            <div class="fare-box-header">Adults</div>
                            <div class="fare-box-content">
                                <div class="fare-detail"><span class="label">Quantity</span><span class="value"><?= nf($invoice['adult_qty']) ?></span></div>
                                <div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value"><?= nf_decimal($invoice['adult_rate']) ?></span></div>
                            </div>
                        </div>
                    <?php endif; ?>
                    <?php if ($invoice['child_qty'] > 0): ?>
                        <div class="fare-box">
                            <div class="fare-box-header">Children</div>
                            <div class="fare-box-content">
                                <div class="fare-detail"><span class="label">Quantity</span><span class="value"><?= nf($invoice['child_qty']) ?></span></div>
                                <div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value"><?= nf_decimal($invoice['child_rate']) ?></span></div>
                            </div>
                        </div>
                    <?php endif; ?>
                    <?php if ($invoice['infant_qty'] > 0): ?>
                        <div class="fare-box">
                            <div class="fare-box-header">Infants</div>
                            <div class="fare-box-content">
                                <div class="fare-detail"><span class="label">Quantity</span><span class="value"><?= nf($invoice['infant_qty']) ?></span></div>
                                <div class="fare-detail"><span class="label">Rate (PKR)</span><span class="value"><?= nf_decimal($invoice['infant_rate']) ?></span></div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </main>

            <footer class="footer-container">
                <div class="footer-notes">
                    <?php if (!empty($invoice['notes'])): ?>
                        <h4>Terms & Conditions / Notes:</h4>
                        <p><?= nl2br(e($invoice['notes'])) ?></p>
                    <?php endif; ?>
                </div>
                <div class="summary-totals">
                    <table>
                        <tr>
                            <td>Total Fare (PKR)</td>
                            <td><?= nf_decimal($invoice['total_fare_pkr']) ?></td>
                        </tr>
                        <?php if ((float)$invoice['service_fee_pkr'] > 0): ?>
                        <tr>
                            <td>Service Fee (PKR)</td>
                            <td><?= nf_decimal($invoice['service_fee_pkr']) ?></td>
                        </tr>
                        <?php endif; ?>
                        <?php if ((float)$invoice['discount_pkr'] > 0): ?>
                            <tr class="discount-row">
                                <td>Discount (PKR)</td>
                                <td>- <?= nf_decimal($invoice['discount_pkr']) ?></td>
                            </tr>
                        <?php endif; ?>
                        <tr class="grand-total">
                            <td>Grand Total (PKR)</td>
                            <td><?= nf_decimal($invoice['grand_total_pkr']) ?></td>
                        </tr>
                        <?php if ($amount_paid > 0): ?>
                            <tr class="payment-received-row">
                                <td>Payment Received</td>
                                <td>- <?= nf_decimal($amount_paid) ?></td>
                            </tr>
                            <tr class="remaining-amount-row">
                                <td>Remaining Amount</td>
                                <td><?= nf_decimal($amount_due) ?></td>
                            </tr>
                        <?php endif; ?>
                    </table>
                </div>
            </footer>

            <div class="final-warning">
                All fares are subject to change without prior notice. Please verify passenger names match their passports. Standard airline penalties apply for cancellation and changes.
            </div>
        </div>
    </div>
</body>
</html>