<?php
/**
 * ticket-invoices-list.php (v4.0 - Bulk Delete Functionality)
 *
 * - NEW: Added checkboxes to each row and a "Select All" checkbox in the header.
 * - NEW: Implemented a "Delete Selected" button to remove multiple invoices at once.
 * - NEW: Added a JavaScript confirmation prompt for bulk deletion to prevent accidental data loss.
 * - MODIFIED: The backend now includes a robust function to handle deleting an array of invoice IDs and all their related records (passengers, flights, payments) in a single transaction.
 */
session_start();
include_once '../db-config.php';

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- HANDLE BULK DELETE REQUEST ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['delete_selected'])) {
    $invoice_ids_to_delete = $_POST['invoice_ids'] ?? [];
    $invoice_ids_to_delete = array_filter(array_map('intval', $invoice_ids_to_delete));

    if (!empty($invoice_ids_to_delete)) {
        $conn->begin_transaction();
        try {
            $placeholders = implode(',', array_fill(0, count($invoice_ids_to_delete), '?'));
            $types = str_repeat('i', count($invoice_ids_to_delete));

            // Delete from all child tables first
            $stmt_passengers = $conn->prepare("DELETE FROM ticket_invoice_passengers WHERE ticket_invoice_id IN ($placeholders)");
            $stmt_passengers->bind_param($types, ...$invoice_ids_to_delete);
            $stmt_passengers->execute();

            $stmt_flights = $conn->prepare("DELETE FROM ticket_invoice_flights WHERE ticket_invoice_id IN ($placeholders)");
            $stmt_flights->bind_param($types, ...$invoice_ids_to_delete);
            $stmt_flights->execute();

            $stmt_payments = $conn->prepare("DELETE FROM payments WHERE invoice_id IN ($placeholders) AND invoice_type = 'ticket'");
            $stmt_payments->bind_param($types, ...$invoice_ids_to_delete);
            $stmt_payments->execute();

            // Finally, delete from the main ticket_invoices table
            $stmt_invoice = $conn->prepare("DELETE FROM ticket_invoices WHERE id IN ($placeholders)");
            $stmt_invoice->bind_param($types, ...$invoice_ids_to_delete);
            $stmt_invoice->execute();

            $conn->commit();
            $_SESSION['success_message'] = count($invoice_ids_to_delete) . " ticket invoice(s) and all related records deleted.";
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "Error deleting ticket invoices: " . $e->getMessage();
        }
    } else {
        $_SESSION['error_message'] = "No invoices were selected for deletion.";
    }
    header("Location: ticket-invoices-list.php");
    exit;
}


// --- HANDLE SINGLE DELETE REQUEST ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['delete_ticket_invoice'])) {
    $invoice_id_to_delete = (int)($_POST['invoice_id'] ?? 0);
    if ($invoice_id_to_delete > 0) {
        $conn->begin_transaction();
        try {
            $stmt_passengers = $conn->prepare("DELETE FROM ticket_invoice_passengers WHERE ticket_invoice_id = ?");
            $stmt_passengers->bind_param("i", $invoice_id_to_delete);
            $stmt_passengers->execute();
            $stmt_flights = $conn->prepare("DELETE FROM ticket_invoice_flights WHERE ticket_invoice_id = ?");
            $stmt_flights->bind_param("i", $invoice_id_to_delete);
            $stmt_flights->execute();
            $stmt_payments = $conn->prepare("DELETE FROM payments WHERE invoice_id = ? AND invoice_type = 'ticket'");
            $stmt_payments->bind_param("i", $invoice_id_to_delete);
            $stmt_payments->execute();
            $stmt_invoice = $conn->prepare("DELETE FROM ticket_invoices WHERE id = ?");
            $stmt_invoice->bind_param("i", $invoice_id_to_delete);
            $stmt_invoice->execute();
            $conn->commit();
            $_SESSION['success_message'] = "Ticket Invoice #" . $invoice_id_to_delete . " and all related records deleted.";
        } catch (Exception $e) {
            $conn->rollback();
            $_SESSION['error_message'] = "Error deleting ticket invoice: " . $e->getMessage();
        }
    } else {
        $_SESSION['error_message'] = "Invalid ticket invoice ID for deletion.";
    }
    header("Location: ticket-invoices-list.php");
    exit;
}

// --- 1. SETUP FILTERS AND PAGINATION ---
$items_per_page = 15;
$current_page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($current_page - 1) * $items_per_page;

// Get filter values from GET request
$filter_invoice_no = trim($_GET['invoice_no'] ?? '');
$filter_guest_name = trim($_GET['guest_name'] ?? '');
$filter_user_id = (int)($_GET['user_id'] ?? 0);
$filter_vendor_id = (int)($_GET['vendor_id'] ?? 0);
$filter_status = $_GET['status'] ?? 'all';
if (!in_array($filter_status, ['all', 'tentative', 'approved', 'cancelled'])) {
    $filter_status = 'all';
}

// --- 2. FETCH DATA for filter dropdowns ---
$users_list = $conn->query("SELECT id, name FROM users WHERE user_type IN ('customer', 'agent') ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- 3. DYNAMICALLY BUILD SQL BASED ON FILTERS ---
$base_sql_select = "SELECT 
                        ti.id, ti.invoice_number, ti.guest_name, ti.issue_date, ti.grand_total_pkr, ti.status, 
                        v.name as vendor_name,
                        u.name as linked_user_name,
                        u.user_type,
                        u.company_name";
$base_sql_from = "FROM ticket_invoices ti 
                  LEFT JOIN users u ON ti.user_id = u.id 
                  LEFT JOIN vendors v ON ti.vendor_id = v.id";

$where_clauses = [];
$params = [];
$types = "";

if (!empty($filter_invoice_no)) {
    $where_clauses[] = "(ti.id = ? OR ti.invoice_number LIKE ?)";
    $params[] = $filter_invoice_no;
    $params[] = "%" . $filter_invoice_no . "%";
    $types .= "is";
}
if (!empty($filter_guest_name)) {
    $where_clauses[] = "ti.guest_name LIKE ?";
    $params[] = "%" . $filter_guest_name . "%";
    $types .= "s";
}
if ($filter_user_id > 0) {
    $where_clauses[] = "ti.user_id = ?";
    $params[] = $filter_user_id;
    $types .= "i";
}
if ($filter_vendor_id > 0) {
    $where_clauses[] = "ti.vendor_id = ?";
    $params[] = $filter_vendor_id;
    $types .= "i";
}
if ($filter_status !== 'all') {
    $where_clauses[] = "ti.status = ?";
    $params[] = ucfirst($filter_status);
    $types .= "s";
}

$where_sql = count($where_clauses) > 0 ? "WHERE " . implode(" AND ", $where_clauses) : "";

// --- 4. GET TOTAL COUNT FOR THE CURRENT FILTERS ---
$count_sql = "SELECT COUNT(ti.id) as total $base_sql_from $where_sql";
$stmt_count = $conn->prepare($count_sql);
if (!empty($params)) {
    $stmt_count->bind_param($types, ...$params);
}
$stmt_count->execute();
$total_items = $stmt_count->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_items / $items_per_page);
$stmt_count->close();

// --- 5. FETCH THE PAGINATED DATA for display ---
$data_sql = "$base_sql_select $base_sql_from $where_sql 
             ORDER BY ti.issue_date DESC, ti.id DESC 
             LIMIT ?, ?";

$final_params = $params;
$final_params[] = $offset;
$final_params[] = $items_per_page;
$final_types = $types . "ii";

$stmt_data = $conn->prepare($data_sql);
$stmt_data->bind_param($final_types, ...$final_params);
$stmt_data->execute();
$invoices_result = $stmt_data->get_result();

$success_message = $_SESSION['success_message'] ?? null;
$error_message = $_SESSION['error_message'] ?? null;
unset($_SESSION['success_message'], $_SESSION['error_message']);
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="icon" type="image/png" href="../images/logo-icon.png">
    <title>Manage Ticket Invoices</title>
    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <button class="menu-toggle" id="menu-toggle"><i class="fas fa-bars"></i></button>
                <div class="user-info"><span>Manage Ticket Invoices</span></div>
            </header>
            <main class="content-body">
                <div class="page-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h1 class="page-title">Ticket Invoices</h1>
                    <a href="ticket-invoice-create.php" class="btn btn-primary"><i class="fas fa-plus"></i> Create New Invoice</a>
                </div>

                <?php if ($success_message): ?><div class="notice success"><?= e($success_message); ?></div><?php endif; ?>
                <?php if ($error_message): ?><div class="notice error"><?= e($error_message); ?></div><?php endif; ?>

                <div class="content-card">
                    <div class="card-body">
                        <form action="ticket-invoices-list.php" method="GET" class="styled-form">
                            <div class="form-grid">
                                <div class="form-group"><label>Invoice #/ID</label><input type="text" name="invoice_no" class="form-control" value="<?= e($filter_invoice_no) ?>" placeholder="e.g. TKT-2024-0123"></div>
                                <div class="form-group"><label>Guest Name</label><input type="text" name="guest_name" class="form-control" value="<?= e($filter_guest_name) ?>" placeholder="Search name..."></div>
                                <div class="form-group"><label>User/Agent</label>
                                    <select name="user_id" class="form-control">
                                        <option value="0">All Users</option>
                                        <?php foreach ($users_list as $user): ?>
                                            <option value="<?= e($user['id']) ?>" <?= ($filter_user_id == $user['id']) ? 'selected' : '' ?>><?= e($user['name']) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group"><label>Vendor</label>
                                    <select name="vendor_id" class="form-control">
                                        <option value="0">All Vendors</option>
                                        <?php foreach ($vendors_list as $vendor): ?>
                                            <option value="<?= e($vendor['id']) ?>" <?= ($filter_vendor_id == $vendor['id']) ? 'selected' : '' ?>><?= e($vendor['name']) ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="form-group"><label>Status</label>
                                    <select name="status" class="form-control">
                                        <option value="all" <?= ($filter_status == 'all') ? 'selected' : '' ?>>All Statuses</option>
                                        <option value="tentative" <?= ($filter_status == 'tentative') ? 'selected' : '' ?>>Tentative</option>
                                        <option value="approved" <?= ($filter_status == 'approved') ? 'selected' : '' ?>>Approved</option>
                                        <option value="cancelled" <?= ($filter_status == 'cancelled') ? 'selected' : '' ?>>Cancelled</option>
                                    </select>
                                </div>
                            </div>
                            <div class="form-actions">
                                <a href="ticket-invoices-list.php" class="btn btn-secondary"><i class="fas fa-times"></i> Clear</a>
                                <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filter</button>
                            </div>
                        </form>
                    </div>
                </div>

                <div class="content-card">
                    <div class="card-body">
                        <form id="bulk-delete-form" action="ticket-invoices-list.php" method="POST">
                            <div class="table-responsive">
                                <table>
                                    <thead>
                                        <tr>
                                            <th style="width: 20px;"><input type="checkbox" id="select-all-checkbox" title="Select All"></th>
                                            <th>Invoice #</th>
                                            <th>Issue Date</th>
                                            <th>Guest Name</th>
                                            <th>Agent/Customer</th>
                                            <th>Grand Total</th>
                                            <th>Status</th>
                                            <th>Vendor</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php if ($invoices_result->num_rows > 0): ?>
                                            <?php while ($invoice = $invoices_result->fetch_assoc()): ?>
                                                <tr>
                                                    <td><input type="checkbox" name="invoice_ids[]" value="<?= $invoice['id'] ?>" class="invoice-checkbox"></td>
                                                    <td><strong><?= e($invoice['invoice_number'] ?: '#' . $invoice['id']) ?></strong></td>
                                                    <td><?= date('d M Y', strtotime($invoice['issue_date'])) ?></td>
                                                    <td><?= e($invoice['guest_name']) ?></td>
                                                    <td>
                                                        <?php
                                                        if (!empty($invoice['linked_user_name'])) {
                                                            echo e($invoice['linked_user_name']);
                                                            if ($invoice['user_type'] === 'agent' && !empty($invoice['company_name'])) {
                                                                echo '<br><small style="font-size: 0.85em; color: var(--text-muted);">' . e($invoice['company_name']) . '</small>';
                                                            }
                                                        } else {
                                                            echo 'Direct Customer';
                                                        }
                                                        ?>
                                                    </td>
                                                    <td><?= number_format($invoice['grand_total_pkr'], 2) ?></td>
                                                    <td><span class="status-badge status-<?= strtolower(e($invoice['status'])) ?>"><?= e($invoice['status']) ?></span></td>
                                                    <td><?= e($invoice['vendor_name'] ?? 'N/A') ?></td>
                                                    <td class="actions-cell">
                                                        <a href="ticket-invoice-view.php?id=<?= $invoice['id'] ?>" class="btn btn-sm btn-info" title="View/Print" target="_blank"><i class="fas fa-eye"></i></a>
                                                        <a href="ticket-invoice-edit.php?id=<?= $invoice['id'] ?>" class="btn btn-sm btn-primary" title="Edit / Add Payment" target="_blank"><i class="fas fa-edit"></i></a>
                                                        <form action="ticket-invoices-list.php" method="POST" onsubmit="return confirm('Delete this ticket invoice and all related data? This is permanent.');" style="display:inline-block;">
                                                            <input type="hidden" name="invoice_id" value="<?= $invoice['id'] ?>">
                                                            <button type="submit" name="delete_ticket_invoice" class="btn btn-sm btn-danger" title="Delete"><i class="fas fa-trash"></i></button>
                                                        </form>
                                                    </td>
                                                </tr>
                                            <?php endwhile; ?>
                                        <?php else: ?>
                                            <tr>
                                                <td colspan="9" class="empty-state">No invoices found for the selected criteria.</td>
                                            </tr>
                                        <?php endif; ?>
                                    </tbody>
                                </table>
                            </div>
                             <?php if ($invoices_result->num_rows > 0): ?>
                                <div class="bulk-actions" style="margin-top: 20px;">
                                    <button type="submit" name="delete_selected" class="btn btn-danger"><i class="fas fa-trash-alt"></i> Delete Selected</button>
                                </div>
                            <?php endif; ?>
                        </form>

                        <?php if ($total_pages > 1): ?>
                            <div class="pagination-controls" style="margin-top: 20px; text-align: right;">
                                <?php $query_params = $_GET; if ($current_page > 1) { $query_params['page'] = $current_page - 1; echo '<a href="ticket-invoices-list.php?' . http_build_query($query_params) . '" class="btn btn-secondary btn-sm">« Previous</a>'; } ?>
                                <span style="padding: 0 10px;">Page <?= $current_page ?> of <?= $total_pages ?></span>
                                <?php if ($current_page < $total_pages) { $query_params['page'] = $current_page + 1; echo '<a href="ticket-invoices-list.php?' . http_build_query($query_params) . '" class="btn btn-secondary btn-sm">Next »</a>'; } ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const selectAllCheckbox = document.getElementById('select-all-checkbox');
        const invoiceCheckboxes = document.querySelectorAll('.invoice-checkbox');
        const bulkDeleteForm = document.getElementById('bulk-delete-form');

        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', function() {
                invoiceCheckboxes.forEach(checkbox => {
                    checkbox.checked = this.checked;
                });
            });
        }

        invoiceCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', function() {
                if (!this.checked && selectAllCheckbox) {
                    selectAllCheckbox.checked = false;
                }
            });
        });

        if (bulkDeleteForm) {
            bulkDeleteForm.addEventListener('submit', function(event) {
                if (!document.activeElement || document.activeElement.name !== 'delete_selected') {
                    return;
                }

                const selectedCheckboxes = document.querySelectorAll('.invoice-checkbox:checked');
                if (selectedCheckboxes.length === 0) {
                    alert('Please select at least one invoice to delete.');
                    event.preventDefault();
                    return;
                }
                
                const confirmed = confirm('Are you sure you want to permanently delete the selected ' + selectedCheckboxes.length + ' invoice(s) and all their related records?');
                if (!confirmed) {
                    event.preventDefault();
                }
            });
        }
    });
    </script>
</body>
</html>