<?php
session_start();
include_once '../db-config.php';

// --- SECURITY CHECK ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'admin') {
    header("location: ../login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- 1. FETCH VENDORS FOR THE FILTER DROPDOWN ---
$vendors_list = $conn->query("SELECT id, name FROM vendors ORDER BY name ASC")->fetch_all(MYSQLI_ASSOC);

// --- 2. GET SELECTED VENDOR & INITIALIZE VARS ---
$selected_vendor_id = (int)($_GET['vendor_id'] ?? 0);
$selected_vendor_details = null;

// Initialize data arrays for each section
$all_transactions = [];
$vendor_hotels = [];

// Initialize pagination variables
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$records_per_page = 20;
$pagination_links = '';

if ($selected_vendor_id > 0) {
    // Fetch the selected vendor's name for the page title
    $stmt_vendor = $conn->prepare("SELECT name FROM vendors WHERE id = ?");
    $stmt_vendor->bind_param("i", $selected_vendor_id);
    $stmt_vendor->execute();
    $selected_vendor_details = $stmt_vendor->get_result()->fetch_assoc();
    $stmt_vendor->close();

    // --- A) FETCH ALL RELEVANT TRANSACTIONS (NEW COMPREHENSIVE LOGIC) ---
    $count_sql_parts = [];
    $count_params = [];
    $count_types = '';

    // Count Vouchers
    $count_sql_parts[] = "(SELECT count(id) as total FROM vouchers WHERE vendor_id = ?)";
    $count_params[] = $selected_vendor_id;
    $count_types .= 'i';

    // Count Ticket Invoices
    $count_sql_parts[] = "(SELECT count(id) as total FROM ticket_invoices WHERE vendor_id = ?)";
    $count_params[] = $selected_vendor_id;
    $count_types .= 'i';

    // Count Package Invoices (This is now a complex count)
    $count_sql_parts[] = "(SELECT COUNT(DISTINCT i.id) as total FROM invoices i 
        LEFT JOIN invoice_hotels ih ON i.id = ih.invoice_id
        LEFT JOIN invoice_other_services ios ON i.id = ios.invoice_id
        WHERE i.vendor_id = ? 
           OR i.pilgrims_vendor_id = ? 
           OR i.transport_vendor_id = ? 
           OR i.tickets_vendor_id = ?
           OR ih.vendor_id = ?
           OR ios.vendor_id = ?
    )";
    array_push($count_params, $selected_vendor_id, $selected_vendor_id, $selected_vendor_id, $selected_vendor_id, $selected_vendor_id, $selected_vendor_id);
    $count_types .= 'iiiiii';

    // Execute Total Count
    $count_sql = "SELECT SUM(total) as total_records FROM (" . implode(" UNION ALL ", $count_sql_parts) . ") as counts";
    $stmt_count = $conn->prepare($count_sql);
    $stmt_count->bind_param($count_types, ...$count_params);
    $stmt_count->execute();
    $total_records = $stmt_count->get_result()->fetch_assoc()['total_records'] ?? 0;
    $total_pages = ceil($total_records / $records_per_page);
    $stmt_count->close();

    // --- B) FETCH PAGINATED TRANSACTIONS ---
    $offset = ($page - 1) * $records_per_page;
    $sql_parts = [];
    $sql_params = [];
    $sql_types = '';
    $collation = "COLLATE utf8mb4_unicode_ci";

    // Vouchers
    // *** FIX IS HERE: The `vouchers` table does not have a `grand_total_pkr` column. Selecting NULL as amount. ***
    $sql_parts[] = "(SELECT id, 'voucher' as record_type, voucher_date as date, booking_ref_no as number, family_head_name as customer, NULL as amount, 'Directly Tagged' as reason FROM vouchers WHERE vendor_id = ?)";
    $sql_params[] = $selected_vendor_id;
    $sql_types .= 'i';

    // Ticket Invoices
    $sql_parts[] = "(SELECT id, 'ticket_invoice' as record_type, issue_date as date, invoice_number as number, guest_name as customer, grand_total_pkr as amount, 'Directly Tagged' as reason FROM ticket_invoices WHERE vendor_id = ?)";
    $sql_params[] = $selected_vendor_id;
    $sql_types .= 'i';

    // Package Invoices (Complex Union)
    $package_invoice_unions = [
    "SELECT id, 'package_invoice' as record_type, issue_date as date, invoice_number as number, guest_name as customer, grand_total_pkr as amount, 'Main Vendor' as reason FROM invoices WHERE vendor_id = ?",
    "SELECT id, 'package_invoice' as record_type, issue_date as date, invoice_number as number, guest_name as customer, grand_total_pkr as amount, 'Pilgrim Vendor' as reason FROM invoices WHERE pilgrims_vendor_id = ?",
    "SELECT id, 'package_invoice' as record_type, issue_date as date, invoice_number as number, guest_name as customer, grand_total_pkr as amount, 'Transport Vendor' as reason FROM invoices WHERE transport_vendor_id = ?",
    "SELECT id, 'package_invoice' as record_type, issue_date as date, invoice_number as number, guest_name as customer, grand_total_pkr as amount, 'Ticket Vendor' as reason FROM invoices WHERE tickets_vendor_id = ?",
    "SELECT DISTINCT i.id, 'package_invoice' as record_type, i.issue_date as date, i.invoice_number as number, i.guest_name as customer, i.grand_total_pkr as amount, 'Hotel Vendor' as reason FROM invoices i JOIN invoice_hotels ih ON i.id = ih.invoice_id WHERE ih.vendor_id = ?",
    "SELECT DISTINCT i.id, 'package_invoice' as record_type, i.issue_date as date, i.invoice_number as number, i.guest_name as customer, i.grand_total_pkr as amount, 'Service Vendor' as reason FROM invoices i JOIN invoice_other_services ios ON i.id = ios.invoice_id WHERE ios.vendor_id = ?"
];
    foreach ($package_invoice_unions as $union_sql) {
        $sql_parts[] = "($union_sql)";
        $sql_params[] = $selected_vendor_id;
        $sql_types .= 'i';
    }

    $final_sql = "SELECT DISTINCT id, record_type, date, number, customer, amount, reason FROM (" . implode(" UNION ALL ", $sql_parts) . ") as all_records ORDER BY date DESC, id DESC LIMIT ?, ?";

    // Add pagination params at the end
    array_push($sql_params, $offset, $records_per_page);
    $sql_types .= 'ii';

    $stmt = $conn->prepare($final_sql);
    $stmt->bind_param($sql_types, ...$sql_params);
    $stmt->execute();
    $all_transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();

    // --- C) FETCH ALL HOTELS ASSOCIATED WITH THIS VENDOR (from rate sheets) ---
    $stmt_hotels = $conn->prepare("SELECT hotel_name, city, stars FROM rate_sheets WHERE vendor_id = ? ORDER BY city, hotel_name ASC");
    $stmt_hotels->bind_param("i", $selected_vendor_id);
    $stmt_hotels->execute();
    $vendor_hotels = $stmt_hotels->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt_hotels->close();
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Vendor Log</title>
    <link rel="icon" type="image/png" href="../images/logo-icon.png">

    <link rel="stylesheet" href="admin-style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        .filter-container {
            background-color: #fff;
            padding: 1.5rem;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .filter-form {
            display: flex;
            align-items: flex-end;
            gap: 1rem;
        }

        .pagination {
            margin-top: 1.5rem;
            text-align: center;
        }

        .pagination a,
        .pagination span {
            display: inline-block;
            padding: 8px 12px;
            margin: 0 2px;
            border: 1px solid #ddd;
            border-radius: 4px;
            text-decoration: none;
            color: #333;
        }

        .pagination a:hover {
            background-color: #f0f0f0;
        }

        .pagination .current {
            background-color: #007bff;
            color: white;
            border-color: #007bff;
        }

        .pagination .disabled {
            color: #ccc;
            cursor: default;
        }
    </style>
</head>

<body>
    <div class="dashboard-wrapper">
        <?php include 'sidebar.php'; ?>
        <div class="main-content">
            <header class="main-header">
                <div class="user-info"><span>Vendor Activity Log</span></div>
            </header>
            <main class="content-body">

                <div class="filter-container">
                    <form method="GET" action="vendor-log.php" class="filter-form">
                        <div class="form-group">
                            <label for="vendor_id">Select a Vendor to View Log</label>
                            <select name="vendor_id" id="vendor_id" class="form-control" required onchange="this.form.submit()">
                                <option value="">-- Please Select a Vendor --</option>
                                <?php foreach ($vendors_list as $vendor): ?>
                                    <option value="<?= e($vendor['id']) ?>" <?= ($selected_vendor_id == $vendor['id']) ? 'selected' : '' ?>>
                                        <?= e($vendor['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <a href="vendor-log.php" class="btn btn-secondary"><i class="fas fa-times"></i> Clear</a>
                    </form>
                </div>

                <?php if ($selected_vendor_id > 0 && $selected_vendor_details): ?>

                    <div class="content-card">
                        <div class="card-header">
                            <h2>All Transactions for: <?= e($selected_vendor_details['name']) ?></h2>
                            <small>Showing <?= count($all_transactions) ?> of <?= $total_records ?> total records</small>
                        </div>
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Type</th>
                                        <th>Reference #</th>
                                        <th>Customer/Guest</th>
                                        <th>Reason for Link</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($all_transactions)): ?>
                                        <tr>
                                            <td colspan="6" class="empty-state">No transactions found for this vendor.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($all_transactions as $trans):
                                            $type_label = '';
                                            $view_url = '#';
                                            $edit_url = '#';
                                            switch ($trans['record_type']) {
                                                case 'voucher':
                                                    $type_label = '<span class="status-badge status-blue">Voucher</span>';
                                                    $view_url = 'view-voucher.php?id=' . e($trans['id']);
                                                    $edit_url = 'edit-voucher.php?id=' . e($trans['id']);
                                                    break;
                                                case 'package_invoice':
                                                    $type_label = '<span class="status-badge status-green">Package Invoice</span>';
                                                    $view_url = 'view-invoice.php?id=' . e($trans['id']);
                                                    $edit_url = 'edit-invoice.php?id=' . e($trans['id']);
                                                    break;
                                                case 'ticket_invoice':
                                                    $type_label = '<span class="status-badge status-orange">Ticket Invoice</span>';
                                                    $view_url = 'ticket-invoice-view.php?id=' . e($trans['id']);
                                                    $edit_url = 'ticket-invoice-edit.php?id=' . e($trans['id']);
                                                    break;
                                            }
                                        ?>
                                            <tr>
                                                <td><?= date('d M, Y', strtotime(e($trans['date']))) ?></td>
                                                <td><?= $type_label ?></td>
                                                <td><?= e($trans['number']) ?></td>
                                                <td><?= e($trans['customer']) ?></td>
                                                <td><strong><?= e($trans['reason']) ?></strong></td>
                                                <td class="actions-cell">
                                                    <a href="<?= $view_url ?>" class="btn btn-sm btn-primary" title="View"><i class="fas fa-eye"></i></a>
                                                    <a href="<?= $edit_url ?>" class="btn btn-sm btn-secondary" title="Edit"><i class="fas fa-edit"></i></a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="pagination">
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <a href="?vendor_id=<?= $selected_vendor_id ?>&page=<?= $i ?>" class="<?= ($i == $page) ? 'current' : '' ?>"><?= $i ?></a>
                            <?php endfor; ?>
                        </div>
                    </div>

                    <div class="content-card">
                        <div class="card-header">
                            <h2>Associated Hotels (from Rate Sheets)</h2>
                        </div>
                        <div class="table-responsive">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>Hotel Name</th>
                                        <th>City</th>
                                        <th>Star Rating</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($vendor_hotels)): ?>
                                        <tr>
                                            <td colspan="3" class="empty-state">This vendor is not associated with any hotels in the rate sheets.</td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($vendor_hotels as $hotel): ?>
                                            <tr>
                                                <td><?= e($hotel['hotel_name']) ?></td>
                                                <td><?= e($hotel['city']) ?></td>
                                                <td><?= str_repeat('⭐', e($hotel['stars'])) ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>

                <?php elseif ($selected_vendor_id > 0): ?>
                    <div class="notice error">The selected vendor could not be found.</div>
                <?php else: ?>
                    <div class="notice info">Please select a vendor from the dropdown above to view their activity log.</div>
                <?php endif; ?>

            </main>
        </div>
    </div>
    <script>
        document.getElementById('vendor_id').addEventListener('change', function() {
            if (this.value) {
                this.form.submit();
            }
        });
    </script>


    <script>
        // This script disables the right-click context menu on the entire page.
        document.addEventListener('contextmenu', function(e) {
            e.preventDefault();
        });
    </script>
</body>

</html>