<?php
session_start(); // Ensure session is started at the very top
include 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php");
    exit;
}

// Get the logged-in agent's ID and name from the session
$logged_in_agent_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// --- FETCH FLIGHT BOOKINGS ---
$sql_flights = "SELECT b.*, u.name as customer_name 
                FROM bookings b 
                JOIN users u ON b.user_id = u.id 
                WHERE b.user_id = ? AND b.booking_type = 'flight'
                ORDER BY b.created_at DESC";
$stmt_flights = $conn->prepare($sql_flights);
$stmt_flights->bind_param("i", $logged_in_agent_id);
$stmt_flights->execute();
$flights_result = $stmt_flights->get_result();
$stmt_flights->close();

// --- FETCH GROUP BOOKINGS ---
$sql_groups = "SELECT b.*, u.name as customer_name 
               FROM bookings b 
               JOIN users u ON b.user_id = u.id 
               WHERE b.user_id = ? AND b.booking_type = 'group'
               ORDER BY b.created_at DESC";
$stmt_groups = $conn->prepare($sql_groups);
$stmt_groups->bind_param("i", $logged_in_agent_id);
$stmt_groups->execute();
$groups_result = $stmt_groups->get_result();
$stmt_groups->close();

?>
<!DOCTYPE html>
<html>

<head>
    <title>Agent Dashboard</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .section-header {
            margin-top: 2rem;
            margin-bottom: 1rem;
            font-size: 1.4rem;
            color: #333;
            border-bottom: 2px solid #f0f0f0;
            padding-bottom: 0.5rem;
        }

        .section-header:first-of-type {
            margin-top: 0;
        }

        .actions-cell a {
            text-decoration: none;
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">

            <?php include 'agent-sidebar.php'; ?>

            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-suitcase-rolling"></i> My Bookings</h2>
                    <p class="content-description">Review and manage all bookings you have placed. Click the button to view the ticket voucher.</p>

                    <!-- FLIGHT BOOKINGS SECTION -->
                    <h3 class="section-header"><i class="fa-solid fa-plane"></i> Flight Bookings</h3>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Booking Ref</th>
                                    <th>Primary Passenger</th>
                                    <th>Route</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Price</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($flights_result && $flights_result->num_rows > 0): ?>
                                    <?php while ($booking = $flights_result->fetch_assoc()): ?>
                                        <?php
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $passenger_details = json_decode($booking['passenger_details'], true);

                                        $primary_passenger_name = 'N/A';
                                        if (isset($passenger_details['adults'])) {
                                            $first_adult = reset($passenger_details['adults']);
                                            if ($first_adult) {
                                                $primary_passenger_name = htmlspecialchars(($first_adult['title'] ?? '') . '. ' . ($first_adult['firstName'] ?? '') . ' ' . ($first_adult['lastName'] ?? ''));
                                            }
                                        }

                                        $route = 'N/A';
                                        if (isset($flight_details['outbound']['origin'])) {
                                            $route = htmlspecialchars($flight_details['outbound']['origin']) . ' - ' . htmlspecialchars($flight_details['outbound']['destination']);
                                        }

                                        $date = 'N/A';
                                        if (isset($flight_details['outbound']['departureDate'])) {
                                            $date = date('M j, Y', strtotime($flight_details['outbound']['departureDate']));
                                        }

                                        $status_class = 'status-' . strtolower($booking['status']);
                                        ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($booking['booking_ref']); ?></strong></td>
                                            <td><?php echo $primary_passenger_name; ?></td>
                                            <td><?php echo $route; ?></td>
                                            <td><?php echo $date; ?></td>
                                            <td><span class="status-badge <?php echo $status_class; ?>"><?php echo ucfirst($booking['status']); ?></span></td>
                                            <td><?php echo htmlspecialchars($booking['price_currency']) . ' ' . number_format($booking['total_price'], 0); ?></td>
                                            <td class="actions-cell">
                                                <a href="ticket-voucher.php?booking_id=<?= $booking['id']; ?>"
                                                    class="btn-sm btn-action"
                                                    title="View Voucher">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            </td>

                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="no-bookings-found">You have not made any flight bookings yet.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- GROUP BOOKINGS SECTION -->
                    <h3 class="section-header"><i class="fa-solid fa-users"></i> Group Bookings</h3>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Booking Ref</th>
                                    <th>Primary Passenger</th>
                                    <th>Route</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Price</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($groups_result && $groups_result->num_rows > 0): ?>
                                    <?php while ($booking = $groups_result->fetch_assoc()): ?>
                                        <?php
                                        $flight_details = json_decode($booking['flight_details'], true);
                                        $passenger_details = json_decode($booking['passenger_details'], true);

                                        $primary_passenger_name = 'N/A';
                                        if (isset($passenger_details['adults'])) {
                                            $first_adult = reset($passenger_details['adults']);
                                            if ($first_adult) {
                                                $primary_passenger_name = htmlspecialchars(($first_adult['title'] ?? '') . '. ' . ($first_adult['given_name'] ?? '') . ' ' . ($first_adult['surname'] ?? ''));
                                            }
                                        }

                                        $route = htmlspecialchars($flight_details['route'] ?? 'Group Fare');

                                        $date = 'N/A';
                                        if (!empty($flight_details['departure_date'])) {
                                            $date = date('M j, Y', strtotime($flight_details['departure_date']));
                                        }

                                        $status_class = 'status-' . strtolower($booking['status']);
                                        ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($booking['booking_ref']); ?></strong></td>
                                            <td><?php echo $primary_passenger_name; ?></td>
                                            <td><?php echo $route; ?></td>
                                            <td><?php echo $date; ?></td>
                                            <td><span class="status-badge <?php echo $status_class; ?>"><?php echo ucfirst($booking['status']); ?></span></td>
                                            <td><?php echo htmlspecialchars($booking['price_currency']) . ' ' . number_format($booking['total_price'], 0); ?></td>
                                            <td class="actions-cell">
                                                <a href="ticket-voucher.php?booking_id=<?= $booking['id']; ?>"
                                                    class="btn-sm btn-action"
                                                    title="View Voucher">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="7" class="no-bookings-found">You have not made any group bookings yet.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>

</body>

</html>