<?php
session_start();
include_once 'db-config.php';

// --- SECURITY CHECKS ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: login.php");
    exit;
}

// Helper function
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- 1. SETUP FILTERS & AGENT ID ---
$logged_in_agent_id = (int)$_SESSION['user_id'];
$filter_start_date = $_GET['start_date'] ?? '';
$filter_end_date = $_GET['end_date'] ?? '';

// --- 2. FETCH AGENT DETAILS ---
$agent_details = null;
$stmt_user = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt_user->bind_param("i", $logged_in_agent_id);
$stmt_user->execute();
$result_user = $stmt_user->get_result();
if ($result_user) {
    $agent_details = $result_user->fetch_assoc();
}
$stmt_user->close();
if (!$agent_details) {
    die("Could not find agent details.");
}

// --- 3. CALCULATE THE OPENING BALANCE (SYNCED WITH AGENT-LEDGER.PHP) ---
$opening_balance = 0;
if (!empty($filter_start_date)) {
    $sql_opening_base = "
        SELECT SUM(balance) as opening_balance FROM (
            SELECT SUM(i.grand_total_pkr) as balance FROM invoices i WHERE i.issue_date < ? AND i.user_id = ?
            UNION ALL
            SELECT SUM(ti.grand_total_pkr) as balance FROM ticket_invoices ti WHERE ti.issue_date < ? AND ti.user_id = ?
            UNION ALL
            SELECT -SUM(p.credit_amount) as balance FROM payments p 
            WHERE p.payment_date < ? AND p.vendor_id IS NULL AND ( -- FIX: Ensure no vendor payments are ever included
                p.user_id = ? OR
                EXISTS (SELECT 1 FROM invoices i WHERE i.id = p.invoice_id AND p.invoice_type = 'package' AND i.user_id = ?) OR
                EXISTS (SELECT 1 FROM ticket_invoices ti WHERE ti.id = p.invoice_id AND p.invoice_type = 'ticket' AND ti.user_id = ?)
            )
        ) AS opening_parts
    ";
    $stmt_opening = $conn->prepare($sql_opening_base);
    if ($stmt_opening) {
        $stmt_opening->bind_param("sisisiii", $filter_start_date, $logged_in_agent_id, $filter_start_date, $logged_in_agent_id, $filter_start_date, $logged_in_agent_id, $logged_in_agent_id, $logged_in_agent_id);
        $stmt_opening->execute();
        $result_opening = $stmt_opening->get_result()->fetch_assoc();
        $opening_balance = (float)($result_opening['opening_balance'] ?? 0);
        $stmt_opening->close();
    }
}

// --- 4. BUILD & FETCH MAIN TRANSACTION QUERY (SYNCED WITH AGENT-LEDGER.PHP) ---
$params = [];
$types = '';
$where_clauses = [
    'pkg' => ['i.user_id = ?'],
    'tkt' => ['ti.user_id = ?'],
    'pay' => [
        'p.vendor_id IS NULL', // FIX: Ensure no vendor payments are ever included
        '(p.user_id = ? OR EXISTS (SELECT 1 FROM invoices i2 WHERE i2.id = p.invoice_id AND p.invoice_type = \'package\' AND i2.user_id = ?) OR EXISTS (SELECT 1 FROM ticket_invoices ti2 WHERE ti2.id = p.invoice_id AND p.invoice_type = \'ticket\' AND ti2.user_id = ?))'
    ]
];

$date_params_pkg = [];
$date_types_pkg = '';
$date_params_tkt = [];
$date_types_tkt = '';
$date_params_pay = [];
$date_types_pay = '';
if (!empty($filter_start_date) && !empty($filter_end_date)) {
    $where_clauses['pkg'][] = "i.issue_date BETWEEN ? AND ?";
    $date_params_pkg = [$filter_start_date, $filter_end_date];
    $date_types_pkg = 'ss';
    $where_clauses['tkt'][] = "ti.issue_date BETWEEN ? AND ?";
    $date_params_tkt = [$filter_start_date, $filter_end_date];
    $date_types_tkt = 'ss';
    $where_clauses['pay'][] = "p.payment_date BETWEEN ? AND ?";
    $date_params_pay = [$filter_start_date, $filter_end_date];
    $date_types_pay = 'ss';
} else if (!empty($filter_start_date)) {
    $where_clauses['pkg'][] = "i.issue_date >= ?";
    $date_params_pkg = [$filter_start_date];
    $date_types_pkg = 's';
    $where_clauses['tkt'][] = "ti.issue_date >= ?";
    $date_params_tkt = [$filter_start_date];
    $date_types_tkt = 's';
    $where_clauses['pay'][] = "p.payment_date >= ?";
    $date_params_pay = [$filter_start_date];
    $date_types_pay = 's';
} else if (!empty($filter_end_date)) {
    $where_clauses['pkg'][] = "i.issue_date <= ?";
    $date_params_pkg = [$filter_end_date];
    $date_types_pkg = 's';
    $where_clauses['tkt'][] = "ti.issue_date <= ?";
    $date_params_tkt = [$filter_end_date];
    $date_types_tkt = 's';
    $where_clauses['pay'][] = "p.payment_date <= ?";
    $date_params_pay = [$filter_end_date];
    $date_types_pay = 's';
}

$params = array_merge([$logged_in_agent_id], $date_params_pkg, [$logged_in_agent_id], $date_params_tkt, [$logged_in_agent_id, $logged_in_agent_id, $logged_in_agent_id], $date_params_pay);
$types = 'i' . $date_types_pkg . 'i' . $date_types_tkt . 'iii' . $date_types_pay;
$where_pkg_str = 'WHERE ' . implode(' AND ', $where_clauses['pkg']);
$where_tkt_str = 'WHERE ' . implode(' AND ', $where_clauses['tkt']);
$where_pay_str = 'WHERE ' . implode(' AND ', $where_clauses['pay']);

$sql_period_base = "
(SELECT i.id AS original_id, i.issue_date AS transaction_date, 'Booking' AS transaction_type, i.invoice_number AS trans_num, CONCAT(i.guest_name, ' x ', (SELECT COUNT(*) FROM invoice_pilgrims ip WHERE ip.invoice_id = i.id), ' Pax') AS particulars, '' as invoice_reference, i.grand_total_pkr AS debit, 0 AS credit FROM invoices i $where_pkg_str)
UNION ALL
(SELECT ti.id AS original_id, ti.issue_date AS transaction_date, 'Ticket' AS transaction_type, ti.invoice_number AS trans_num, CONCAT(ti.guest_name, ' x ', (SELECT COUNT(*) FROM ticket_invoice_passengers tip WHERE tip.ticket_invoice_id = ti.id), ' Pax') AS particulars, '' as invoice_reference, ti.grand_total_pkr AS debit, 0 AS credit FROM ticket_invoices ti $where_tkt_str)
UNION ALL
(SELECT p.id AS original_id, p.payment_date AS transaction_date, 'Payment Received' AS transaction_type, CASE WHEN p.payment_method IN ('Bank Transfer', 'Card') THEN CONCAT('BR-', p.id) WHEN p.payment_method = 'Cash' THEN CONCAT('CR-', p.id) ELSE CONCAT('PAY-', p.id) END AS trans_num, p.notes AS particulars, p.invoice_reference, 0 AS debit, p.credit_amount AS credit FROM payments p $where_pay_str)
ORDER BY transaction_date ASC, original_id ASC";

$transactions_raw = [];
$stmt_period = $conn->prepare($sql_period_base);
if ($stmt_period) {
    if (!empty($params)) {
        $stmt_period->bind_param($types, ...$params);
    }
    $stmt_period->execute();
    $result = $stmt_period->get_result();
    if ($result) {
        $transactions_raw = $result->fetch_all(MYSQLI_ASSOC);
    }
    $stmt_period->close();
}

// --- 5. FINAL CALCULATIONS FOR DISPLAY ---
$transactions = [];
$running_balance = $opening_balance;
$total_debit_in_period = 0;
$total_credit_in_period = 0;
foreach ($transactions_raw as $transaction) {
    $debit = (float)$transaction['debit'];
    $credit = (float)$transaction['credit'];
    $total_debit_in_period += $debit;
    $total_credit_in_period += $credit;
    $running_balance += $debit - $credit;
    $transaction['balance'] = $running_balance;
    $transactions[] = $transaction;
}
$closing_balance = $running_balance;
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>Ledger Statement for <?= e($agent_details['name']) ?></title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <style>
        :root {
            --theme-color: #f0f0f0;
            --border-color: #ccc;
        }

        body {
            font-family: Arial, sans-serif;
            background-color: #e9e9e9;
            margin: 0;
            padding: 20px;
            font-size: 10pt;
            color: #333;
        }

        .actions-bar {
            max-width: 1100px;
            margin: 0 auto 15px auto;
            text-align: right;
        }

        .btn {
            padding: 8px 15px;
            border: none;
            border-radius: 4px;
            color: white;
            font-size: 14px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
        }

        .btn-print {
            background-color: #0d2d4c;
        }

        .print-wrapper {
            max-width: 1100px;
            margin: 0 auto;
            padding: 30px;
            border: 1px solid #ccc;
            background-color: #fff;
            box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
        }

        table {
            width: 100%;
            border-collapse: collapse;
        }

        td,
        th {
            padding: 4px;
            vertical-align: top;
        }

        .header-table td {
            border: none;
            padding: 0;
        }

        .agent-logo {
            width: 33%;
            height: 80px;
            text-align: left;
        }

        .agent-logo img {
            max-height: 80px;
            max-width: 180px;
        }

        .company-logo-container {
            width: 34%;
            text-align: center;
        }

        .company-logo-container img {
            max-height: 50px;
        }

        .company-details {
            font-size: 9pt;
            line-height: 1.4;
            padding-top: 5px;
        }

        .statement-meta {
            width: 33%;
        }

        .statement-meta table {
            border: 1px solid var(--border-color);
        }

        .statement-meta td {
            padding: 5px 8px;
            font-size: 9pt;
        }

        .statement-meta td:first-child {
            font-weight: bold;
            background-color: var(--theme-color);
            width: 100px;
        }

        .customer-details {
            border: 1px solid var(--border-color);
            margin-top: 20px;
            padding: 15px;
            background: #fafafa;
            border-radius: 5px;
        }

        .customer-details h3 {
            margin: 0 0 10px 0;
            font-size: 12pt;
            border-bottom: 1px solid #eee;
            padding-bottom: 5px;
        }

        .customer-details table td {
            padding: 3px 0;
        }

        .customer-details table td:first-child {
            font-weight: bold;
            width: 120px;
        }

        .summary-container {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 1rem;
            margin: 20px 0;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e0e0e0;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        .summary-item .value {
            font-size: 1.5em;
            font-weight: 600;
        }

        .summary-item .debit {
            color: #c23616;
        }

        .summary-item .credit {
            color: #44bd32;
        }

        .summary-item .balance {
            color: #0056b3;
        }

        .ledger-table th {
            background-color: var(--theme-color);
            border: 1px solid var(--border-color);
            padding: 8px;
            text-align: left;
        }

        .ledger-table td {
            border: 1px solid var(--border-color);
            padding: 7px;
            vertical-align: middle;
        }

        .ledger-table td.number {
            text-align: right;
            font-family: monospace;
        }

        .ledger-table .particulars {
            white-space: normal;
            word-break: break-word;
        }

        .ledger-table .balance-row,
        .ledger-table .totals-row {
            font-weight: bold;
            background-color: #f9f9f9;
        }

        .footer {
            text-align: center;
            margin-top: 30px;
            font-size: 9pt;
            color: #777;
            border-top: 1px solid #eee;
            padding-top: 15px;
        }

        @media print {
            body {
                background-color: #fff;
                margin: 0;
                padding: 0;
                font-size: 9pt;
            }

            .actions-bar {
                display: none;
            }

            .print-wrapper {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
                max-width: 100%;
            }

            * {
                color-adjust: exact !important;
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
            }
        }
    </style>
</head>

<body>
    <div class="actions-bar">
        <a href="javascript:window.print()" class="btn btn-print">Print Statement</a>
    </div>

    <div class="print-wrapper">
        <header>
            <table class="header-table">
                <tr>
                    <td class="agent-logo">
                        <?php if (!empty($agent_details['logo_path'])): ?>
                            <img src="uploads/logos/<?= e($agent_details['logo_path']) ?>" alt="Agent Logo">
                        <?php endif; ?>
                    </td>
                    <td class="company-logo-container">
                        <img src="images/logo.png" alt="Company Logo">
                        <div class="company-details">AL Quresh Near Railway Pahatak,  Infront of Al Quresh Housing Scheme Sher Shah Road Multan</div>
                    </td>
                    <td class="statement-meta">
                        <table>
                            <tr>
                                <td colspan="2" style="text-align:center; font-weight:bold; font-size: 14pt; background: #333; color: #fff;">Ledger Statement</td>
                            </tr>
                            <tr>
                                <td>Statement Date:</td>
                                <td><?= date('d M, Y') ?></td>
                            </tr>
                            <tr>
                                <td>Period:</td>
                                <td><?= !empty($filter_start_date) || !empty($filter_end_date) ? e(date('d M Y', strtotime($filter_start_date))) . ' to ' . e(date('d M Y', strtotime($filter_end_date))) : 'All Time' ?></td>
                            </tr>
                        </table>
                    </td>
                </tr>
            </table>
        </header>

        <section class="customer-details">
            <h3>Statement For:</h3>
            <table>
                <?php if (!empty($agent_details['name'])): ?><tr>
                        <td>Name:</td>
                        <td><?= e($agent_details['name']) ?></td>
                    </tr><?php endif; ?>
                <?php if (!empty($agent_details['company_name'])): ?><tr>
                        <td>Company:</td>
                        <td><?= e($agent_details['company_name']) ?></td>
                    </tr><?php endif; ?>
                <?php if (!empty($agent_details['email'])): ?><tr>
                        <td>Email:</td>
                        <td><?= e($agent_details['email']) ?></td>
                    </tr><?php endif; ?>
                <?php if (!empty($agent_details['mobile_number'])): ?><tr>
                        <td>Mobile:</td>
                        <td><?= e($agent_details['mobile_number']) ?></td>
                    </tr><?php endif; ?>
                <?php if (!empty($agent_details['company_address'])): ?><tr>
                        <td>Address:</td>
                        <td><?= e($agent_details['company_address']) ?></td>
                    </tr><?php endif; ?>
            </table>
        </section>

        <section class="summary-container">
            <div class="summary-item">
                <div class="label">Opening Balance</div>
                <div class="value balance"><?= number_format($opening_balance, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">New Invoices</div>
                <div class="value debit"><?= number_format($total_debit_in_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Payments Made</div>
                <div class="value credit"><?= number_format($total_credit_in_period, 2) ?></div>
            </div>
            <div class="summary-item">
                <div class="label">Closing Balance</div>
                <div class="value balance"><?= number_format($closing_balance, 2) ?></div>
            </div>
        </section>

        <main>
            <table class="ledger-table">
                <thead>
                    <tr>
                        <th style="width: 10%;">Date</th>
                        <th style="width: 12%;">Type</th>
                        <th style="width: 10%;">Trans.#</th>
                        <th>Particulars</th>
                        <th style="width: 12%;">Inv/Ref</th>
                        <th class="number" style="width: 10%;">Debit</th>
                        <th class="number" style="width: 10%;">Credit</th>
                        <th class="number" style="width: 10%;">Balance</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (!empty($filter_start_date)): ?>
                        <tr class="balance-row">
                            <td colspan="7"><strong>Opening Balance</strong></td>
                            <td class="number"><strong><?= number_format($opening_balance, 2) ?></strong></td>
                        </tr>
                    <?php endif; ?>
                    <?php if (empty($transactions)): ?>
                        <tr>
                            <td colspan="8" style="text-align: center; padding: 20px;">No transactions found in the selected period.</td>
                        </tr>
                        <?php else:
                        foreach ($transactions as $transaction):
                        ?>
                            <tr>
                                <td><?= date('d M, Y', strtotime(e($transaction['transaction_date']))) ?></td>
                                <td><?= e($transaction['transaction_type']) ?></td>
                                <td><?= e($transaction['trans_num']) ?></td>
                                <td class="particulars"><?= e($transaction['particulars']) ?></td>
                                <td><?= e($transaction['invoice_reference']) ?></td>
                                <td class="number"><?= (float)$transaction['debit'] > 0 ? number_format((float)$transaction['debit'], 2) : '' ?></td>
                                <td class="number"><?= (float)$transaction['credit'] > 0 ? number_format((float)$transaction['credit'], 2) : '' ?></td>
                                <td class="number"><?= number_format((float)$transaction['balance'], 2) ?></td>
                            </tr>
                    <?php endforeach;
                    endif; ?>
                    <tr class="totals-row">
                        <td colspan="5" style="text-align:right;"><strong>Period Totals</strong></td>
                        <td class="number"><strong><?= number_format($total_debit_in_period, 2) ?></strong></td>
                        <td class="number"><strong><?= number_format($total_credit_in_period, 2) ?></strong></td>
                        <td></td>
                    </tr>
                    <tr class="balance-row">
                        <td colspan="7" style="text-align:right;"><strong>Closing Balance</strong></td>
                        <td class="number"><strong><?= number_format($closing_balance, 2) ?></strong></td>
                    </tr>
                </tbody>
            </table>
        </main>
        <footer class="footer">This is a computer-generated statement and does not require a signature.</footer>
    </div>
</body>

</html>