<?php
session_start();
include_once 'db-config.php';

// --- SECURITY CHECK 1: Is user logged in? ---
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

// --- SECURITY CHECK 2: Is the logged-in user an AGENT? ---
if (!isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: my-account.php"); // Redirect non-agents
    exit;
}

// A helper function for safely echoing output
function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// --- 1. SETUP FILTERS ---
$logged_in_agent_id = (int)$_SESSION['user_id'];
$filter_start_date = $_GET['start_date'] ?? '';
$filter_end_date = $_GET['end_date'] ?? '';

// --- 2. CALCULATE THE OPENING BALANCE ---
$opening_balance = 0;
if (!empty($filter_start_date)) {
    // This query correctly calculates the opening balance based on all transactions before the start date for this agent.
    $sql_opening_base = "
        SELECT SUM(balance) as opening_balance FROM (
            -- Invoices (Debit)
            SELECT SUM(i.grand_total_pkr) as balance FROM invoices i WHERE i.issue_date < ? AND i.user_id = ?
            UNION ALL
            SELECT SUM(ti.grand_total_pkr) as balance FROM ticket_invoices ti WHERE ti.issue_date < ? AND ti.user_id = ?
            UNION ALL
            -- Payments (Credit)
            SELECT -SUM(p.credit_amount) as balance FROM payments p 
            WHERE p.payment_date < ? AND p.vendor_id IS NULL AND ( -- FIX: Ensure no vendor payments are ever included
                p.user_id = ? OR -- Condition 1: Payment is directly linked to the agent
                EXISTS (SELECT 1 FROM invoices i WHERE i.id = p.invoice_id AND p.invoice_type = 'package' AND i.user_id = ?) OR -- Condition 2: Linked via a package invoice
                EXISTS (SELECT 1 FROM ticket_invoices ti WHERE ti.id = p.invoice_id AND p.invoice_type = 'ticket' AND ti.user_id = ?)  -- Condition 3: Linked via a ticket invoice
            )
        ) AS opening_parts
    ";
    $stmt_opening = $conn->prepare($sql_opening_base);
    if ($stmt_opening) {
        // Types: s,i, s,i, s, i,i,i (for date, user_id combos)
        $stmt_opening->bind_param(
            "sisisiii",
            $filter_start_date,
            $logged_in_agent_id,
            $filter_start_date,
            $logged_in_agent_id,
            $filter_start_date,
            $logged_in_agent_id,
            $logged_in_agent_id,
            $logged_in_agent_id
        );
        $stmt_opening->execute();
        $result_opening = $stmt_opening->get_result()->fetch_assoc();
        $opening_balance = (float)($result_opening['opening_balance'] ?? 0);
        $stmt_opening->close();
    }
}

// --- 3. BUILD MAIN TRANSACTION QUERY ---
$params = [];
$types = '';

$where_clauses = [
    'pkg' => ['i.user_id = ?'],
    'tkt' => ['ti.user_id = ?'],
    'pay' => [
        'p.vendor_id IS NULL', // FIX: Ensure no vendor payments are ever included
        '(p.user_id = ? OR ' . // Condition 1: Direct link
            'EXISTS (SELECT 1 FROM invoices i2 WHERE i2.id = p.invoice_id AND p.invoice_type = \'package\' AND i2.user_id = ?) OR ' . // Condition 2: Via package invoice
            'EXISTS (SELECT 1 FROM ticket_invoices ti2 WHERE ti2.id = p.invoice_id AND p.invoice_type = \'ticket\' AND ti2.user_id = ?))' // Condition 3: Via ticket invoice
    ]
];

$date_params_pkg = [];
$date_types_pkg = '';
$date_params_tkt = [];
$date_types_tkt = '';
$date_params_pay = [];
$date_types_pay = '';
if (!empty($filter_start_date) && !empty($filter_end_date)) {
    $where_clauses['pkg'][] = "i.issue_date BETWEEN ? AND ?";
    $date_params_pkg = [$filter_start_date, $filter_end_date];
    $date_types_pkg = 'ss';
    $where_clauses['tkt'][] = "ti.issue_date BETWEEN ? AND ?";
    $date_params_tkt = [$filter_start_date, $filter_end_date];
    $date_types_tkt = 'ss';
    $where_clauses['pay'][] = "p.payment_date BETWEEN ? AND ?";
    $date_params_pay = [$filter_start_date, $filter_end_date];
    $date_types_pay = 'ss';
} else if (!empty($filter_start_date)) {
    $where_clauses['pkg'][] = "i.issue_date >= ?";
    $date_params_pkg = [$filter_start_date];
    $date_types_pkg = 's';
    $where_clauses['tkt'][] = "ti.issue_date >= ?";
    $date_params_tkt = [$filter_start_date];
    $date_types_tkt = 's';
    $where_clauses['pay'][] = "p.payment_date >= ?";
    $date_params_pay = [$filter_start_date];
    $date_types_pay = 's';
} else if (!empty($filter_end_date)) {
    $where_clauses['pkg'][] = "i.issue_date <= ?";
    $date_params_pkg = [$filter_end_date];
    $date_types_pkg = 's';
    $where_clauses['tkt'][] = "ti.issue_date <= ?";
    $date_params_tkt = [$filter_end_date];
    $date_types_tkt = 's';
    $where_clauses['pay'][] = "p.payment_date <= ?";
    $date_params_pay = [$filter_end_date];
    $date_types_pay = 's';
}

$params = array_merge(
    [$logged_in_agent_id],
    $date_params_pkg,
    [$logged_in_agent_id],
    $date_params_tkt,
    [$logged_in_agent_id, $logged_in_agent_id, $logged_in_agent_id],
    $date_params_pay
);
$types = 'i' . $date_types_pkg . 'i' . $date_types_tkt . 'iii' . $date_types_pay;

$where_pkg_str = 'WHERE ' . implode(' AND ', $where_clauses['pkg']);
$where_tkt_str = 'WHERE ' . implode(' AND ', $where_clauses['tkt']);
$where_pay_str = 'WHERE ' . implode(' AND ', $where_clauses['pay']);

// --- 4. FETCH TRANSACTIONS & STATS ---
// UPDATED SQL TO MATCH THE NEW PATTERN
$sql_period_base = "
(SELECT i.id AS original_id, i.issue_date AS transaction_date, 'Booking' AS transaction_type, i.invoice_number AS trans_num, 
 CONCAT(i.guest_name, ' x ', (SELECT COUNT(*) FROM invoice_pilgrims ip WHERE ip.invoice_id = i.id), ' Pax') AS particulars, 
 '' as invoice_reference, i.grand_total_pkr AS debit, 0 AS credit 
 FROM invoices i $where_pkg_str)
UNION ALL
(SELECT ti.id AS original_id, ti.issue_date AS transaction_date, 'Ticket' AS transaction_type, ti.invoice_number AS trans_num, 
 CONCAT(ti.guest_name, ' x ', (SELECT COUNT(*) FROM ticket_invoice_passengers tip WHERE tip.ticket_invoice_id = ti.id), ' Pax') AS particulars, 
 '' as invoice_reference, ti.grand_total_pkr AS debit, 0 AS credit 
 FROM ticket_invoices ti $where_tkt_str)
UNION ALL
(SELECT p.id AS original_id, p.payment_date AS transaction_date, 'Payment Received' AS transaction_type, 
 CASE 
    WHEN p.payment_method IN ('Bank Transfer', 'Card') THEN CONCAT('BR-', p.id)
    WHEN p.payment_method = 'Cash' THEN CONCAT('CR-', p.id)
    ELSE CONCAT('PAY-', p.id)
 END AS trans_num,
 p.notes AS particulars, p.invoice_reference, 0 AS debit, p.credit_amount AS credit 
 FROM payments p $where_pay_str)
ORDER BY transaction_date ASC, original_id ASC";

$transactions_raw = [];
$stmt_period = $conn->prepare($sql_period_base);
if ($stmt_period) {
    if (!empty($params)) {
        $stmt_period->bind_param($types, ...$params);
    }
    $stmt_period->execute();
    $result = $stmt_period->get_result();
    if ($result) {
        $transactions_raw = $result->fetch_all(MYSQLI_ASSOC);
    }
    $stmt_period->close();
} else {
    die("SQL Error (Period Transactions): " . $conn->error);
}

// --- CALCULATE SUMMARY STATISTICS FOR THE AGENT ---
$summary_stats = ['package_invoice_count' => 0, 'ticket_invoice_count' => 0, 'pilgrim_count' => 0, 'hotel_count' => 0];
$stats_sql_base = "
    SELECT
        (SELECT COUNT(DISTINCT i.id) FROM invoices i %s) as package_invoice_count,
        (SELECT COUNT(DISTINCT ti.id) FROM ticket_invoices ti %s) as ticket_invoice_count,
        (SELECT COUNT(ip.id) FROM invoice_pilgrims ip JOIN invoices i ON ip.invoice_id = i.id %s) as pilgrim_count,
        (SELECT COUNT(ih.id) FROM invoice_hotels ih JOIN invoices i ON ih.invoice_id = i.id %s) as hotel_count
";
$stats_params = array_merge([$logged_in_agent_id], $date_params_pkg, [$logged_in_agent_id], $date_params_tkt, [$logged_in_agent_id], $date_params_pkg, [$logged_in_agent_id], $date_params_pkg);
$stats_types = 'i' . $date_types_pkg . 'i' . $date_types_tkt . 'i' . $date_types_pkg . 'i' . $date_types_pkg;

$final_stats_sql = sprintf($stats_sql_base, $where_pkg_str, $where_tkt_str, $where_pkg_str, $where_pkg_str);
$stmt_stats = $conn->prepare($final_stats_sql);
if ($stmt_stats) {
    if (!empty($stats_params)) {
        $stmt_stats->bind_param($stats_types, ...$stats_params);
    }
    $stmt_stats->execute();
    $stats_result = $stmt_stats->get_result();
    if ($stats_result) {
        $summary_stats = $stats_result->fetch_assoc();
    }
    $stmt_stats->close();
}

// --- 5. PROCESS DATA & CALCULATE TOTALS ---
$transactions = [];
$running_balance = $opening_balance;
$total_debit_in_period = 0;
$total_credit_in_period = 0;
foreach ($transactions_raw as $transaction) {
    $debit = (float)$transaction['debit'];
    $credit = (float)$transaction['credit'];
    $total_debit_in_period += $debit;
    $total_credit_in_period += $credit;
    $running_balance += $debit - $credit;
    $transaction['balance'] = $running_balance;
    $transactions[] = $transaction;
}
?>
<!DOCTYPE html>
<html>

<head>
    <title>Agent Ledger</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        .filter-container {
            background-color: #f8f9fa;
            padding: 1.5rem;
            border: 1px solid #e9ecef;
            border-radius: 8px;
            margin-bottom: 2rem;
        }

        .filter-main-row {
            display: flex;
            align-items: flex-end;
            gap: 1.5rem;
            flex-wrap: wrap;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
        }

        .filter-group label {
            font-size: 0.9em;
            margin-bottom: 0.5rem;
            color: #495057;
            font-weight: 500;
        }

        .form-control {
            border: 1px solid #ced4da;
            border-radius: 0.375rem;
            padding: 0.5rem 0.75rem;
            font-size: 1rem;
        }

        .btn-action {
            padding: 0.5rem 1rem;
            border-radius: 0.375rem;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            color: #fff;
            border: none;
            cursor: pointer;
        }

        .btn-filter {
            background-color: #0d2d4c;
        }

        .btn-clear {
            background-color: #6c757d;
        }

        .filter-secondary-row {
            margin-top: 1rem;
        }

        .btn-print {
            background-color: #17a2b8;
        }

        .summary-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }

        .summary-item {
            text-align: center;
            padding: 1rem;
            background-color: #fff;
            border-radius: 6px;
            border: 1px solid #e9ecef;
        }

        .summary-item .label {
            font-size: 0.9em;
            color: #6c757d;
            margin-bottom: 5px;
            text-transform: uppercase;
        }

        .summary-item .value {
            font-size: 1.75em;
            font-weight: 600;
        }

        .summary-item .debit {
            color: #c23616;
        }

        .summary-item .credit {
            color: #44bd32;
        }

        .summary-item .balance {
            color: #0056b3;
        }

        .data-summary {
            background-color: #e6f7ff;
            border: 1px solid #b3e0ff;
            padding: 1rem;
            border-radius: 8px;
        }

        .data-summary .value {
            color: #0056b3;
        }

        .bookings-table .number {
            text-align: right;
        }

        .bookings-table .balance {
            font-weight: bold;
        }

        .bookings-table th,
        .bookings-table td {
            white-space: nowrap;
        }

        .bookings-table td:nth-child(4),
        .bookings-table td:nth-child(5) {
            white-space: normal;
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <h2><i class="fa-solid fa-file-invoice-dollar"></i> My Ledger</h2>
                    <p class="content-description">Review your complete transaction history and current balance.</p>

                    <div class="filter-container">
                        <form method="GET" action="agent-ledger.php">
                            <div class="filter-main-row">
                                <div class="filter-group"><label for="start_date">Start Date</label><input type="date" name="start_date" id="start_date" value="<?= e($filter_start_date) ?>" class="form-control"></div>
                                <div class="filter-group"><label for="end_date">End Date</label><input type="date" name="end_date" id="end_date" value="<?= e($filter_end_date) ?>" class="form-control"></div>
                                <div class="filter-group" style="flex-direction: row; gap: 0.5rem;">
                                    <button type="submit" class="btn-action btn-filter"><i class="fas fa-filter"></i> Filter</button>
                                    <a href="agent-ledger.php" class="btn-action btn-clear"><i class="fas fa-times"></i> Clear</a>
                                </div>
                            </div>
                            <div class="filter-secondary-row">
                                <a href="agent-ledger-print.php?<?= http_build_query($_GET) ?>" target="_blank" class="btn-action btn-print"><i class="fas fa-print"></i> Print View</a>
                            </div>
                        </form>
                    </div>

                    <div class="summary-container">
                        <div class="summary-item">
                            <div class="label">Opening Balance</div>
                            <div class="value balance"><?= number_format($opening_balance, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">New Invoices</div>
                            <div class="value debit"><?= number_format($total_debit_in_period, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Payments Made</div>
                            <div class="value credit"><?= number_format($total_credit_in_period, 2) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Closing Balance</div>
                            <div class="value balance"><?= number_format($running_balance, 2) ?></div>
                        </div>
                    </div>

                    <div class="summary-container data-summary">
                        <div class="summary-item">
                            <div class="label">Bookings</div>
                            <div class="value"><?= e($summary_stats['package_invoice_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Ticket</div>
                            <div class="value"><?= e($summary_stats['ticket_invoice_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Total Pilgrims</div>
                            <div class="value"><?= e($summary_stats['pilgrim_count']) ?></div>
                        </div>
                        <div class="summary-item">
                            <div class="label">Hotel Bookings</div>
                            <div class="value"><?= e($summary_stats['hotel_count']) ?></div>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Trans.#</th>
                                    <th>Particulars</th>
                                    <th>Inv/Ref</th>
                                    <th class="number">Debit</th>
                                    <th class="number">Credit</th>
                                    <th class="number">Balance</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($filter_start_date)): ?>
                                    <tr>
                                        <td><?= date('d M, Y', strtotime($filter_start_date)) ?></td>
                                        <td></td>
                                        <td></td>
                                        <td><strong>Opening Balance</strong></td>
                                        <td></td>
                                        <td></td>
                                        <td></td>
                                        <td class="number balance"><strong><?= number_format($opening_balance, 2) ?></strong></td>
                                    </tr>
                                <?php endif; ?>
                                <?php if (empty($transactions)): ?>
                                    <tr>
                                        <td colspan="8" class="no-bookings-found">No transactions found for the selected criteria.</td>
                                    </tr>
                                <?php else: ?>
                                    <?php foreach ($transactions as $transaction): ?>
                                        <tr>
                                            <td><?= date('d M, Y', strtotime(e($transaction['transaction_date']))) ?></td>
                                            <td><?= e($transaction['transaction_type']) ?></td>
                                            <td><?= e($transaction['trans_num']) ?></td>
                                            <td><?= e($transaction['particulars']) ?></td>
                                            <td><?= e($transaction['invoice_reference']) ?></td>
                                            <td class="number debit"><?= (float)$transaction['debit'] > 0 ? number_format((float)$transaction['debit'], 2) : '' ?></td>
                                            <td class="number credit"><?= (float)$transaction['credit'] > 0 ? number_format((float)$transaction['credit'], 2) : '' ?></td>
                                            <td class="number balance"><strong><?= number_format((float)$transaction['balance'], 2) ?></strong></td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </tbody>
                            <tfoot>
                                <tr style="background-color: #f8f9fa; font-weight: bold;">
                                    <td colspan="5" style="text-align: right;">Total for Period:</td>
                                    <td class="number debit"><?= number_format($total_debit_in_period, 2) ?></td>
                                    <td class="number credit"><?= number_format($total_credit_in_period, 2) ?></td>
                                    <td class="number balance"><?= number_format($running_balance, 2) ?></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>
</body>

</html>