<?php

/**
 * agent-vouchers.php (v2)
 * - Dedicated voucher management dashboard for agents.
 * - Fetches and lists all vouchers assigned to the agent.
 * - Clicking a row navigates to a new page to view the full voucher.
 * - Edit button is only shown for vouchers with 'Tentative' status.
 */

session_start(); // Ensure session is started
include 'db-config.php';

// --- SECURITY CHECKS ---
if (!isset($_SESSION['user_id']) || !isset($_SESSION['user_type']) || $_SESSION['user_type'] !== 'agent') {
    header("location: login.php");
    exit;
}

// --- Get the logged-in agent's details ---
$user_id = $_SESSION['user_id'];
$stmt_user = $conn->prepare("SELECT name, email FROM users WHERE id = ?");
$stmt_user->bind_param("i", $user_id);
$stmt_user->execute();
$current_user = $stmt_user->get_result()->fetch_assoc();
$stmt_user->close();

$user_name = $current_user['name'] ?? 'Agent';
$user_email = $current_user['email'] ?? 'N/A';


// --- FETCH VOUCHER LIST (Simplified for speed) ---
// We only need the data for the list view, not all sub-table details.
$vouchers_data = [];
$sql_vouchers = "SELECT id, family_head_name, voucher_date, status, package_type, package_duration_nights FROM vouchers WHERE user_id = ? ORDER BY voucher_date DESC, id DESC";
$stmt_vouchers = $conn->prepare($sql_vouchers);
$stmt_vouchers->bind_param("i", $user_id);
$stmt_vouchers->execute();
$vouchers_result = $stmt_vouchers->get_result();

if ($vouchers_result) {
    $vouchers_data = $vouchers_result->fetch_all(MYSQLI_ASSOC);
}
$stmt_vouchers->close();

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}
?>
<!DOCTYPE html>
<html>

<head>
    <title>My Vouchers</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    <link rel="stylesheet" href="css/account-style.css">
    <style>
        /* --- HEADER ALIGNMENT FIX --- */
        .content-card-header {
            display: flex;
            justify-content: space-between;
            /* Pushes title and button to opposite ends */
            align-items: center;
            /* Vertically aligns them in the middle */
            padding-bottom: 10px;
            /* Adds space below the header */
        }

        /* --- TABLE ALIGNMENT FIXES --- */
        .bookings-table th,
        .bookings-table td {
            vertical-align: middle;
            padding: 14px 10px;
        }

        .bookings-table th,
        .bookings-table td {
            text-align: left;
        }

        .bookings-table th:nth-child(1),
        /* Voucher # Header */
        .bookings-table td:nth-child(1),
        /* Voucher # Data */
        .bookings-table th:nth-child(4),
        /* Status Header */
        .bookings-table td:nth-child(4),
        /* Status Data */
        .bookings-table th:nth-child(6),
        /* Actions Header */
        .bookings-table td:nth-child(6)

        /* Actions Data */
            {
            text-align: center;
        }

        .bookings-table .action-buttons {
            width: 100px;
        }

        .action-buttons a {
            margin: 0 5px;
            /* Provides even spacing between action icons */
        }
    </style>
</head>

<body>

    <?php include 'header.php'; ?>

    <main class="account-page-wrapper">
        <div class="account-container">
            <?php include 'agent-sidebar.php'; ?>
            <div class="account-content">
                <div class="content-card">
                    <div class="content-card-header">
                        <h2><i class="fa-solid fa-ticket"></i> My Vouchers</h2>
                        <a href="manage-requests.php?action=create" class="btn btn-primary"><i class="fa-solid fa-plus"></i> Request New Voucher</a>
                    </div>
                    <p class="content-description">Review all vouchers created for you. Click on any row to view full details on a new page.</p>
                    <div class="table-responsive">
                        <table class="bookings-table">
                            <thead>
                                <tr>
                                    <th>Voucher #</th>
                                    <th>Family Head</th>
                                    <th>Voucher Date</th>
                                    <th>Status</th>
                                    <th>Package</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (!empty($vouchers_data)): ?>
                                    <?php foreach ($vouchers_data as $voucher): ?>
                                        <tr class="voucher-row" data-voucher-id="<?= e($voucher['id']) ?>" style="cursor: pointer;">
                                            <td><strong>#<?= e($voucher['id']) ?></strong></td>
                                            <td><?= e($voucher['family_head_name']) ?></td>
                                            <td><?= date('M j, Y', strtotime($voucher['voucher_date'])) ?></td>
                                            <td><span class="status-badge status-<?= strtolower(e($voucher['status'])) ?>"><?= e($voucher['status']) ?></span></td>
                                            <td><?= e($voucher['package_type']) ?> (<?= e($voucher['package_duration_nights']) ?>N)</td>
                                            <td class="action-buttons">
                                                <!-- Always show View button -->
                                                <a href="view-voucher-agent.php?id=<?= $voucher['id'] ?>" class="btn-sm btn-view" title="View">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <!-- Only show Edit button if status is Tentative -->

                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="6" class="no-bookings-found">You do not have any vouchers yet.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <?php include 'footer.php'; ?>

    <!-- NO MODAL HTML NEEDED -->

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const voucherRows = document.querySelectorAll('.voucher-row');

            voucherRows.forEach(row => {
                row.addEventListener('click', function(event) {
                    // Prevent navigation if an action button (<a> tag) inside the row was clicked
                    if (event.target.closest('a')) {
                        return;
                    }

                    // Get the voucher ID from the data attribute
                    const voucherId = this.dataset.voucherId;
                    if (voucherId) {
                        // Navigate to the view page for this voucher
                        window.location.href = `view-voucher-agent.php?id=${voucherId}`;
                    }
                });
            });
        });
    </script>

</body>

</html>