<?php
// THIS LINE IS THE FIX. It ensures the database connection is always available.
if (session_status() === PHP_SESSION_NONE) { session_start(); }
include_once 'db-config.php';

// ===================================================================
//  CONFIGURATION
// ===================================================================
define('DEFAULT_SAR_TO_PKR_RATE', 75.50); // Fallback rate if a hotel-specific rate isn't set.

// ===================================================================
//  PREPARE AGENT/COMPANY DATA FOR LOGOS AND POSTS
// ===================================================================
$is_agent = (isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent');
$display_logo = 'images/logo.png'; // Default company logo
$footer_address = 'Al Quresh, Railway Phatak, Multan'; // Default address
$footer_phone_1 = '923052394810'; // Default WhatsApp/Phone number (no +)

if ($is_agent && isset($_SESSION['user_id'])) {
    $agent_id = $_SESSION['user_id'];
    $stmt_agent = $conn->prepare("SELECT logo_path, company_address, company_phone_1 FROM users WHERE id = ?");
    if ($stmt_agent) {
        $stmt_agent->bind_param("i", $agent_id);
        $stmt_agent->execute();
        $result_agent = $stmt_agent->get_result();
        if ($agent_data = $result_agent->fetch_assoc()) {
            if (!empty($agent_data['logo_path'])) { $display_logo = 'admin/' . $agent_data['logo_path']; }
            if (!empty($agent_data['company_address'])) { $footer_address = $agent_data['company_address']; }
            if (!empty($agent_data['company_phone_1'])) { $footer_phone_1 = preg_replace('/[^0-9]/', '', $agent_data['company_phone_1']); }
        }
        $stmt_agent->close();
    }
}
// Helper functions to prevent errors and reuse code
if (!function_exists('safe_htmlspecialchars')) {
    function safe_htmlspecialchars($string) { return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8'); }
}
if (!function_exists('generate_star_rating_html')) {
    function generate_star_rating_html($rating) {
        $html = ''; $rating = (float) $rating;
        for ($i = 1; $i <= 5; $i++) {
            if ($rating >= $i) { $html .= '<i class="fa-solid fa-star"></i>'; }
            elseif ($rating > ($i - 1) && $rating < $i) { $html .= '<i class="fa-solid fa-star-half-stroke"></i>'; }
            else { $html .= '<i class="fa-regular fa-star"></i>'; }
        } return $html;
    }
}
?>
<!-- =================================================================== -->
<!-- === ALL HOTELS (RESPONSIVE FILTERS & PAGINATION) - FINAL VERSION === -->
<!-- =================================================================== -->
<section class="all-hotels-listings-section">
    <div class="listings-container">
        <?php
        // --- 1. FETCH DYNAMIC FILTER OPTIONS FROM DATABASE ---
        $available_locations = [];
        try {
            $location_result = $conn->query("SELECT DISTINCT location FROM hotels WHERE is_active = 1 AND location IS NOT NULL AND location != '' ORDER BY location ASC");
            if ($location_result) {
                while ($row = $location_result->fetch_assoc()) {
                    $available_locations[] = $row['location'];
                }
            }
        } catch (Exception $e) { error_log("Failed to fetch hotel locations for filter: " . $e->getMessage()); }

        // --- 2. SETUP PAGINATION & GET USER'S FILTER SELECTIONS ---
        $results_per_page = 9; // Changed to 9 for a perfect 3x3 grid
        $current_page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $offset = ($current_page - 1) * $results_per_page;
        $filter_location = $_GET['location'] ?? 'all';

        // --- 3. DYNAMICALLY BUILD THE MAIN SQL QUERY ---
        $sql_base = "FROM hotels"; $where_clauses = ['is_active = 1']; $params = []; $types = '';
        if ($filter_location !== 'all') { $where_clauses[] = "location = ?"; $params[] = $filter_location; $types .= 's'; }
        $where_sql = " WHERE " . implode(" AND ", $where_clauses);
        
        $count_sql = "SELECT COUNT(id) as total " . $sql_base . $where_sql;
        $stmt_count = $conn->prepare($count_sql);
        if(!empty($types)) { $stmt_count->bind_param($types, ...$params); }
        $stmt_count->execute();
        $total_records = $stmt_count->get_result()->fetch_assoc()['total'];
        $total_pages = ceil($total_records / $results_per_page);

        $sql_select = "SELECT * " . $sql_base . $where_sql . " ORDER BY display_order ASC, last_updated DESC LIMIT ? OFFSET ?";
        $params[] = $results_per_page; $params[] = $offset; $types .= 'ii';
        $stmt = $conn->prepare($sql_select);
        if ($stmt && !empty($types)) { $stmt->bind_param($types, ...$params); }
        $stmt->execute();
        $result = $stmt->get_result();
        ?>

        <!-- Section Header -->
        <div class="listings-header">
            <div class="listings-header-text"><h2 class="listings-title">All Our Hotels</h2><p class="listings-subtitle">Browse our complete collection of handpicked hotels.</p></div>
        </div>

        <!-- Dynamic Location Filter Controls -->
        <?php if (!empty($available_locations)): ?>
        <div class="filter-controls-wrapper">
            <div class="location-filter-tags">
                <a href="?" class="filter-tag <?= ($filter_location == 'all' ? 'active' : '') ?>">All Locations</a>
                <?php foreach ($available_locations as $location): ?>
                    <a href="?location=<?= urlencode($location) ?>" class="filter-tag <?= ($filter_location == $location ? 'active' : '') ?>"><?= htmlspecialchars($location) ?></a>
                <?php endforeach; ?>
            </div>
            <div class="location-filter-select-wrapper">
                <i class="fa-solid fa-map-location-dot select-icon"></i>
                <select class="location-filter-select" onchange="window.location.href = this.value;">
                    <option value="?" <?= ($filter_location == 'all' ? 'selected' : '') ?>>All Locations</option>
                    <?php foreach ($available_locations as $location): ?>
                        <option value="?location=<?= urlencode($location) ?>" <?= ($filter_location == $location ? 'selected' : '') ?>"><?= htmlspecialchars($location) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
        <?php endif; ?>
        
        <span id="all-hotels-anchor"></span>

        <div class="hotel-grid-wrapper">
            <div class="hotel-grid">
                <?php
                if ($result && $result->num_rows > 0):
                    while ($hotel = $result->fetch_assoc()):
                        $sar_price = ($is_agent && !empty($hotel['price_per_night_agent'])) ? $hotel['price_per_night_agent'] : $hotel['price_per_night'];
                        $exchange_rate = !empty($hotel['exchange_rate_pkr']) ? $hotel['exchange_rate_pkr'] : DEFAULT_SAR_TO_PKR_RATE;
                        $pkr_price = number_format($sar_price * $exchange_rate);
                        $image_url_with_version = safe_htmlspecialchars($hotel['image_url']) . '?v=' . strtotime($hotel['last_updated']);
                        $logo_bg_class = ($hotel['logo_background_style'] === 'transparent') ? 'transparent-logo' : 'white-bg-logo';
                ?>
                        <div class="hotel-card-v2">
                            <div class="hotel-card-v2__banner">
                                <div class="hotel-card-v2__logo-overlay <?= $logo_bg_class ?>">
                                    <img src="<?= safe_htmlspecialchars($display_logo) ?>" alt="Agency Logo" crossorigin="anonymous">
                                </div>
                                <img src="<?= $image_url_with_version ?>" alt="Image of <?= safe_htmlspecialchars($hotel['hotel_name']) ?>" class="hotel-card-v2__image" crossorigin="anonymous">
                            </div>
                            <div class="hotel-card-v2__content">
                                <h3 class="hotel-card-v2__title"><?= safe_htmlspecialchars($hotel['hotel_name']) ?></h3>
                                <p class="hotel-card-v2__location"><i class="fa-solid fa-location-dot"></i> <?= safe_htmlspecialchars($hotel['location']) ?></p>
                                <div class="hotel-card-v2__rating"><?= generate_star_rating_html($hotel['rating']); ?></div>
                                <p class="hotel-card-v2__price">from <span>PKR <?= $pkr_price ?></span>/night</p>
                            </div>
                            <div class="hotel-card-v2__actions">
                                <?php $whatsapp_message = urlencode("Hi, I'm interested in the hotel: '" . $hotel['hotel_name'] . "' in " . $hotel['location']); ?>
                                <a href="https://wa.me/<?= $footer_phone_1 ?>?text=<?= $whatsapp_message ?>" target="_blank" class="btn-hotel-primary">Enquire</a>
                                <button class="btn-hotel-secondary btn-download-hotel" data-download-id="hotel-<?= $hotel['id'] ?>"><i class="fa-solid fa-download"></i> Download</button>
                            </div>
                        </div>

                        <div class="downloadable-hotel-post" id="download-post-hotel-<?= $hotel['id'] ?>">
                            <img src="<?= $image_url_with_version ?>" class="hotel-post-bg-image" crossorigin="anonymous" alt="">
                            <div class="hotel-post-overlay">
                                <header class="hotel-post-header">
                                    <div class="hotel-post-logo <?= $logo_bg_class ?>">
                                        <img src="<?= safe_htmlspecialchars($display_logo) ?>" alt="Agency Logo" crossorigin="anonymous">
                                    </div>
                                </header>
                                <main class="hotel-post-body">
                                    <div class="hotel-post-title"><h1><?= safe_htmlspecialchars($hotel['hotel_name']) ?></h1><h2><?= safe_htmlspecialchars($hotel['location']) ?></h2></div>
                                    <div class="hotel-post-details-box">
                                        <div class="hotel-post-detail-item"><div class="rating-display"><?= generate_star_rating_html($hotel['rating']) ?></div><h3>Star Rating</h3></div>
                                        <div class="hotel-post-detail-item"><p>PKR <?= $pkr_price ?></p><h3>Price Per Night</h3></div>
                                    </div>
                                </main>
                                <footer class="hotel-post-footer">
                                    <span><i class="fas fa-map-marker-alt"></i> <?= safe_htmlspecialchars($footer_address) ?></span>
                                    <span><i class="fab fa-whatsapp"></i> <?= safe_htmlspecialchars($footer_phone_1) ?></span>
                                </footer>
                            </div>
                        </div>
                <?php
                    endwhile;
                else:
                    echo '<p style="text-align: center; width: 100%; padding: 2rem 0; grid-column: 1 / -1;">No hotels found for the selected location. Please try another filter.</p>';
                endif;
                ?>
            </div>
        </div>

        <?php if ($total_pages > 1): ?>
            <nav class="pagination-container">
                <ul class="pagination">
                    <?php
                    $query_params = $_GET;
                    if ($current_page > 1) {
                        $query_params['page'] = $current_page - 1;
                        echo '<li><a href="?' . http_build_query($query_params) . '#all-hotels-anchor">« Prev</a></li>';
                    }
                    for ($i = 1; $i <= $total_pages; $i++) {
                        $query_params['page'] = $i;
                        echo '<li><a href="?' . http_build_query($query_params) . '#all-hotels-anchor" class="' . ($i == $current_page ? 'active' : '') . '">' . $i . '</a></li>';
                    }
                    if ($current_page < $total_pages) {
                        $query_params['page'] = $current_page + 1;
                        echo '<li><a href="?' . http_build_query($query_params) . '#all-hotels-anchor">Next »</a></li>';
                    }
                    ?>
                </ul>
            </nav>
        <?php endif; ?>
    </div>
</section>

<style>
    /* --- Main Layout & Filters --- */
    .all-hotels-listings-section {
    padding: 40px 0;
    background-color: #f8f9fa;
}

/* Tablets and smaller devices */
@media (max-width: 768px) {
    .all-hotels-listings-section {
        padding: 30px 0; /* slightly reduced padding */
        margin-top: 200px;
    }
}

/* Mobile phones */
@media (max-width: 480px) {
    .all-hotels-listings-section {
        padding: 20px 0; /* tighter spacing for small screens */
    }
}

    .listings-container { max-width: 1700px; margin: 0 auto; padding: 0 2rem; }
    .listings-header { text-align: left; margin-bottom: 40px; }
    .filter-controls-wrapper { margin-bottom: 40px; }
    .location-filter-tags { display: flex; flex-wrap: wrap; justify-content: center; gap: 10px; }
    .filter-tag { display: inline-block; padding: 8px 18px; border: 1px solid #ddd; border-radius: 30px; background-color: var(--white, #fff); color: var(--text-dark, #333); text-decoration: none; font-size: 0.9rem; font-weight: 500; transition: all 0.2s ease-in-out; }
    .filter-tag:hover { border-color: var(--primary-dark, #2c3e50); background-color: var(--primary-dark, #2c3e50); color: var(--white, #fff); }
    .filter-tag.active { border-color: var(--primary-dark, #2c3e50); background-color: var(--primary-dark, #2c3e50); color: var(--white, #fff); cursor: default; }
    .location-filter-select-wrapper { display: none; position: relative; }
    .location-filter-select { width: 100%; padding: 12px 15px 12px 45px; border: 1px solid #ddd; border-radius: 8px; font-size: 1rem; cursor: pointer; -webkit-appearance: none; appearance: none; background-color: var(--white, #fff); background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3e%3cpath fill='none' stroke='%23343a40' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='M2 5l6 6 6-6'/%3e%3c/svg%3e"); background-repeat: no-repeat; background-position: right 0.75rem center; background-size: 16px 12px; }
    .select-icon { position: absolute; left: 15px; top: 50%; transform: translateY(-50%); color: #999; pointer-events: none; }
    .hotel-grid-wrapper { min-height: 500px; }
    /* --- ** GRID FIX: Max 3 columns on desktop ** --- */
    .hotel-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 2rem; }
    @media (min-width: 1200px) {
        .hotel-grid { grid-template-columns: repeat(3, 1fr); }
    }
    #all-hotels-anchor { display: block; position: relative; top: -100px; visibility: hidden; }
    .hotel-card-v2 { background: #fff; border: 1px solid #eef2f6; border-radius: 12px; overflow: hidden; display: flex; flex-direction: column; transition: all 0.3s ease; }
    .hotel-card-v2:hover { transform: translateY(-5px); box-shadow: 0 8px 25px rgba(0,0,0,0.08); }
    .hotel-card-v2__banner { position: relative; }
    .hotel-card-v2__image { width: 100%; height: 220px; object-fit: cover; }
    .hotel-card-v2__logo-overlay { position: absolute; top: 15px; left: 15px; width: 45px; height: 45px; display: flex; align-items: center; justify-content: center; }
    .hotel-card-v2__logo-overlay.white-bg-logo { background: rgba(255,255,255,0.9); backdrop-filter: blur(4px); border-radius: 50%; padding: 5px; box-shadow: 0 1px 5px rgba(0,0,0,0.2); }
    .hotel-card-v2__logo-overlay.transparent-logo { background: none; backdrop-filter: none; box-shadow: none; }
    .hotel-card-v2__logo-overlay.transparent-logo img { filter: drop-shadow(0 2px 3px rgba(0,0,0,0.5)); }
    .hotel-card-v2__logo-overlay img { max-width: 100%; max-height: 100%; object-fit: contain; }
    .hotel-card-v2__content { padding: 1.25rem; flex-grow: 1; display: flex; flex-direction: column; }
    .hotel-card-v2__title { font-size: 1.25rem; font-weight: 600; color: var(--text-dark); margin-bottom: 0.25rem; }
    .hotel-card-v2__location { font-size: 0.9rem; color: var(--text-light); margin-bottom: 1rem; }
    .hotel-card-v2__rating { color: #f39c12; margin-bottom: 1rem; }
    .hotel-card-v2__price { margin-top: auto; font-size: 0.9rem; color: var(--text-light); }
    .hotel-card-v2__price span { font-weight: 700; font-size: 1.5rem; color: var(--text-dark); }
    .hotel-card-v2__actions { display: grid; grid-template-columns: 1fr 1fr; gap: 10px; padding: 1rem; background-color: #f8f9fa; border-top: 1px solid #eef2f6; }
    .btn-hotel-primary, .btn-hotel-secondary { padding: 12px; font-size: 0.9rem; font-weight: 600; border-radius: 8px; cursor: pointer; text-align: center; text-decoration: none; border: 1px solid transparent; transition: all 0.3s; }
    .btn-hotel-primary { background-color: #212529; color: #fff; }
    .btn-hotel-primary:hover { background-color: #343a40; }
    .btn-hotel-secondary { background-color: #f8f9fa; color: #555; border-color: #dee2e6; }
    .btn-hotel-secondary:hover:not(:disabled) { background-color: #e9ecef; }
    .btn-hotel-secondary:disabled { opacity: 0.7; cursor: not-allowed; }
    .btn-hotel-secondary i { margin-right: 5px; }
    .downloadable-hotel-post { display: none; width: 1080px; height: 1080px; font-family: 'Poppins', sans-serif; position: relative; color: #fff; }
    .hotel-post-bg-image { position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; z-index: 1; }
    .hotel-post-overlay { position: relative; z-index: 2; width: 100%; height: 100%; padding: 60px; display: flex; flex-direction: column; justify-content: space-between; background: linear-gradient(to top, rgba(10, 15, 20, 0.98) 0%, rgba(10, 15, 20, 0.9) 50%, rgba(10, 15, 20, 0.4) 100%); }
    .hotel-post-header { display: flex; }
    .hotel-post-logo { width: 140px; height: 140px; display: flex; align-items: center; justify-content: center; padding: 15px; }
    .hotel-post-logo.white-bg-logo { background-color: rgba(255, 255, 255, 0.95); border-radius: 50%; box-shadow: 0 4px 15px rgba(0,0,0,0.3); }
    .hotel-post-logo.transparent-logo img { filter: drop-shadow(0px 3px 5px rgba(0,0,0,0.6)); }
    .hotel-post-logo img { max-width: 100%; max-height: 100%; }
    .hotel-post-body { text-align: center; flex-grow: 1; display: flex; flex-direction: column; justify-content: center; }
    .hotel-post-title h1 { font-size: 90px; font-weight: 700; line-height: 1.1; margin: 0; text-shadow: 2px 3px 8px rgba(0,0,0,0.6); }
    .hotel-post-title h2 { font-size: 40px; font-weight: 400; opacity: 0.9; margin: 5px 0 50px 0; }
    .hotel-post-details-box { background-color: rgba(0, 0, 0, 0.5); backdrop-filter: blur(12px); border-radius: 20px; padding: 30px 50px; display: grid; grid-template-columns: 1fr 1fr; gap: 20px; max-width: 700px; margin: 0 auto; border: 1px solid rgba(255,255,255,0.1); }
    .hotel-post-detail-item .rating-display { font-size: 36px; color: #f39c12; margin-bottom: 10px; }
    .hotel-post-detail-item p { font-size: 48px; font-weight: 700; margin: 0 0 5px 0; }
    .hotel-post-detail-item h3 { font-size: 22px; color: #ccc; font-weight: 500; margin: 0; }
    .hotel-post-footer { background-color: #3498db; border-radius: 20px; padding: 25px 40px; display: flex; justify-content: space-between; align-items: center; font-size: 24px; font-weight: 500; }
    .hotel-post-footer i { margin-right: 15px; font-size: 28px; }
    .pagination-container { margin-top: 50px; display: flex; justify-content: center; }
    .pagination { list-style: none; padding: 0; display: flex; flex-wrap: wrap; justify-content: center; gap: 8px; }
    .pagination li a { display: block; padding: 10px 15px; text-decoration: none; color: var(--text-dark, #333); background-color: var(--white, #fff); border: 1px solid #ddd; border-radius: 8px; font-weight: 500; transition: all 0.3s ease; }
    .pagination li a:hover, .pagination li a.active { background-color: var(--primary-dark, #2c3e50); color: var(--white, #fff); border-color: var(--primary-dark, #2c3e50); }
    .pagination li a.active { cursor: default; }
    @media (max-width: 768px) { .listings-container { padding: 0 1rem; } .location-filter-tags { display: none; } .location-filter-select-wrapper { display: block; } }
</style>

<script>
document.addEventListener('DOMContentLoaded', function () {
    let html2canvasLoaded = false;
    function loadHtml2Canvas(callback) {
        if (html2canvasLoaded) { if (callback) callback(); return; }
        const script = document.createElement('script'); script.src = 'js/html2canvas.min.js';
        script.onload = function () { html2canvasLoaded = true; if (callback) callback(); };
        script.onerror = function() { console.error("Failed to load html2canvas.min.js"); document.querySelectorAll('.btn-download-hotel:disabled').forEach(button => { button.innerHTML = '<i class="fa-solid fa-download"></i> Download'; button.disabled = false; }); };
        document.head.appendChild(script);
    }
    document.querySelectorAll('.btn-download-hotel').forEach(button => {
        button.addEventListener('click', function () {
            const originalButtonHTML = this.innerHTML; this.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i>'; this.disabled = true;
            loadHtml2Canvas(() => {
                const downloadId = this.dataset.downloadId; const postToDownload = document.getElementById(`download-post-${downloadId}`);
                if (!postToDownload) { console.error('Downloadable post element not found for ID:', downloadId); this.innerHTML = originalButtonHTML; this.disabled = false; return; }
                const hotelName = postToDownload.querySelector('h1')?.textContent || 'hotel-post'; const sanitizedFileName = hotelName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
                document.body.appendChild(postToDownload); postToDownload.style.cssText = 'position: absolute; left: -9999px; top: 0; display: block;';
                html2canvas(postToDownload, { useCORS: true, scale: 1.5, logging: false, removeContainer: true }).then(canvas => {
                    const link = document.createElement('a'); link.href = canvas.toDataURL('image/jpeg', 0.9); link.download = `${sanitizedFileName}.jpg`; document.body.appendChild(link); link.click(); document.body.removeChild(link);
                }).catch(err => { console.error('Download failed!', err); alert("Download Failed. Ensure all images are uploaded to your server.");
                }).finally(() => { if (document.body.contains(postToDownload)) { postToDownload.style.display = 'none'; } this.innerHTML = originalButtonHTML; this.disabled = false; });
            });
        });
    });
});
</script>