<?php
// This is the main page for displaying visa services.
// It connects to the database, fetches data based on filters, and then displays it.

// Include your database configuration and any session handling
include_once 'db-config.php';

// =======================================================
//  1. SETUP & HELPER FUNCTIONS
// =======================================================

// A helper function to prevent XSS attacks.
if (!function_exists('safe_htmlspecialchars')) {
    function safe_htmlspecialchars($string) {
        return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
    }
}
// =======================================================
//  2. FETCH DATA FOR THE PAGE
// =======================================================

// --- Get the selected city from the URL ---
$selected_city = isset($_GET['city']) ? urldecode($_GET['city']) : 'all';

// --- Fetch all available cities for the filter navigation ---
$cities_result = $conn->query("SELECT * FROM cities ORDER BY name ASC");

// --- Construct the main query to fetch visa services based on the filter ---
$sql = "";
$params = [];
$types = '';

if ($selected_city !== 'all' && !empty($selected_city)) {
    // QUERY FOR A SPECIFIC CITY
    $sql = "SELECT DISTINCT v.* FROM visas v
              INNER JOIN visa_cities vc ON v.id = vc.visa_id
              INNER JOIN cities c ON vc.city_id = c.id
              WHERE v.is_active = 1 AND c.name = ?
              ORDER BY v.id DESC";
    $params[] = $selected_city;
    $types .= 's';
} else {
    // QUERY FOR "SHOW ALL" - Get all visas and their assigned city for grouping
    $sql = "SELECT c.name as city_name, v.*
            FROM visas v
            JOIN visa_cities vc ON v.id = vc.visa_id
            JOIN cities c ON vc.city_id = c.id
            WHERE v.is_active = 1
            ORDER BY c.name ASC, v.id DESC";
}


// Prepare and execute the statement securely
$stmt = $conn->prepare($sql);
if ($stmt === false) {
    die("Error preparing statement: " . $conn->error);
}
if (!empty($types)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$visa_services_result = $stmt->get_result();

// --- Placeholder for branding & user status variables ---
$footer_logo = 'images/logo.png'; 
$logo_style_class = 'white-bg-logo';
$footer_address = '123 Visa St, Main City, Country';
$footer_phone_1 = '+92 123 4567890';
$is_agent = isset($_SESSION['user_type']) && $_SESSION['user_type'] === 'agent';

?>

<!-- 
    The rest of this file is the HTML and display logic.
    You might have a header file included before this section.
    e.g., <?php // include 'header.php'; ?>
-->

<section class="visa-listings-section-fullwidth">
    <div class="listings-container-fullwidth">
        <div class="listings-header">
            <h2 class="listings-title">All Our Visa Services</h2>
            <p class="listings-subtitle">Browse our complete collection of available visa services.</p>
        </div>

        <!-- City Filter Navigation -->
        <?php if ($cities_result && $cities_result->num_rows > 0): ?>
        <nav class="city-filter-nav">
            <a href="visa-services.php" class="filter-btn <?= !$selected_city || $selected_city === 'all' ? 'active' : '' ?>">Show All</a>
            <?php 
                $cities_result->data_seek(0); 
                while($city = $cities_result->fetch_assoc()): 
            ?>
                <a href="visa-services.php?city=<?= urlencode($city['name']) ?>" class="filter-btn <?= ($selected_city === $city['name']) ? 'active' : '' ?>">
                    <?= safe_htmlspecialchars($city['name']) ?>
                </a>
            <?php endwhile; ?>
        </nav>
        <?php endif; ?>
        
        <div class="visa-listings-container">
            <?php if ($visa_services_result && $visa_services_result->num_rows > 0): ?>
                
                <?php
                // ===========================================================
                //  DISPLAY LOGIC: BASED ON "SHOW ALL" OR "SPECIFIC CITY"
                // ===========================================================
                ?>

                <?php if ($selected_city === 'all'): ?>
                    <?php
                    // First, group all visas by their city name
                    $visas_by_city = [];
                    while ($row = $visa_services_result->fetch_assoc()) {
                        $visas_by_city[$row['city_name']][] = $row;
                    }
                    ?>

                    <?php foreach ($visas_by_city as $city_name => $visas): ?>
                        <div class="city-group">
                            <h2 class="city-group-title"><?= safe_htmlspecialchars($city_name) ?></h2>
                            <div class="visa-grid">
                                <?php foreach ($visas as $visa): ?>
                                    <?php
                                    // Pricing Logic for each visa
                                    $sar_price_to_show = null;
                                    if ($is_agent && !empty($visa['price_agent'])) { $sar_price_to_show = $visa['price_agent']; } 
                                    elseif (!empty($visa['price'])) { $sar_price_to_show = $visa['price']; }
                                    $pkr_price_to_show = null;
                                    if ($sar_price_to_show && !empty($visa['exchange_rate'])) { $pkr_price_to_show = $sar_price_to_show * $visa['exchange_rate']; }
                                    ?>
                                    <!-- VISIBLE CARD -->
                                    <div class="visa-card-v2">
                                        <div class="card-v2-banner">
                                            <div class="card-v2-logo-overlay <?= $logo_style_class ?>"><img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous"></div>
                                            <?php if (!empty($visa['destination_city'])): ?><div class="card-v2-city-badge"><?= safe_htmlspecialchars($visa['destination_city']) ?></div><?php endif; ?>
                                            <img src="<?= safe_htmlspecialchars($visa['image_url'] ?: 'images/default-visa.jpg') ?>" alt="<?= safe_htmlspecialchars($visa['image_alt']) ?>" class="card-v2-image" loading="lazy" crossorigin="anonymous">
                                        </div>
                                        <div class="card-v2-content">
                                            <h3 class="visa-name-v2"><?= safe_htmlspecialchars($visa['visa_name']) ?></h3>
                                            <p class="visa-type-v2"><?= safe_htmlspecialchars($visa['visa_type']) ?></p>
                                            <div class="visa-details-grid">
                                                <?php if (!empty($visa['processing_time'])): ?><div class="detail-item-v2"><i class="fas fa-hourglass-start"></i> <span><?= safe_htmlspecialchars($visa['processing_time']) ?></span></div><?php endif; ?>
                                                <?php if (!empty($visa['entry_type'])): ?><div class="detail-item-v2"><i class="fas fa-right-to-bracket"></i> <span><?= safe_htmlspecialchars($visa['entry_type']) ?></span></div><?php endif; ?>
                                            </div>
                                            <div class="visa-price-v2">
                                                <?php if ($pkr_price_to_show): ?><span><?= safe_htmlspecialchars($visa['price_note'] ?: 'Starts From') ?></span><strong>PKR <?= number_format($pkr_price_to_show, 0) ?></strong><?php else: ?><span>Price</span><strong>On Request</strong><?php endif; ?>
                                            </div>
                                        </div>
                                        <div class="card-v2-actions">
                                            <?php $visa_page_url = !empty($visa['page_link']) ? safe_htmlspecialchars($visa['page_link']) . '.php' : '#'; ?>
                                            <a href="<?= $visa_page_url ?>" class="btn-view-deal">View Details</a>
                                            <button class="btn-download-visa-card" data-visa-id="<?= $visa['id'] ?>" data-visa-name="<?= safe_htmlspecialchars($visa['visa_name']) ?>"><i class="fa-solid fa-download"></i> Download JPG</button>
                                        </div>
                                    </div>
                                    <!-- HIDDEN DOWNLOADABLE POST -->
                                    <div class="downloadable-visa-post-v2" id="download-post-<?= $visa['id'] ?>"><img src="<?= safe_htmlspecialchars($visa['image_url'] ?: 'images/default-visa.jpg') ?>" class="post-v2-bg-image" crossorigin="anonymous" alt=""><div class="post-v2-overlay"><header class="post-v2-header"><div class="post-v2-logo"><img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous"></div></header><main class="post-v2-body"><div class="post-v2-title"><h1 class="post-v2-title-main"><?= safe_htmlspecialchars($visa['visa_name']) ?></h1><h2 class="post-v2-title-sub"><?= safe_htmlspecialchars($visa['visa_type']) ?></h2></div><div class="post-v2-details-box"><div class="post-v2-detail-item"><i class="fas fa-map-marker-alt"></i><h3>Destination</h3><p><?= safe_htmlspecialchars($visa['destination_city'] ?: 'N/A') ?></p></div><div class="post-v2-detail-item"><i class="fas fa-hourglass-half"></i><h3>Processing</h3><p><?= safe_htmlspecialchars($visa['processing_time'] ?: 'N/A') ?></p></div><div class="post-v2-detail-item"><i class="fas fa-tags"></i><h3>Price</h3><p><?= $pkr_price_to_show ? 'PKR ' . number_format($pkr_price_to_show, 0) : 'On Request' ?></p></div></div></main><footer class="post-v2-footer"><span><i class="fas fa-map-marker-alt"></i> <?= safe_htmlspecialchars($footer_address) ?></span><span><i class="fab fa-whatsapp"></i> <?= safe_htmlspecialchars($footer_phone_1) ?></span></footer></div></div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>

                <?php else: // This is the view for a SPECIFIC selected city ?>
                    
                    <div class="visa-grid">
                        <?php while ($visa = $visa_services_result->fetch_assoc()): ?>
                            <?php
                            // Pricing Logic for each visa
                            $sar_price_to_show = null;
                            if ($is_agent && !empty($visa['price_agent'])) { $sar_price_to_show = $visa['price_agent']; } 
                            elseif (!empty($visa['price'])) { $sar_price_to_show = $visa['price']; }
                            $pkr_price_to_show = null;
                            if ($sar_price_to_show && !empty($visa['exchange_rate'])) { $pkr_price_to_show = $sar_price_to_show * $visa['exchange_rate']; }
                            ?>
                            <!-- VISIBLE CARD -->
                            <div class="visa-card-v2">
                                <div class="card-v2-banner">
                                    <div class="card-v2-logo-overlay <?= $logo_style_class ?>"><img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous"></div>
                                    <?php if (!empty($visa['destination_city'])): ?><div class="card-v2-city-badge"><?= safe_htmlspecialchars($visa['destination_city']) ?></div><?php endif; ?>
                                    <img src="<?= safe_htmlspecialchars($visa['image_url'] ?: 'images/default-visa.jpg') ?>" alt="<?= safe_htmlspecialchars($visa['image_alt']) ?>" class="card-v2-image" loading="lazy" crossorigin="anonymous">
                                </div>
                                <div class="card-v2-content">
                                    <h3 class="visa-name-v2"><?= safe_htmlspecialchars($visa['visa_name']) ?></h3>
                                    <p class="visa-type-v2"><?= safe_htmlspecialchars($visa['visa_type']) ?></p>
                                    <div class="visa-details-grid">
                                        <?php if (!empty($visa['processing_time'])): ?><div class="detail-item-v2"><i class="fas fa-hourglass-start"></i> <span><?= safe_htmlspecialchars($visa['processing_time']) ?></span></div><?php endif; ?>
                                        <?php if (!empty($visa['entry_type'])): ?><div class="detail-item-v2"><i class="fas fa-right-to-bracket"></i> <span><?= safe_htmlspecialchars($visa['entry_type']) ?></span></div><?php endif; ?>
                                    </div>
                                    <div class="visa-price-v2">
                                        <?php if ($pkr_price_to_show): ?><span><?= safe_htmlspecialchars($visa['price_note'] ?: 'Starts From') ?></span><strong>PKR <?= number_format($pkr_price_to_show, 0) ?></strong><?php else: ?><span>Price</span><strong>On Request</strong><?php endif; ?>
                                    </div>
                                </div>
                                <div class="card-v2-actions">
                                    <?php $visa_page_url = !empty($visa['page_link']) ? safe_htmlspecialchars($visa['page_link']) . '.php' : '#'; ?>
                                    <a href="<?= $visa_page_url ?>" class="btn-view-deal">View Details</a>
                                    <button class="btn-download-visa-card" data-visa-id="<?= $visa['id'] ?>" data-visa-name="<?= safe_htmlspecialchars($visa['visa_name']) ?>"><i class="fa-solid fa-download"></i> Download JPG</button>
                                </div>
                            </div>
                            <!-- HIDDEN DOWNLOADABLE POST -->
                            <div class="downloadable-visa-post-v2" id="download-post-<?= $visa['id'] ?>"><img src="<?= safe_htmlspecialchars($visa['image_url'] ?: 'images/default-visa.jpg') ?>" class="post-v2-bg-image" crossorigin="anonymous" alt=""><div class="post-v2-overlay"><header class="post-v2-header"><div class="post-v2-logo"><img src="<?= safe_htmlspecialchars($footer_logo) ?>" alt="Agency Logo" crossorigin="anonymous"></div></header><main class="post-v2-body"><div class="post-v2-title"><h1 class="post-v2-title-main"><?= safe_htmlspecialchars($visa['visa_name']) ?></h1><h2 class="post-v2-title-sub"><?= safe_htmlspecialchars($visa['visa_type']) ?></h2></div><div class="post-v2-details-box"><div class="post-v2-detail-item"><i class="fas fa-map-marker-alt"></i><h3>Destination</h3><p><?= safe_htmlspecialchars($visa['destination_city'] ?: 'N/A') ?></p></div><div class="post-v2-detail-item"><i class="fas fa-hourglass-half"></i><h3>Processing</h3><p><?= safe_htmlspecialchars($visa['processing_time'] ?: 'N/A') ?></p></div><div class="post-v2-detail-item"><i class="fas fa-tags"></i><h3>Price</h3><p><?= $pkr_price_to_show ? 'PKR ' . number_format($pkr_price_to_show, 0) : 'On Request' ?></p></div></div></main><footer class="post-v2-footer"><span><i class="fas fa-map-marker-alt"></i> <?= safe_htmlspecialchars($footer_address) ?></span><span><i class="fab fa-whatsapp"></i> <?= safe_htmlspecialchars($footer_phone_1) ?></span></footer></div></div>
                        <?php endwhile; ?>
                    </div>

                <?php endif; ?>

            <?php else: ?>
                <p style="text-align: center; grid-column: 1 / -1; padding: 40px; background: #fff; border-radius: 8px;">No visa services found for the selected city. Please check back later or view all services.</p>
            <?php endif; ?>
        </div>
    </div>
</section>

<!-- STYLES AND SCRIPT FOR THIS COMPONENT -->
<style>
    /* ====================================================================== */
    /* === VISA CARDS & DOWNLOADABLE POST STYLES V6 (GROUPED LIST) === */
    /* ====================================================================== */
    
    .visa-listings-section-fullwidth { padding: 60px 0; background-color: #f8f9fa; }
    .listings-container-fullwidth { max-width: 1700px; margin: 0 auto; padding: 0 40px; box-sizing: border-box; }
    .listings-header { text-align: center; margin-bottom: 30px; }
    .listings-title { font-size: 2.8rem; font-weight: 700; color: #2c3e50; margin-bottom: 10px; }
    .listings-subtitle { font-size: 1.1rem; color: #7f8c8d; }
    .city-filter-nav { display: flex; flex-wrap: wrap; justify-content: center; gap: 10px; margin-bottom: 40px; }
    .filter-btn { padding: 10px 20px; background-color: #fff; color: #34495e; border: 1px solid #ddd; border-radius: 30px; text-decoration: none; font-weight: 500; transition: all 0.3s ease; width: 17ch; text-align: center; box-sizing: border-box; display: inline-block; }
    .filter-btn:hover { background-color: #f0f0f0; border-color: #ccc; }
    .filter-btn.active { background-color: #6c757d; color: #fff; border-color: #6c757d; }
    
    /* --- NEW STYLES FOR CITY GROUPING --- */
    .city-group { margin-bottom: 50px; }
    .city-group:last-of-type { margin-bottom: 0; }
    .city-group-title {
        font-size: 2.2rem;
        font-weight: 600;
        color: #34495e;
        margin-bottom: 25px;
        padding-bottom: 15px;
        border-bottom: 2px solid #e0e0e0;
    }

    .visa-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 30px; }
    .visa-card-v2 { background: #fff; border-radius: 12px; box-shadow: 0 8px 25px rgba(0,0,0,0.08); overflow: hidden; display: flex; flex-direction: column; transition: transform 0.3s, box-shadow 0.3s; }
    .visa-card-v2:hover { transform: translateY(-8px); box-shadow: 0 12px 35px rgba(0,0,0,0.12); }
    .card-v2-banner { position: relative; }
    .card-v2-image { display: block; width: 100%; height: 220px; object-fit: cover; }
    .card-v2-logo-overlay { position: absolute; top: 15px; left: 15px; width: 50px; height: 50px; display: flex; align-items: center; justify-content: center; }
    .card-v2-logo-overlay.white-bg-logo { background: rgba(255,255,255,0.85); backdrop-filter: blur(5px); border-radius: 50%; padding: 6px; box-shadow: 0 1px 5px rgba(0,0,0,0.15); }
    .card-v2-logo-overlay img { max-width: 100%; max-height: 100%; object-fit: contain; }
    .card-v2-city-badge { position: absolute; bottom: 15px; right: 15px; background: var(--primary-color); color: #fff; padding: 5px 12px; border-radius: 20px; font-size: 0.8rem; font-weight: 600; box-shadow: 0 2px 6px rgba(0,0,0,0.25); }
    .card-v2-content { padding: 1.25rem; display: flex; flex-direction: column; flex-grow: 1; }
    .visa-name-v2 { font-size: 1.4rem; font-weight: 600; color: var(--text-dark); margin: 0 0 0.25rem 0; }
    .visa-type-v2 { font-size: 1rem; color: #7f8c8d; margin-bottom: 1rem; }
    .visa-details-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 1rem; font-size: 0.9rem; color: #555; }
    .detail-item-v2 { display: flex; align-items: center; gap: 8px; }
    .detail-item-v2 i { color: var(--primary-gold); }
    .visa-price-v2 { text-align: right; margin-top: auto; padding-top: 1rem; border-top: 1px solid #f0f0f1; }
    .visa-price-v2 span { font-size: 0.9rem; color: #777; display: block; margin-bottom: 2px; }
    .visa-price-v2 strong { font-size: 1.6rem; font-weight: 700; color: var(--primary-color); }
    .card-v2-actions { display: grid; grid-template-columns: 1fr 1fr; gap: 0.75rem; padding: 1rem; background-color: #f8f9fa; }
    .btn-download-visa-card { width: 100%; padding: 12px; font-size: 0.9rem; font-weight: 600; text-align: center; border-radius: 8px; cursor: pointer; transition: background-color 0.3s, color 0.3s; background-color: #e9ecef; color: #333; border: 1px solid #ddd; }
    .btn-download-visa-card:hover:not(:disabled) { background-color: #dee2e6; }
    .btn-download-visa-card:disabled { cursor: not-allowed; opacity: 0.7; } .btn-download-visa-card i { margin-right: 8px; }
    
    .downloadable-visa-post-v2 { display: none; width: 1080px; height: 1080px; font-family: 'Poppins', sans-serif; position: relative; color: #fff; }
    .post-v2-bg-image { position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; z-index: 1; }
    .post-v2-overlay { position: relative; z-index: 2; width: 100%; height: 100%; padding: 60px; display: flex; flex-direction: column; justify-content: space-between; background: linear-gradient(to top, rgba(25, 35, 47, 0.98) 0%, rgba(25, 35, 47, 0.9) 50%, rgba(25, 35, 47, 0.4) 100%); }
    .post-v2-logo { width: 140px; height: 140px; background-color: rgba(255, 255, 255, 0.95); border-radius: 50%; display: flex; align-items: center; justify-content: center; padding: 15px; box-shadow: 0 4px 15px rgba(0,0,0,0.3); }
    .post-v2-logo img { max-width: 100%; max-height: 100%; }
    .post-v2-body { text-align: center; flex-grow: 1; display: flex; flex-direction: column; justify-content: center; }
    .post-v2-title-main { font-size: 110px; font-weight: 800; line-height: 1.1; margin: 0; text-transform: uppercase; text-shadow: 2px 4px 8px rgba(0,0,0,0.7); }
    .post-v2-title-sub { font-size: 45px; font-weight: 400; line-height: 1.3; margin: 10px 0 50px 0; opacity: 0.9; text-shadow: 1px 2px 5px rgba(0,0,0,0.7); }
    .post-v2-details-box { background-color: rgba(18, 26, 35, 0.85); backdrop-filter: blur(10px); -webkit-backdrop-filter: blur(10px); border-radius: 20px; padding: 30px 50px; display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; max-width: 800px; margin: 0 auto; border: 1px solid rgba(255,255,255,0.1); }
    .post-v2-detail-item { text-align: center; }
    .post-v2-detail-item i { font-size: 36px; color: #3498db; margin-bottom: 15px; }
    .post-v2-detail-item h3 { font-size: 22px; color: #a0a0a0; font-weight: 500; margin: 0 0 8px 0; }
    .post-v2-detail-item p { font-size: 32px; font-weight: 600; margin: 0; }
    .post-v2-footer { background-color: #3498db; border-radius: 20px; padding: 25px 40px; display: flex; justify-content: space-between; align-items: center; font-size: 24px; font-weight: 500; }
    .post-v2-footer i { margin-right: 15px; font-size: 28px; vertical-align: middle; }
    
    @media (max-width: 1024px) { .visa-grid { grid-template-columns: repeat(2, 1fr); } }
    @media (max-width: 768px) {
        .visa-grid { grid-template-columns: 1fr; }
        .listings-container-fullwidth { padding: 0 20px; }
        .city-group-title { font-size: 1.8rem; }
        .city-filter-nav { display: flex; flex-wrap: nowrap; overflow-x: auto; justify-content: flex-start; -webkit-overflow-scrolling: touch; padding: 5px 5px 15px 5px; margin-left: -20px; margin-right: -20px; scrollbar-width: thin; scrollbar-color: #cccccc #f8f9fa; }
        .city-filter-nav::-webkit-scrollbar { height: 6px; }
        .city-filter-nav::-webkit-scrollbar-track { background: #f8f9fa; }
        .city-filter-nav::-webkit-scrollbar-thumb { background-color: #cccccc; border-radius: 10px; border: 2px solid #f8f9fa; }
        .filter-btn { flex-shrink: 0; }
    }
</style>

<!-- This script assumes you have loaded the html2canvas library on your page -->
<!-- e.g., <script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script> -->
<script>
    document.addEventListener('DOMContentLoaded', function() {
        let html2canvasLoaded = typeof html2canvas !== 'undefined';
        document.querySelectorAll('.btn-download-visa-card').forEach(button => {
            button.addEventListener('click', function () {
                const originalButtonHTML = this.innerHTML;
                this.innerHTML = '<i class="fa-solid fa-spinner fa-spin"></i> Please Wait';
                this.disabled = true;
                if (!html2canvasLoaded) {
                    console.error("html2canvas script not loaded.");
                    alert("Download functionality is currently unavailable. Please contact support.");
                    this.innerHTML = originalButtonHTML; this.disabled = false; return;
                }
                const visaId = this.dataset.visaId;
                const postToDownload = document.getElementById(`download-post-${visaId}`);
                if (!postToDownload) {
                    console.error('Downloadable post element not found for visa ID:', visaId);
                    this.innerHTML = originalButtonHTML; this.disabled = false; return;
                }
                const visaName = this.dataset.visaName || 'visa-service';
                const sanitizedFileName = visaName.replace(/[^a-z0-9]/gi, '_').toLowerCase();
                document.body.appendChild(postToDownload);
                postToDownload.style.cssText = 'position: absolute; left: -9999px; top: 0; display: block;';
                html2canvas(postToDownload, { useCORS: true, scale: 1.5, logging: false, removeContainer: true })
                .then(canvas => {
                    const link = document.createElement('a');
                    link.href = canvas.toDataURL('image/jpeg', 0.9);
                    link.download = `${sanitizedFileName}.jpg`;
                    document.body.appendChild(link);
                    link.click();
                    document.body.removeChild(link);
                }).catch(err => {
                    console.error('Download failed!', err);
                    alert("Download Failed. This can happen if an image is linked from another website. Please ensure all visa images are uploaded to your own server.");
                }).finally(() => {
                    if (document.body.contains(postToDownload)) {
                        postToDownload.style.display = 'none';
                    }
                    this.innerHTML = originalButtonHTML; this.disabled = false;
                });
            });
        });
    });
</script>

<!-- 
    You might have a footer file included after this section.
    e.g., <?php // include 'footer.php'; ?>
-->