<?php
session_start();
// --- DEFINITIVE FIX: Corrected the path to the database configuration file for the root directory ---
include_once 'db-config.php';

// --- SECURITY CHECK ---
// Allow any logged-in user (admin, agent, customer) to view their own/related vouchers.
if (!isset($_SESSION['user_id'])) {
    header("location: login.php");
    exit;
}

function e($string)
{
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

$booking_id = (int)($_GET['booking_id'] ?? 0);
if ($booking_id <= 0) {
    die("Error: Invalid booking reference provided.");
}

// --- FETCH CORE BOOKING DATA ---
$sql_booking = "SELECT b.*, u.name as user_name, ti.id as ticket_invoice_id
                FROM bookings b
                JOIN users u ON b.user_id = u.id
                LEFT JOIN ticket_invoices ti ON b.id = ti.booking_id
                WHERE b.id = ?";

// Add security layer for non-admins to prevent them from viewing other users' bookings
if ($_SESSION['user_type'] !== 'admin') {
    $sql_booking .= " AND b.user_id = ?";
}

$stmt_booking = $conn->prepare($sql_booking);
if ($_SESSION['user_type'] !== 'admin') {
    $stmt_booking->bind_param("ii", $booking_id, $_SESSION['user_id']);
} else {
    $stmt_booking->bind_param("i", $booking_id);
}
$stmt_booking->execute();
$booking = $stmt_booking->get_result()->fetch_assoc();
$stmt_booking->close();

if (!$booking) {
    die("Booking not found or you do not have permission to view it.");
}

// --- DATA INITIALIZATION ---
$airline_name = "N/A";
$airline_logo = "https://daisycon.io/images/airline/?width=150&height=150&iata=PK"; // Default
$departure_date_display = "N/A";
$passengers = [];
$itineraries = [];
$booking_status_text = 'Reserved'; // Default
$baggage_allowance = 'N/A';
$meal_service = 'N/A';
$travel_class = 'Economy'; // Default

// --- DYNAMIC BOOKING STATUS ---
if (!empty($booking['status'])) {
    switch (strtolower($booking['status'])) {
        case 'confirmed':
            $booking_status_text = 'Ticketed';
            break;
        case 'cancelled':
            $booking_status_text = 'Cancelled';
            break;
        case 'pending':
        default:
            $booking_status_text = 'Reserved';
            break;
    }
}

// --- LOGIC FOR GROUP BOOKINGS ---
if ($booking['booking_type'] === 'group' && !empty($booking['ticket_invoice_id'])) {
    $flight_details = json_decode($booking['flight_details'], true) ?: [];
    $airline_name = $flight_details['airline_name'] ?? 'N/A';

    if (isset($flight_details['flight_details_json']['baggage'])) {
        $baggage_allowance = $flight_details['flight_details_json']['baggage'];
    }
    if (!empty($flight_details['logo_url'])) {
        $airline_logo = (strpos($flight_details['logo_url'], '//') === 0) ? 'https:' . $flight_details['logo_url'] : $flight_details['logo_url'];
    }

    $stmt_pax = $conn->prepare("SELECT * FROM ticket_invoice_passengers WHERE ticket_invoice_id = ?");
    $stmt_pax->bind_param("i", $booking['ticket_invoice_id']);
    $stmt_pax->execute();
    $pax_result = $stmt_pax->get_result()->fetch_all(MYSQLI_ASSOC);
    foreach ($pax_result as $p) {
        $pax_status = 'PENDING';
        if (strtolower($booking['status']) === 'confirmed') $pax_status = 'CONFIRMED';
        if (strtolower($booking['status']) === 'cancelled') $pax_status = 'CANCELLED';
        $passengers[] = ['name' => $p['full_name'], 'passport' => $p['passport_no'], 'pnr' => $p['pnr'], 'ticket_number' => $p['ticket_number'], 'status' => $pax_status];
    }

    $stmt_flights = $conn->prepare("SELECT * FROM ticket_invoice_flights WHERE ticket_invoice_id = ? ORDER BY departure_datetime ASC");
    $stmt_flights->bind_param("i", $booking['ticket_invoice_id']);
    $stmt_flights->execute();
    $flights_result = $stmt_flights->get_result()->fetch_all(MYSQLI_ASSOC);

    foreach ($flights_result as $flight_leg) {
        $itineraries[]['segments'][] = $flight_leg;
    }
    if (!empty($flights_result[0])) {
        $departure_date_display = date('D, d M', strtotime($flights_result[0]['departure_datetime']));
    }

    // --- LOGIC FOR STANDARD FLIGHT BOOKINGS ---
} elseif ($booking['booking_type'] === 'flight') {
    $flight_details = json_decode($booking['flight_details'], true) ?: [];
    $passenger_details = json_decode($booking['passenger_details'], true) ?: [];

    $baggage_allowance = $flight_details['baggage'] ?? 'Not Specified';
    $meal_service = $flight_details['meal'] ?? 'Not Included';

    if (isset($flight_details['itineraries'][0])) {
        $first_itinerary = $flight_details['itineraries'][0];
        $airline_name = $first_itinerary['airlineName'] ?? 'N/A';
        $airline_logo_path = $first_itinerary['airlineLogo'] ?? '';
        $airline_logo = str_starts_with($airline_logo_path, '//') ? 'https:' . $airline_logo_path : $airline_logo_path;
        $departure_date_display = $first_itinerary['departureDate'] ?? 'N/A';
    }
    $itineraries = $flight_details['itineraries'] ?? [];

    // === THE FIX: Fetch PNR/Ticket data if an invoice exists for this flight booking ===
    $ticketed_passengers_data = [];
    if (!empty($booking['ticket_invoice_id'])) {
        $stmt_pax_ticketed = $conn->prepare("SELECT full_name, pnr, ticket_number FROM ticket_invoice_passengers WHERE ticket_invoice_id = ?");
        $stmt_pax_ticketed->bind_param("i", $booking['ticket_invoice_id']);
        $stmt_pax_ticketed->execute();
        $pax_result = $stmt_pax_ticketed->get_result();
        while ($row = $pax_result->fetch_assoc()) {
            // Key by a normalized full name to handle potential spacing/case issues
            $normalized_name = strtolower(trim(preg_replace('/\s+/', ' ', $row['full_name'])));
            $ticketed_passengers_data[$normalized_name] = [
                'pnr' => $row['pnr'],
                'ticket_number' => $row['ticket_number']
            ];
        }
        $stmt_pax_ticketed->close();
    }
    // ==============================================================================

    if (isset($passenger_details)) {
        foreach ($passenger_details as $type => $pax_group) {
            if (empty($pax_group) || !is_array($pax_group)) continue;
            foreach ($pax_group as $p) {
                $full_name = trim(($p['title'] ?? '') . ' ' . ($p['firstName'] ?? '') . ' ' . ($p['lastName'] ?? ''));
                $normalized_name_check = strtolower(trim(preg_replace('/\s+/', ' ', $full_name)));

                // Get PNR and Ticket Number from ticketed data first, then fall back to the booking JSON
                $pnr = $ticketed_passengers_data[$normalized_name_check]['pnr'] ?? ($p['pnr'] ?? '');
                $ticket_number = $ticketed_passengers_data[$normalized_name_check]['ticket_number'] ?? ($p['ticket_number'] ?? '');

                $pax_status = 'PENDING';
                if (strtolower($booking['status']) === 'confirmed') $pax_status = 'CONFIRMED';
                if (strtolower($booking['status']) === 'cancelled') $pax_status = 'CANCELLED';

                $passengers[] = ['name' => $full_name, 'passport' => $p['passport'] ?? '', 'pnr' => $pnr, 'ticket_number' => $ticket_number, 'status' => $pax_status];
            }
        }
    }
}

// Helper function to render flight segments
function renderFlightSegment($segment, $booking_type, $airline_name_fallback, $baggage_allowance, $meal_service, $travel_class)
{
    $dep_iata = $arr_iata = $flight_no = $dep_date = $arr_date = $dep_time = $arr_time = $duration = 'N/A';
    $airline_name = $airline_name_fallback;

    if ($booking_type === 'group' && is_array($segment) && !empty($segment['departure_datetime'])) {
        $departure_datetime = strtotime($segment['departure_datetime']);
        $arrival_datetime = strtotime($segment['arrival_datetime']);
        $dep_date = date('l, d M Y', $departure_datetime);
        $arr_date = date('l, d M Y', $arrival_datetime);
        $dep_time = date('H:i', $departure_datetime);
        $arr_time = date('H:i', $arrival_datetime);
        list($dep_iata, $arr_iata) = explode('-', $segment['sector'] ?? 'N/A-N/A');
        $flight_no = e($segment['flight_no']);
        $duration = ($arrival_datetime > $departure_datetime) ? gmdate("H\h i\m", $arrival_datetime - $departure_datetime) : 'N/A';
        $airline_name = e($segment['airline']);
    } elseif ($booking_type === 'flight' && is_array($segment)) {
        $dep_date = e($segment['departure']['date']);
        $arr_date = e($segment['arrival']['date']);
        $dep_time = e($segment['departure']['time']);
        $arr_time = e($segment['arrival']['time']);
        $dep_iata = e($segment['departure']['iata']);
        $arr_iata = e($segment['arrival']['iata']);
        $flight_no = e($segment['flightNumber']);
        $duration = e($segment['duration']);
        $airline_name = e($segment['airlineName']);
    }

    $meal_icon = (strtolower($meal_service) === 'included' || strtolower($meal_service) === 'yes') ? 'fas fa-utensils' : 'fas fa-utensils-slash';
    $meal_text = (strtolower($meal_service) === 'included' || strtolower($meal_service) === 'yes') ? 'Yes' : 'No';

    return <<<HTML
    <div class="flight-leg">
        <div class="leg-header"><h4><i class="fas fa-plane"></i> Departure from {$dep_iata} (Flight {$flight_no})</h4></div>
        <div class="leg-body">
            <div class="times-airports">
                <div class="time-block"> <span class="day-date">{$dep_date}</span> <span class="time">{$dep_time}</span> <span class="iata">{$dep_iata}</span> </div>
                 <div class="duration-line"><i class="fas fa-plane"></i> <small>{$duration}</small></div>
                <div class="time-block"> <span class="day-date">{$arr_date}</span> <span class="time">{$arr_time}</span> <span class="iata">{$arr_iata}</span> </div>
            </div>
            <div class="class-amenities">
                <h4>{$travel_class}</h4>
                <span><i class="fas fa-suitcase"></i> {$baggage_allowance}</span>
                <span><i class="{$meal_icon}"></i> {$meal_text}</span>
                <span><i class="fas fa-chair"></i> Unassigned</span>
                <span><i class="fas fa-shopping-cart"></i> Buy on board</span>
            </div>
        </div>
    </div>
HTML;
}
?>
<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <title>E-Ticket Voucher #<?= e($booking['booking_ref']) ?></title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">

    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Roboto', sans-serif;
            background-color: #f0f2f5;
            margin: 0;
            padding: 10px;
            color: #333;
        }

        .no-print {
            text-align: right;
            max-width: 850px;
            margin: 0 auto 15px auto;
        }

        .btn-print {
            background-color: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
        }

        .voucher-container {
            background-color: #fff;
            max-width: 800px;
            margin: auto;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            border-bottom: 2px solid #e9ecef;
            padding-bottom: 20px;
        }

        .header-left .logo img {
            max-height: 100px;
        }

        .header-left .title {
            font-size: 24px;
            font-weight: 700;
            color: #444;
            letter-spacing: -1px;
            margin-top: 5px;
        }

        .header-right .logo img {
            max-height: 50px;
        }

        .booking-status {
            margin-top: 20px;
            font-size: 14px;
            color: #555;
        }

        .booking-status strong {
            font-weight: 700;
        }

        .status-ticketed {
            color: #28a745;
        }

        .status-reserved {
            color: #fd7e14;
        }

        .status-cancelled {
            color: #dc3545;
        }

        .passengers-section {
            margin-top: 20px;
        }

        .passengers-section-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-end;
            margin-bottom: 10px;
        }

        .passengers-section-header h4 {
            margin: 0;
            font-size: 16px;
        }

        .passengers-section-header .departure-date {
            color: #007bff;
            font-weight: 700;
        }

        .passengers-section table {
            width: 100%;
            border-collapse: collapse;
        }

        .passengers-section th,
        .passengers-section td {
            padding: 10px 12px;
            border: 1px solid #dee2e6;
            text-align: left;
        }

        .passengers-section th {
            background-color: #008770;
            color: #fff;
            font-weight: 500;
            font-size: 14px;
        }

        .flight-leg {
            border: 1px solid #e9ecef;
            border-radius: 6px;
            margin-top: 20px;
            overflow: hidden;
        }

        .leg-header {
            background-color: #008770;
            color: #fff;
            padding: 10px 15px;
            font-size: 16px;
        }

        .leg-header h4 {
            margin: 0;
        }

        .leg-body {
            display: flex;
            justify-content: space-between;
            align-items: stretch;
        }

        .times-airports {
            flex: 3;
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 20px;
        }

        .time-block {
            text-align: center;
        }

        .time-block .day-date {
            display: block;
            font-weight: 500;
            font-size: 14px;
            color: #666;
        }

        .time-block .time {
            display: block;
            font-weight: 700;
            font-size: 22px;
            color: #000;
            margin: 4px 0;
        }

        .time-block .iata {
            display: block;
            font-size: 18px;
            font-weight: 700;
            color: #444;
        }

        .duration-line {
            text-align: center;
            color: #888;
            flex-grow: 1;
            padding: 0 1rem;
        }

        .duration-line i {
            font-size: 1.2rem;
        }

        .class-amenities {
            flex: 1;
            background-color: #f8f9fa;
            border-left: 1px solid #e9ecef;
            padding: 15px;
            display: flex;
            flex-direction: column;
            justify-content: center;
        }

        .class-amenities h4 {
            margin: 0 0 10px 0;
            font-size: 14px;
            text-align: center;
            color: #000;
        }

        .class-amenities span {
            display: block;
            margin-bottom: 8px;
            font-size: 13px;
            color: #555;
        }

        .class-amenities i {
            width: 20px;
            color: #888;
            text-align: center;
            margin-right: 5px;
        }

        .emergency-contact {
            margin-top: 25px;
            border-top: 2px solid #e9ecef;
            padding-top: 20px;
            text-align: center;
            font-size: 14px;
        }

        .emergency-contact p {
            margin: 5px 0;
            font-weight: 500;
        }

        @media print {
            body {
                background-color: #fff;
                margin: 0;
                padding: 0;
                font-size: 10pt;
            }

            .no-print {
                display: none;
            }

            .voucher-container {
                box-shadow: none;
                border-radius: 0;
                max-width: 100%;
                border: none;
            }
        }
    </style>
</head>

<body>
    <div class="no-print"><button class="btn-print" onclick="window.print()"><i class="fas fa-print"></i> Print Voucher</button></div>
    <div class="voucher-container">
        <div class="header">
            <div class="header-left">
                <div class="logo"><img src="<?= e($airline_logo) ?>" alt="Airline Logo"></div>
                <div class="title">E-Ticket Voucher</div>
            </div>
            <div class="header-right">
                <div class="logo"><img src="images/logo.png" alt="RF Travel & Tours Logo"></div>
            </div>
        </div>
        <div class="booking-status">
            <p><strong class="status-<?= strtolower(e($booking_status_text)) ?>"><i class="fas fa-check-circle"></i> Your booking is <?= e($booking_status_text) ?>.</strong> Thank you for booking with us.</p>
        </div>

        <div class="passengers-section">
            <div class="passengers-section-header">
                <h4>Passenger details</h4>
                <span class="departure-date">Departure Date: <?= e($departure_date_display) ?></span>
            </div>
            <table>
                <thead>
                    <tr>
                        <th>Passenger Name</th>
                        <th>Passport No</th>
                        <th>PNR</th>
                        <th>Ticket Number</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($passengers)): ?>
                        <tr>
                            <td colspan="5" style="text-align:center;">No passenger details available.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($passengers as $pax): ?>
                            <tr>
                                <td><?= e($pax['name']) ?></td>
                                <td><?= e($pax['passport'] ?: '—') ?></td>
                                <td><?= e($pax['pnr'] ?: '—') ?></td>
                                <td><?= e($pax['ticket_number'] ?: '—') ?></td>
                                <td><?= e($pax['status']) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php
        if (!empty($itineraries)) {
            foreach ($itineraries as $itinerary) {
                if (isset($itinerary['segments'])) {
                    foreach ($itinerary['segments'] as $segment) {
                        echo renderFlightSegment($segment, $booking['booking_type'], $airline_name, $baggage_allowance, $meal_service, $travel_class);
                    }
                }
            }
        }
        ?>

        <div class="emergency-contact">
            <h4>Emergency Contact</h4>
            <p><i class="fas fa-building"></i> Agency: RF Travel & Tours</p>
            <p><i class="fas fa-phone"></i> Contact No: 0305 2394810</p>
            <p><i class="fas fa-map-marker-alt"></i> Address: AL Quresh Near Railway Pahatak, Infront of Al Quresh Housing Scheme Sher Shah Road Multan
</p>
        </div>
    </div>
</body>

</html>