<?php
// ======================================================================
//  0. START SESSION & INITIALIZE
// ======================================================================
session_start();
require_once 'db-config.php';

// --- CONFIGURATION ---
$whatsapp_number = "923052394810";

// ======================================================================
//  1. FETCH VISA DATA FOR DISPLAY (Adapted from Umrah logic)
// ======================================================================
$visa = null;
$error_message = '';

// Get the unique identifier (slug) from the filename of this page
$current_page_filename = basename($_SERVER['PHP_SELF']);
$page_slug = str_replace('.php', '', $current_page_filename);

if (!empty($page_slug)) {
    try {
        // Fetch the visa details only if it's active
        $sql = "SELECT * FROM visas WHERE page_link = ? AND is_active = 1";
        $stmt = $conn->prepare($sql);
        if ($stmt === false) { throw new Exception("Database prepare failed: " . $conn->error); }
        $stmt->bind_param("s", $page_slug);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $visa = $result->fetch_assoc();
        } else {
            $error_message = "This visa service could not be found or is currently unavailable.";
        }
        $stmt->close();
    } catch (Exception $e) {
        error_log("Visa Page Fetch Error: " . $e->getMessage());
        $error_message = "A server error occurred while retrieving visa details.";
    }
} else {
    $error_message = "Invalid page request.";
}

if ($visa === null) {
    http_response_code(404);
}

// Helper function to render text with line breaks into a proper HTML list
function echo_as_list_items($text) {
    if (empty(trim($text))) return;
    $items = explode("\n", trim($text));
    echo "<ul>";
    foreach ($items as $item) {
        $trimmed_item = trim($item);
        if (!empty($trimmed_item)) {
            echo "<li>" . htmlspecialchars($trimmed_item) . "</li>\n";
        }
    }
    echo "</ul>";
}

// Helper function to render text with line breaks as simple paragraphs
function echo_as_paragraphs($text) {
    if (empty(trim($text))) return;
    $items = explode("\n", trim($text));
    foreach ($items as $item) {
        $trimmed_item = trim($item);
        if (!empty($trimmed_item)) {
            echo "<p>" . htmlspecialchars($trimmed_item) . "</p>\n";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $visa ? htmlspecialchars($visa['visa_name']) : 'Visa Service Not Found' ?> - RF Travel & Tours</title>
    <link rel="icon" type="image/png" href="images/logo-icon.png">
    <link rel="stylesheet" href="css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.2.0/css/all.min.css">
    
    <style>
        /* ==================================================================== */
        /* === EXACT CSS FROM YOUR UMRAH TEMPLATE FOR A PERFECT DUPLICATE === */
        /* ==================================================================== */
        .tour-detail-page { padding: 40px 0; background-color: #fff; }
        .container { max-width: 1700px; margin: 0 auto; padding: 0 2rem; }
        .tour-layout-grid { display: grid; grid-template-columns: minmax(0, 2fr) minmax(0, 1fr); gap: 50px; align-items: flex-start; }
        .tour-header { margin-bottom: 30px; }
        .tour-header h1 { font-size: 2.5rem; font-weight: 700; margin-bottom: 10px; color: var(--text-dark); }
        .tour-meta-info { display: flex; align-items: center; gap: 20px; color: var(--text-light); }
        .tour-rating { display: flex; align-items: center; gap: 5px; }
        .tour-rating i { color: var(--primary-gold); }
        .image-gallery { display: grid; grid-template-columns: 1fr; /* Simplified for single visa image */ gap: 10px; }
        .gallery-main-image { aspect-ratio: 16 / 9; border-radius: 12px; overflow: hidden; }
        .gallery-main-image img { width: 100%; height: 100%; object-fit: cover; }
        .guide-info-card { display: flex; align-items: center; gap: 20px; background: var(--light-bg); padding: 25px; border-radius: 12px; margin: 40px 0; }
        .guide-avatar i { color: var(--primary-dark); }
        .guide-details { flex-grow: 1; }
        .guide-details span { color: var(--text-light); }
        .guide-details h3 { font-size: 1.3rem; margin: 5px 0; color: var(--text-dark); }
        .btn-message { background: #e7f0f7; color: var(--primary-dark); border: none; padding: 12px 20px; border-radius: 8px; font-weight: 600; cursor: pointer; text-decoration: none; display: inline-flex; align-items: center; }
        .btn-message i { margin-right: 8px; }
        .tour-section { padding: 30px 0; border-bottom: 1px solid var(--border-color); }
        .tour-section:last-of-type { border-bottom: none; }
        .tour-section h2 { font-size: 1.5rem; font-weight: 700; margin-bottom: 15px; color: var(--text-dark); }
        .tour-section p, .tour-section ul li { line-height: 1.7; color: var(--text-light); }
        .tour-section ul { list-style-type: disc; padding-left: 20px; }
        .tour-booking-sidebar { position: sticky; top: 100px; }
        .booking-card { border: 1px solid var(--border-color); border-radius: 16px; box-shadow: 0 8px 30px rgba(0, 0, 0, 0.1); overflow: hidden; }
        .booking-card .cta-content { padding: 25px; }
        .booking-price { font-size: 2rem; font-weight: 700; text-align: center; margin-bottom: 20px; margin-top: 20px; color: var(--text-dark); }
        .btn-booking { width: 100%; background: var(--primary-gold); color: var(--white); border: none; padding: 15px; border-radius: 8px; font-size: 1.1rem; font-weight: 600; cursor: pointer; transition: filter 0.3s; text-decoration: none; text-align: center; display: inline-flex; align-items: center; justify-content: center; gap: 10px; }
        .btn-booking:hover { filter: brightness(110%); }
        .details-grid { display: grid; gap: 15px; }
        
        /* NEW STYLES FOR VISA INFO IN SIDEBAR - DESIGNED TO MATCH */
        .visa-sidebar-info { padding: 0 25px 25px 25px; }
        .info-item { display: flex; justify-content: space-between; align-items: center; padding: 12px 0; border-bottom: 1px solid var(--border-color); font-size: 0.95rem; }
        .info-item:last-child { border-bottom: none; }
        .info-item .label { font-weight: 600; color: var(--text-dark); display: flex; align-items: center; gap: 8px; }
        .info-item .value { font-weight: 500; color: var(--text-light); text-align: right; }
        .info-item .price { font-weight: 700; font-size: 1.1rem; color: var(--primary-dark); }

        @media (max-width: 992px) {
            .tour-layout-grid { grid-template-columns: 1fr; }
            .tour-booking-sidebar { position: static; margin-top: 40px; }
        }
        @media (max-width: 768px) {
            .container { padding: 0 1rem; }
            .tour-header h1 { font-size: 2rem; }
            .guide-info-card { flex-direction: column; text-align: center; }
        }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>

    <main class="tour-detail-page">
        <div class="container">
        <?php if ($visa): ?>
            <div class="tour-layout-grid">
                
                <!-- Left Column: Main Visa Information -->
                <div class="tour-main-content">
                    <div class="tour-header">
                        <h1><?= htmlspecialchars($visa['visa_name']) ?></h1>
                        <div class="tour-meta-info">
                            <span><?= htmlspecialchars($visa['visa_type']) ?></span>
                        </div>
                    </div>
                    <div class="image-gallery">
                        <div class="gallery-main-image">
                            <img src="<?= htmlspecialchars($visa['image_url']) ?>" alt="<?= htmlspecialchars($visa['image_alt']) ?>">
                        </div>
                    </div>
                    <section class="guide-info-card">
                        <div class="guide-avatar"><i class="fa-solid fa-headset fa-2x"></i></div>
                        <div class="guide-details">
                            <span>Unsure? Talk to our Visa Experts</span>
                            <h3>We are here to answer all your questions.</h3>
                        </div>
                        <div class="guide-message">
                             <?php $whatsapp_message = urlencode("Hello, I'm interested in the " . $visa['visa_name'] . " service. Please provide me with more details."); ?>
                            <a href="https://wa.me/<?= $whatsapp_number ?>?text=<?= $whatsapp_message ?>" target="_blank" class="btn-message">
                                <i class="fa-brands fa-whatsapp"></i> Message Us
                            </a>
                        </div>
                    </section>
                    
                    <?php if (!empty($visa['overview'])): ?>
                    <section class="tour-section">
                        <h2>Visa Overview</h2>
                        <?= echo_as_paragraphs($visa['overview']) ?>
                    </section>
                    <?php endif; ?>

                    <?php if (!empty($visa['documents_required'])): ?>
                    <section class="tour-section">
                        <h2>Documents Required</h2>
                        <?= echo_as_list_items($visa['documents_required']) ?>
                    </section>
                    <?php endif; ?>

                    <?php if (!empty($visa['how_to_apply'])): ?>
                    <section class="tour-section">
                        <h2>How to Apply</h2>
                        <?= echo_as_list_items($visa['how_to_apply']) ?>
                    </section>
                    <?php endif; ?>
                    
                    <?php if (!empty($visa['fees_and_charges'])): ?>
                    <section class="tour-section">
                        <h2>Fees & Charges</h2>
                        <?= echo_as_paragraphs($visa['fees_and_charges']) ?>
                    </section>
                    <?php endif; ?>
                    
                    <?php if (!empty($visa['important_notes'])): ?>
                    <section class="tour-section">
                        <h2>Important Notes</h2>
                        <?= echo_as_paragraphs($visa['important_notes']) ?>
                    </section>
                    <?php endif; ?>
                </div>

                <!-- Right Column: Sticky Sidebar (Adapted for Visa) -->
                <aside class="tour-booking-sidebar">
                    <div class="booking-card">
                        <div class="booking-price">Service Details</div>
                        <div class="visa-sidebar-info">
                            <?php if (!empty($visa['price'])): ?>
                            <div class="info-item">
                                <span class="label"><i class="fa-solid fa-tags"></i> Price</span>
                                <span class="value price">$<?= htmlspecialchars(number_format($visa['price'], 2)) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($visa['processing_time'])): ?>
                            <div class="info-item">
                                <span class="label"><i class="fa-solid fa-clock"></i> Processing</span>
                                <span class="value"><?= htmlspecialchars($visa['processing_time']) ?></span>
                            </div>
                            <?php endif; ?>
                            <?php if (!empty($visa['entry_type'])): ?>
                            <div class="info-item">
                                <span class="label"><i class="fa-solid fa-right-to-bracket"></i> Entry Type</span>
                                <span class="value"><?= htmlspecialchars($visa['entry_type']) ?></span>
                            </div>
                            <?php endif; ?>
                        </div>
                        <div class="cta-content">
                            <a href="https://wa.me/<?= $whatsapp_number ?>?text=<?= $whatsapp_message ?>" target="_blank" class="btn-booking">
                                <i class="fa-brands fa-whatsapp"></i> Apply on WhatsApp
                            </a>
                        </div>
                    </div>
                </aside>
            </div>
        <?php else: ?>
            <div class="error-container">
                <h1>Service Not Found</h1>
                <p><?= htmlspecialchars($error_message) ?></p>
                <p><a href="/">Return to our Homepage</a></p>
            </div>
        <?php endif; ?>
        </div>
    </main>
<?php
include 'floating-icon.php';?>
    <?php include 'footer.php'; ?>
    
    <script>
        // Hamburger menu script from your Umrah template for consistency
        document.addEventListener('DOMContentLoaded', function() {
            const hamburger = document.getElementById('hamburger');
            if (hamburger) {
                hamburger.addEventListener('click', () => {
                    const navMenu = document.getElementById('nav-menu');
                    if (navMenu) {
                        navMenu.classList.toggle('active');
                    }
                    hamburger.classList.toggle('active');
                });
            }
        });
    </script>
</body>
</html>